import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { GlassCard } from '@/components/ui/glass-card';
import { PageHeader } from '@/components/ui/page-header';
import { ModernTable, ModernTableHeader, ModernTableBody, ModernTableHead, ModernTableRow, ModernTableCell } from '@/components/ui/modern-table';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Badge } from '@/components/ui/badge';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, router, useForm } from '@inertiajs/react';
import { Users, Plus, Search, Edit, Trash2, Shield, UserCog, Phone, Mail, Calculator, Wallet, UserCheck, Truck } from 'lucide-react';
import { useState, useEffect } from 'react';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Team',
        href: '/societies/{society}/team',
    },
];

interface Role {
    id: number;
    name: string;
    display_name: string | null;
}

interface TeamMember {
    id: number;
    name: string;
    email: string | null;
    phone: string;
    status: 'active' | 'inactive' | 'suspended';
    role: Role;
    created_at: string;
}

interface IndexProps {
    society: {
        id: number;
        name: string;
    };
    team: TeamMember[];
    roles: Role[];
    stats: {
        total_members: number;
        active_members: number;
        by_role: Record<string, number>;
    };
}

export default function Index({ society, team, roles, stats }: IndexProps) {
    const [search, setSearch] = useState('');
    const [roleFilter, setRoleFilter] = useState('all');

    // Modal States
    const [isCreateOpen, setIsCreateOpen] = useState(false);
    const [isEditOpen, setIsEditOpen] = useState(false);
    const [editingMember, setEditingMember] = useState<TeamMember | null>(null);

    // Create Form
    const createForm = useForm({
        name: '',
        email: '',
        phone: '',
        password: '',
        password_confirmation: '',
        role_id: '',
        status: 'active',
    });

    // Edit Form
    const editForm = useForm({
        name: '',
        email: '',
        phone: '',
        password: '',
        password_confirmation: '',
        role_id: '',
        status: 'active',
    });

    // Reset forms when modals close
    useEffect(() => {
        if (!isCreateOpen) {
            createForm.reset();
        }
    }, [isCreateOpen]);

    useEffect(() => {
        if (!isEditOpen) {
            editForm.reset();
            setEditingMember(null);
        }
    }, [isEditOpen]);

    const handleCreateSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        createForm.post(`/societies/${society.id}/team`, {
            onSuccess: () => setIsCreateOpen(false),
        });
    };

    const openEditModal = (member: TeamMember) => {
        setEditingMember(member);
        editForm.setData({
            name: member.name,
            email: member.email || '',
            phone: member.phone,
            password: '',
            password_confirmation: '',
            role_id: member.role.id.toString(),
            status: member.status,
        });
        setIsEditOpen(true);
    };

    const handleEditSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        if (!editingMember) return;
        
        editForm.put(`/societies/${society.id}/team/${editingMember.id}`, {
            onSuccess: () => setIsEditOpen(false),
        });
    };

    const handleDelete = (id: number) => {
        if (confirm('Are you sure you want to remove this team member?')) {
            router.delete(`/societies/${society.id}/team/${id}`, {
                preserveScroll: true,
            });
        }
    };

    // Filter team members based on search and role
    const filteredTeam = team.filter(member => {
        const matchesSearch = search === '' || 
            member.name.toLowerCase().includes(search.toLowerCase()) ||
            member.phone.includes(search) ||
            (member.email && member.email.toLowerCase().includes(search.toLowerCase()));
        
        const matchesRole = roleFilter === 'all' || member.role.id.toString() === roleFilter;
        
        return matchesSearch && matchesRole;
    });

    const getRoleIcon = (roleName: string) => {
        switch (roleName) {
            case 'sub_admin': return <Shield className="h-4 w-4" />;
            case 'accountant': return <Calculator className="h-4 w-4" />;
            case 'collector': return <Wallet className="h-4 w-4" />;
            case 'guard': return <UserCheck className="h-4 w-4" />;
            case 'staff': return <UserCog className="h-4 w-4" />;
            case 'vendor': return <Truck className="h-4 w-4" />;
            default: return <Users className="h-4 w-4" />;
        }
    };

    const getRoleColor = (roleName: string) => {
        switch (roleName) {
            case 'sub_admin': return 'bg-purple-100 text-purple-800 dark:bg-purple-900/30 dark:text-purple-400';
            case 'accountant': return 'bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400';
            case 'collector': return 'bg-emerald-100 text-emerald-800 dark:bg-emerald-900/30 dark:text-emerald-400';
            case 'guard': return 'bg-orange-100 text-orange-800 dark:bg-orange-900/30 dark:text-orange-400';
            case 'staff': return 'bg-slate-100 text-slate-800 dark:bg-slate-700 dark:text-slate-300';
            case 'vendor': return 'bg-cyan-100 text-cyan-800 dark:bg-cyan-900/30 dark:text-cyan-400';
            default: return 'bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-400';
        }
    };

    const getStatusColor = (status: string) => {
        switch (status) {
            case 'active': return 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400';
            case 'inactive': return 'bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-400';
            case 'suspended': return 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400';
            default: return 'bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-400';
        }
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Team Management" />
            <div className="flex h-full flex-1 flex-col gap-6 overflow-x-auto rounded-xl p-6">
                <PageHeader
                    title="Team Management"
                    description="Manage staff members for your society"
                    breadcrumbs={breadcrumbs}
                    actions={
                        <Button onClick={() => setIsCreateOpen(true)} className="bg-primary hover:bg-primary/90 shadow-lg shadow-primary/20">
                            <Plus className="mr-2 h-4 w-4" />
                            Add Team Member
                        </Button>
                    }
                />


                {/* KPI Cards */}
                <div className="grid gap-4 md:grid-cols-4">
                    <GlassCard variant="hover" className="border-l-4 border-l-blue-500 py-6 flex flex-col gap-4">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Total Members</CardTitle>
                            <Users className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.total_members}</div>
                            <p className="text-xs text-muted-foreground">Team members</p>
                        </CardContent>
                    </GlassCard>
                    <GlassCard variant="hover" className="border-l-4 border-l-green-500 py-6 flex flex-col gap-4">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Active Members</CardTitle>
                            <UserCheck className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.active_members}</div>
                            <p className="text-xs text-muted-foreground">Currently active</p>
                        </CardContent>
                    </GlassCard>
                    <GlassCard variant="hover" className="border-l-4 border-l-purple-500 py-6 flex flex-col gap-4">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Admins</CardTitle>
                            <Shield className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.by_role?.sub_admin || 0}</div>
                            <p className="text-xs text-muted-foreground">Sub Admins</p>
                        </CardContent>
                    </GlassCard>
                    <GlassCard variant="hover" className="border-l-4 border-l-orange-500 py-6 flex flex-col gap-4">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Guards</CardTitle>
                            <UserCheck className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.by_role?.guard || 0}</div>
                            <p className="text-xs text-muted-foreground">Security guards</p>
                        </CardContent>
                    </GlassCard>
                </div>

                {/* Filters */}
                <GlassCard>
                    <CardContent className="py-6">
                        <div className="grid gap-4 md:grid-cols-3">
                            <div className="md:col-span-2">
                                <Label htmlFor="search">Search</Label>
                                <div className="relative mt-1">
                                    <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                    <Input
                                        id="search"
                                        placeholder="Search by name, phone, or email..."
                                        value={search}
                                        onChange={(e) => setSearch(e.target.value)}
                                        className="pl-9"
                                    />
                                </div>
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="role_filter">Role</Label>
                                <Select value={roleFilter} onValueChange={setRoleFilter}>
                                    <SelectTrigger id="role_filter">
                                        <SelectValue />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Roles</SelectItem>
                                        {roles.map((role) => (
                                            <SelectItem key={role.id} value={role.id.toString()}>
                                                {role.display_name || role.name}
                                            </SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                            </div>
                        </div>
                    </CardContent>
                </GlassCard>

                {/* Team Table */}
                <div className="rounded-xl border border-border/50 bg-card shadow-sm overflow-hidden">
                    <ModernTable>
                        <ModernTableHeader>
                            <ModernTableRow>
                                <ModernTableHead>Name</ModernTableHead>
                                <ModernTableHead>Contact</ModernTableHead>
                                <ModernTableHead>Role</ModernTableHead>
                                <ModernTableHead>Status</ModernTableHead>
                                <ModernTableHead className="text-right">Actions</ModernTableHead>
                            </ModernTableRow>
                        </ModernTableHeader>
                        <ModernTableBody>
                            {filteredTeam.length > 0 ? (
                                filteredTeam.map((member) => (
                                    <ModernTableRow key={member.id}>
                                        <ModernTableCell>
                                            <div className="flex items-center gap-2">
                                                <div className="h-8 w-8 rounded-full bg-indigo-100 dark:bg-indigo-900/50 flex items-center justify-center text-indigo-600 dark:text-indigo-400 font-bold text-xs">
                                                    {member.name.charAt(0)}
                                                </div>
                                                <span className="font-medium">{member.name}</span>
                                            </div>
                                        </ModernTableCell>
                                        <ModernTableCell className="text-sm">
                                            <div className="space-y-1">
                                                <div className="flex items-center gap-1 text-muted-foreground">
                                                    <Phone className="h-3 w-3" />
                                                    {member.phone}
                                                </div>
                                                {member.email && (
                                                    <div className="flex items-center gap-1 text-muted-foreground text-xs">
                                                        <Mail className="h-3 w-3" />
                                                        {member.email}
                                                    </div>
                                                )}
                                            </div>
                                        </ModernTableCell>
                                        <ModernTableCell>
                                            <Badge className={`${getRoleColor(member.role.name)} flex items-center gap-1 w-fit`}>
                                                {getRoleIcon(member.role.name)}
                                                {member.role.display_name || member.role.name}
                                            </Badge>
                                        </ModernTableCell>
                                        <ModernTableCell>
                                            <span className={`inline-flex rounded-full px-2 py-1 text-xs font-medium ${getStatusColor(member.status)}`}>
                                                {member.status.charAt(0).toUpperCase() + member.status.slice(1)}
                                            </span>
                                        </ModernTableCell>
                                        <ModernTableCell className="text-right">
                                            <div className="flex items-center justify-end gap-2">
                                                <Button 
                                                    variant="ghost" 
                                                    size="icon" 
                                                    className="h-8 w-8 hover:bg-amber-50 hover:text-amber-600 dark:hover:bg-amber-900/30"
                                                    onClick={() => openEditModal(member)}
                                                >
                                                    <Edit className="h-4 w-4" />
                                                </Button>
                                                <Button
                                                    variant="ghost"
                                                    size="icon"
                                                    className="h-8 w-8 hover:bg-red-50 hover:text-red-600 dark:hover:bg-red-900/30"
                                                    onClick={() => handleDelete(member.id)}
                                                >
                                                    <Trash2 className="h-4 w-4" />
                                                </Button>
                                            </div>
                                        </ModernTableCell>
                                    </ModernTableRow>
                                ))
                            ) : (
                                <ModernTableRow>
                                    <ModernTableCell colSpan={5} className="h-32 text-center">
                                        <div className="flex flex-col items-center gap-2 text-muted-foreground">
                                            <Users className="h-10 w-10 text-muted-foreground/30" />
                                            <p>No team members found</p>
                                            <p className="text-sm">Add your first team member to get started</p>
                                        </div>
                                    </ModernTableCell>
                                </ModernTableRow>
                            )}
                        </ModernTableBody>
                    </ModernTable>
                </div>

                {/* Edit Dialog */}
                <Dialog open={isEditOpen} onOpenChange={setIsEditOpen}>
                    <DialogContent className="sm:max-w-[600px] max-h-[90vh] overflow-y-auto">
                        <DialogHeader>
                            <DialogTitle>Edit Team Member</DialogTitle>
                            <DialogDescription>
                                Update team member details.
                            </DialogDescription>
                        </DialogHeader>
                        <form onSubmit={handleEditSubmit} className="space-y-4 py-4">
                            <div className="grid grid-cols-2 gap-4">
                                <div className="space-y-2">
                                    <Label htmlFor="edit_name">Full Name *</Label>
                                    <Input
                                        id="edit_name"
                                        value={editForm.data.name}
                                        onChange={(e) => editForm.setData('name', e.target.value)}
                                        required
                                    />
                                    <InputError message={editForm.errors.name} />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_phone">Phone Number *</Label>
                                    <Input
                                        id="edit_phone"
                                        value={editForm.data.phone}
                                        onChange={(e) => editForm.setData('phone', e.target.value)}
                                        required
                                    />
                                    <InputError message={editForm.errors.phone} />
                                </div>
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="edit_email">Email Address</Label>
                                <Input
                                    id="edit_email"
                                    type="email"
                                    value={editForm.data.email}
                                    onChange={(e) => editForm.setData('email', e.target.value)}
                                />
                                <InputError message={editForm.errors.email} />
                            </div>

                            <div className="grid grid-cols-2 gap-4">
                                <div className="space-y-2">
                                    <Label htmlFor="edit_role_id">Role *</Label>
                                    <Select 
                                        value={editForm.data.role_id} 
                                        onValueChange={(value) => editForm.setData('role_id', value)}
                                    >
                                        <SelectTrigger id="edit_role_id">
                                            <SelectValue />
                                        </SelectTrigger>
                                        <SelectContent>
                                            {roles.map((role) => (
                                                <SelectItem key={role.id} value={role.id.toString()}>
                                                    {role.display_name || role.name}
                                                </SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                    <InputError message={editForm.errors.role_id} />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_status">Status</Label>
                                    <Select 
                                        value={editForm.data.status} 
                                        onValueChange={(value) => editForm.setData('status', value)}
                                    >
                                        <SelectTrigger id="edit_status">
                                            <SelectValue />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="active">Active</SelectItem>
                                            <SelectItem value="inactive">Inactive</SelectItem>
                                            <SelectItem value="suspended">Suspended</SelectItem>
                                        </SelectContent>
                                    </Select>
                                </div>
                            </div>

                            <div className="grid grid-cols-2 gap-4">
                                <div className="space-y-2">
                                    <Label htmlFor="edit_password">New Password (Optional)</Label>
                                    <Input
                                        id="edit_password"
                                        type="password"
                                        value={editForm.data.password}
                                        onChange={(e) => editForm.setData('password', e.target.value)}
                                        placeholder="Leave blank to keep current"
                                    />
                                    <InputError message={editForm.errors.password} />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_password_confirmation">Confirm New Password</Label>
                                    <Input
                                        id="edit_password_confirmation"
                                        type="password"
                                        value={editForm.data.password_confirmation}
                                        onChange={(e) => editForm.setData('password_confirmation', e.target.value)}
                                    />
                                </div>
                            </div>

                            <DialogFooter>
                                <Button type="button" variant="outline" onClick={() => setIsEditOpen(false)}>Cancel</Button>
                                <Button type="submit" disabled={editForm.processing}>
                                    {editForm.processing ? 'Saving...' : 'Save Changes'}
                                </Button>
                            </DialogFooter>
                        </form>
                    </DialogContent>
                </Dialog>

                <Dialog open={isCreateOpen} onOpenChange={setIsCreateOpen}>
                        <DialogContent className="sm:max-w-[600px] max-h-[90vh] overflow-y-auto">
                            <DialogHeader>
                                <DialogTitle>Add Team Member</DialogTitle>
                                <DialogDescription>
                                    Add a new staff member to your society.
                                </DialogDescription>
                            </DialogHeader>
                            <form onSubmit={handleCreateSubmit} className="space-y-4 py-4">
                                <div className="grid grid-cols-2 gap-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="name">Full Name *</Label>
                                        <Input
                                            id="name"
                                            value={createForm.data.name}
                                            onChange={(e) => createForm.setData('name', e.target.value)}
                                            placeholder="John Doe"
                                            required
                                        />
                                        <InputError message={createForm.errors.name} />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="phone">Phone Number *</Label>
                                        <Input
                                            id="phone"
                                            value={createForm.data.phone}
                                            onChange={(e) => createForm.setData('phone', e.target.value)}
                                            placeholder="9876543210"
                                            required
                                        />
                                        <InputError message={createForm.errors.phone} />
                                    </div>
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="email">Email Address</Label>
                                    <Input
                                        id="email"
                                        type="email"
                                        value={createForm.data.email}
                                        onChange={(e) => createForm.setData('email', e.target.value)}
                                        placeholder="john@example.com"
                                    />
                                    <InputError message={createForm.errors.email} />
                                </div>

                                <div className="grid grid-cols-2 gap-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="role_id">Role *</Label>
                                        <Select 
                                            value={createForm.data.role_id} 
                                            onValueChange={(value) => createForm.setData('role_id', value)}
                                        >
                                            <SelectTrigger id="role_id">
                                                <SelectValue placeholder="Select Role" />
                                            </SelectTrigger>
                                            <SelectContent>
                                                {roles.map((role) => (
                                                    <SelectItem key={role.id} value={role.id.toString()}>
                                                        {role.display_name || role.name}
                                                    </SelectItem>
                                                ))}
                                            </SelectContent>
                                        </Select>
                                        <InputError message={createForm.errors.role_id} />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="status">Status</Label>
                                        <Select 
                                            value={createForm.data.status} 
                                            onValueChange={(value) => createForm.setData('status', value)}
                                        >
                                            <SelectTrigger id="status">
                                                <SelectValue />
                                            </SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="active">Active</SelectItem>
                                                <SelectItem value="inactive">Inactive</SelectItem>
                                                <SelectItem value="suspended">Suspended</SelectItem>
                                            </SelectContent>
                                        </Select>
                                    </div>
                                </div>

                                <div className="grid grid-cols-2 gap-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="password">Password *</Label>
                                        <Input
                                            id="password"
                                            type="password"
                                            value={createForm.data.password}
                                            onChange={(e) => createForm.setData('password', e.target.value)}
                                            required
                                        />
                                        <InputError message={createForm.errors.password} />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="password_confirmation">Confirm Password *</Label>
                                        <Input
                                            id="password_confirmation"
                                            type="password"
                                            value={createForm.data.password_confirmation}
                                            onChange={(e) => createForm.setData('password_confirmation', e.target.value)}
                                            required
                                        />
                                    </div>
                                </div>

                                <DialogFooter>
                                    <Button type="button" variant="outline" onClick={() => setIsCreateOpen(false)}>Cancel</Button>
                                    <Button type="submit" disabled={createForm.processing}>
                                        {createForm.processing ? 'Adding...' : 'Add Member'}
                                    </Button>
                                </DialogFooter>
                            </form>
                        </DialogContent>
                    </Dialog>
            </div>
        </SocietyAdminLayout>
    );
}
