import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Checkbox } from '@/components/ui/checkbox';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Textarea } from '@/components/ui/textarea';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, useForm } from '@inertiajs/react';
import { Settings as SettingsIcon, CreditCard, Mail, MessageSquare, Bell, Shield, Globe, Building, Save } from 'lucide-react';
import { useState } from 'react';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Settings',
        href: '/societies/{society}/settings',
    },
];

interface Society {
    id: number;
    name: string;
    code: string;
    registration_no: string | null;
    address: string | null;
    city: string | null;
    state: string | null;
    pincode: string | null;
    phone: string | null;
    email: string | null;
    gst_no: string | null;
    pan_no: string | null;
}

interface SocietySettings {
    id: number;
    payment_gateway: string;
    razorpay_key_id: string | null;
    razorpay_key_secret: string | null;
    razorpay_webhook_secret?: string | null;
    razorpay_enabled: boolean;
    stripe_public_key: string | null;
    stripe_secret_key: string | null;
    stripe_enabled: boolean;
    payu_merchant_key: string | null;
    payu_merchant_salt: string | null;
    payu_enabled: boolean;
    cashfree_app_id: string | null;
    cashfree_secret_key: string | null;
    cashfree_enabled: boolean;
    email_enabled: boolean;
    email_provider: string;
    smtp_host: string | null;
    smtp_port: number | null;
    smtp_username: string | null;
    smtp_password: string | null;
    smtp_encryption: string | null;
    from_email: string | null;
    from_name: string | null;
    sms_enabled: boolean;
    sms_provider: string | null;
    sms_api_key: string | null;
    sms_sender_id: string | null;
    sms_username: string | null;
    sms_password: string | null;
    whatsapp_enabled: boolean;
    whatsapp_provider: string | null;
    whatsapp_api_key: string | null;
    whatsapp_sender_id: string | null;
    whatsapp_business_id: string | null;
    timezone: string;
    currency: string;
    date_format: string;
    time_format: string;
    language: string;
    notify_bill_generated: boolean;
    notify_payment_received: boolean;
    notify_visitor_arrival: boolean;
    notify_maintenance_request: boolean;
    notify_announcement: boolean;
    enable_visitor_management: boolean;
    enable_maintenance_tickets: boolean;
    enable_complaints: boolean;
    enable_events: boolean;
    enable_documents: boolean;
    enable_directory: boolean;
    enable_notices: boolean;
    enable_polls: boolean;
    enable_marketplace: boolean;
    enable_facility_booking: boolean;
    require_otp_for_login: boolean;
    require_otp_for_payment: boolean;
    session_timeout_minutes: number;
    enable_2fa: boolean;
    late_fee_calculation_type: 'none' | 'fixed_daily' | 'percentage_daily';
    late_fee_value: number;
}

interface IndexProps {
    society: Society;
    settings: SocietySettings;
    errors?: any;
}

export default function Index({ society, settings, errors }: IndexProps) {
    const [activeTab, setActiveTab] = useState('society');

    // Combine society data and settings into one form
    const { data, setData, put, processing } = useForm({
        // Society Details
        name: society.name || '',
        code: society.code || '',
        registration_no: society.registration_no || '',
        address: society.address || '',
        city: society.city || '',
        state: society.state || '',
        pincode: society.pincode || '',
        phone: society.phone || '',
        email: society.email || '',
        gst_no: society.gst_no || '',
        pan_no: society.pan_no || '',

        // Settings
        ...settings,
        // Ensure booleans are booleans (sometimes they come as 0/1 from DB if not casted)
        razorpay_enabled: Boolean(settings.razorpay_enabled),
        stripe_enabled: Boolean(settings.stripe_enabled),
        payu_enabled: Boolean(settings.payu_enabled),
        cashfree_enabled: Boolean(settings.cashfree_enabled),
        email_enabled: Boolean(settings.email_enabled),
        sms_enabled: Boolean(settings.sms_enabled),
        whatsapp_enabled: Boolean(settings.whatsapp_enabled),
        
        notify_bill_generated: Boolean(settings.notify_bill_generated),
        notify_payment_received: Boolean(settings.notify_payment_received),
        notify_visitor_arrival: Boolean(settings.notify_visitor_arrival),
        notify_maintenance_request: Boolean(settings.notify_maintenance_request),
        notify_announcement: Boolean(settings.notify_announcement),

        enable_visitor_management: Boolean(settings.enable_visitor_management),
        enable_maintenance_tickets: Boolean(settings.enable_maintenance_tickets),
        enable_complaints: Boolean(settings.enable_complaints),
        enable_events: Boolean(settings.enable_events),
        enable_documents: Boolean(settings.enable_documents),
        enable_directory: Boolean(settings.enable_directory),
        enable_notices: Boolean(settings.enable_notices),
        enable_polls: Boolean(settings.enable_polls),
        enable_marketplace: Boolean(settings.enable_marketplace),
        enable_facility_booking: Boolean(settings.enable_facility_booking),

        require_otp_for_login: Boolean(settings.require_otp_for_login),
        require_otp_for_payment: Boolean(settings.require_otp_for_payment),
        enable_2fa: Boolean(settings.enable_2fa),

        // Interest Settings
        late_fee_calculation_type: settings.late_fee_calculation_type || 'none',
        late_fee_value: settings.late_fee_value || 0,
    });

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        put(`/societies/${society.id}/settings`, {
            preserveScroll: true,
            preserveState: true,
        });
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Settings" />
            <div className="flex h-full flex-1 flex-col gap-6 overflow-x-auto rounded-xl p-6">
                <div className="flex items-center justify-between">
                    <div>
                        <h1 className="text-2xl font-bold">Settings</h1>
                        <p className="text-muted-foreground">
                            Manage society profile and application preferences
                        </p>
                    </div>
                    <Button onClick={handleSubmit} disabled={processing}>
                        <Save className="mr-2 h-4 w-4" />
                        {processing ? 'Saving...' : 'Save Changes'}
                    </Button>
                </div>

                <form onSubmit={handleSubmit} className="space-y-6">
                    <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
                        <TabsList className="grid w-full grid-cols-2 lg:grid-cols-7 h-auto">
                            <TabsTrigger value="society" className="py-2">
                                <Building className="mr-2 h-4 w-4" />
                                Society
                            </TabsTrigger>
                            <TabsTrigger value="general" className="py-2">
                                <Globe className="mr-2 h-4 w-4" />
                                General
                            </TabsTrigger>
                            <TabsTrigger value="payment" className="py-2">
                                <CreditCard className="mr-2 h-4 w-4" />
                                Payment
                            </TabsTrigger>
                            <TabsTrigger value="email" className="py-2">
                                <Mail className="mr-2 h-4 w-4" />
                                Email
                            </TabsTrigger>
                            <TabsTrigger value="sms" className="py-2">
                                <MessageSquare className="mr-2 h-4 w-4" />
                                SMS
                            </TabsTrigger>
                            <TabsTrigger value="notifications" className="py-2">
                                <Bell className="mr-2 h-4 w-4" />
                                Notify
                            </TabsTrigger>
                            <TabsTrigger value="security" className="py-2">
                                <Shield className="mr-2 h-4 w-4" />
                                Security
                            </TabsTrigger>
                        </TabsList>

                        {/* Society Details Tab */}
                        <TabsContent value="society" className="space-y-6 mt-6">
                            <Card>
                                <CardHeader>
                                    <CardTitle>Society Profile</CardTitle>
                                    <CardDescription>Basic information about your society.</CardDescription>
                                </CardHeader>
                                <CardContent className="grid gap-6">
                                    <div className="grid gap-4 md:grid-cols-2">
                                        <div className="space-y-2">
                                            <Label htmlFor="name">Society Name *</Label>
                                            <Input
                                                id="name"
                                                value={data.name}
                                                onChange={e => setData('name', e.target.value)}
                                            />
                                            <InputError message={errors?.name} />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="code">Society Code *</Label>
                                            <Input
                                                id="code"
                                                value={data.code}
                                                onChange={e => setData('code', e.target.value)}
                                            />
                                            <InputError message={errors?.code} />
                                        </div>
                                    </div>

                                    <div className="space-y-2">
                                        <Label htmlFor="registration_no">Registration Number</Label>
                                        <Input
                                            id="registration_no"
                                            value={data.registration_no}
                                            onChange={e => setData('registration_no', e.target.value)}
                                        />
                                        <InputError message={errors?.registration_no} />
                                    </div>
                                    
                                    <div className="space-y-2">
                                        <Label htmlFor="address">Address *</Label>
                                        <Textarea
                                            id="address"
                                            value={data.address}
                                            onChange={e => setData('address', e.target.value)}
                                            rows={2}
                                        />
                                        <InputError message={errors?.address} />
                                    </div>

                                    <div className="grid gap-4 md:grid-cols-3">
                                        <div className="space-y-2">
                                            <Label htmlFor="city">City *</Label>
                                            <Input
                                                id="city"
                                                value={data.city}
                                                onChange={e => setData('city', e.target.value)}
                                            />
                                            <InputError message={errors?.city} />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="state">State *</Label>
                                            <Input
                                                id="state"
                                                value={data.state}
                                                onChange={e => setData('state', e.target.value)}
                                            />
                                            <InputError message={errors?.state} />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="pincode">Pincode *</Label>
                                            <Input
                                                id="pincode"
                                                value={data.pincode}
                                                onChange={e => setData('pincode', e.target.value)}
                                            />
                                            <InputError message={errors?.pincode} />
                                        </div>
                                    </div>

                                    <div className="grid gap-4 md:grid-cols-2">
                                        <div className="space-y-2">
                                            <Label htmlFor="phone">Phone/Contact *</Label>
                                            <Input
                                                id="phone"
                                                value={data.phone}
                                                onChange={e => setData('phone', e.target.value)}
                                            />
                                            <InputError message={errors?.phone} />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="email">Email *</Label>
                                            <Input
                                                id="email"
                                                type="email"
                                                value={data.email}
                                                onChange={e => setData('email', e.target.value)}
                                            />
                                            <InputError message={errors?.email} />
                                        </div>
                                    </div>

                                    <div className="grid gap-4 md:grid-cols-2">
                                        <div className="space-y-2">
                                            <Label htmlFor="gst_no">GST Number</Label>
                                            <Input
                                                id="gst_no"
                                                value={data.gst_no}
                                                onChange={e => setData('gst_no', e.target.value)}
                                            />
                                            <InputError message={errors?.gst_no} />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="pan_no">PAN Number</Label>
                                            <Input
                                                id="pan_no"
                                                value={data.pan_no}
                                                onChange={e => setData('pan_no', e.target.value)}
                                            />
                                            <InputError message={errors?.pan_no} />
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>
                        </TabsContent>

                        {/* General / Preferences Tab */}
                        <TabsContent value="general" className="space-y-6 mt-6">
                            <Card>
                                <CardHeader>
                                    <CardTitle>Regional Preferences</CardTitle>
                                    <CardDescription>Format settings for dates, currency, and time.</CardDescription>
                                </CardHeader>
                                <CardContent className="grid gap-6">
                                    <div className="grid gap-4 md:grid-cols-2">
                                        <div className="space-y-2">
                                            <Label htmlFor="timezone">Timezone</Label>
                                            <Select value={data.timezone} onValueChange={val => setData('timezone', val)}>
                                                <SelectTrigger id="timezone"><SelectValue /></SelectTrigger>
                                                <SelectContent>
                                                    <SelectItem value="Asia/Kolkata">Asia/Kolkata (IST)</SelectItem>
                                                    <SelectItem value="Asia/Dubai">Asia/Dubai (GST)</SelectItem>
                                                    <SelectItem value="America/New_York">America/New_York (EST)</SelectItem>
                                                    <SelectItem value="Europe/London">Europe/London (GMT)</SelectItem>
                                                </SelectContent>
                                            </Select>
                                            <InputError message={errors?.timezone} />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="currency">Currency</Label>
                                            <Select value={data.currency} onValueChange={val => setData('currency', val)}>
                                                <SelectTrigger id="currency"><SelectValue /></SelectTrigger>
                                                <SelectContent>
                                                    <SelectItem value="INR">INR (₹)</SelectItem>
                                                    <SelectItem value="USD">USD ($)</SelectItem>
                                                    <SelectItem value="EUR">EUR (€)</SelectItem>
                                                    <SelectItem value="GBP">GBP (£)</SelectItem>
                                                </SelectContent>
                                            </Select>
                                            <InputError message={errors?.currency} />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="date_format">Date Format</Label>
                                            <Input
                                                id="date_format"
                                                value={data.date_format}
                                                onChange={e => setData('date_format', e.target.value)}
                                                placeholder="d/m/Y"
                                            />
                                            <InputError message={errors?.date_format} />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="time_format">Time Format</Label>
                                            <Select value={data.time_format} onValueChange={val => setData('time_format', val)}>
                                                <SelectTrigger id="time_format"><SelectValue /></SelectTrigger>
                                                <SelectContent>
                                                    <SelectItem value="H:i">24 Hour (14:30)</SelectItem>
                                                    <SelectItem value="h:i A">12 Hour (2:30 PM)</SelectItem>
                                                </SelectContent>
                                            </Select>
                                            <InputError message={errors?.time_format} />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="language">Language</Label>
                                            <Select value={data.language} onValueChange={val => setData('language', val)}>
                                                <SelectTrigger id="language"><SelectValue /></SelectTrigger>
                                                <SelectContent>
                                                    <SelectItem value="en">English</SelectItem>
                                                    <SelectItem value="hi">Hindi</SelectItem>
                                                    <SelectItem value="mr">Marathi</SelectItem>
                                                    <SelectItem value="gu">Gujarati</SelectItem>
                                                </SelectContent>
                                            </Select>
                                            <InputError message={errors?.language} />
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>

                            <Card>
                            <CardHeader>
                                    <CardTitle>Feature Toggles</CardTitle>
                                    <CardDescription>Enable or disable features for your society portal.</CardDescription>
                                </CardHeader>
                                <CardContent className="grid gap-6">
                                    <div className="grid gap-4 md:grid-cols-2">
                                        {[
                                            { id: 'enable_visitor_management', label: 'Visitor Management' },
                                            { id: 'enable_maintenance_tickets', label: 'Maintenance Tickets' },
                                            { id: 'enable_complaints', label: 'Complaints' },
                                            { id: 'enable_events', label: 'Events' },
                                            { id: 'enable_documents', label: 'Documents' },
                                            { id: 'enable_directory', label: 'Directory' },
                                            { id: 'enable_notices', label: 'Notices' },
                                            { id: 'enable_polls', label: 'Polls' },
                                            { id: 'enable_marketplace', label: 'Marketplace' },
                                            { id: 'enable_facility_booking', label: 'Facility Booking' },
                                        ].map((feature) => (
                                            <div key={feature.id} className="flex items-center gap-2">
                                                <Checkbox
                                                    id={feature.id}
                                                    checked={Boolean(data[feature.id as keyof typeof data])}
                                                    onCheckedChange={(checked) => setData(feature.id as any, checked)}
                                                />
                                                <Label htmlFor={feature.id} className="cursor-pointer">{feature.label}</Label>
                                            </div>
                                        ))}
                                    </div>
                                </CardContent>
                            </Card>
                        </TabsContent>

                        {/* Payment Tab */}
                        <TabsContent value="payment" className="space-y-6 mt-6">
                            <Card>
                                <CardHeader>
                                    <CardTitle>Razorpay Integration</CardTitle>
                                    <CardDescription>Configure Razorpay for accepting online payments.</CardDescription>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                    <div className="flex items-center gap-2 mb-4">
                                        <Checkbox
                                            id="razorpay_enabled"
                                            checked={data.razorpay_enabled}
                                            onCheckedChange={checked => setData('razorpay_enabled', !!checked)}
                                        />
                                        <Label htmlFor="razorpay_enabled" className="cursor-pointer font-medium">Enable Razorpay</Label>
                                    </div>
                                    {data.razorpay_enabled && (
                                        <div className="grid gap-4 md:grid-cols-2">
                                            <div className="space-y-2">
                                                <Label htmlFor="razorpay_key_id">Key ID</Label>
                                                <Input
                                                    id="razorpay_key_id"
                                                    value={data.razorpay_key_id || ''}
                                                    onChange={e => setData('razorpay_key_id', e.target.value)}
                                                />
                                                <InputError message={errors?.razorpay_key_id} />
                                            </div>
                                            <div className="space-y-2">
                                                <Label htmlFor="razorpay_key_secret">Key Secret</Label>
                                                <Input
                                                    id="razorpay_key_secret"
                                                    type="password"
                                                    value={data.razorpay_key_secret || ''}
                                                    onChange={e => setData('razorpay_key_secret', e.target.value)}
                                                />
                                                <InputError message={errors?.razorpay_key_secret} />
                                            </div>
                                            <div className="space-y-2 md:col-span-2">
                                                <Label htmlFor="razorpay_webhook_secret">Webhook Secret</Label>
                                                <Input
                                                    id="razorpay_webhook_secret"
                                                    type="password"
                                                    value={(data as any).razorpay_webhook_secret || ''}
                                                    onChange={e => setData('razorpay_webhook_secret' as any, e.target.value)}
                                                    placeholder="Used to verify Razorpay webhooks"
                                                />
                                                <InputError message={errors?.razorpay_webhook_secret} />
                                                <p className="text-xs text-muted-foreground">
                                                    Set this to the webhook secret configured in your Razorpay dashboard for this society’s account.
                                                </p>
                                            </div>
                                        </div>
                                    )}
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader>
                                    <CardTitle>Stripe Integration</CardTitle>
                                    <CardDescription>Configure Stripe for accepting international payments.</CardDescription>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                    <div className="flex items-center gap-2 mb-4">
                                        <Checkbox
                                            id="stripe_enabled"
                                            checked={data.stripe_enabled}
                                            onCheckedChange={checked => setData('stripe_enabled', !!checked)}
                                        />
                                        <Label htmlFor="stripe_enabled" className="cursor-pointer font-medium">Enable Stripe</Label>
                                    </div>
                                    {data.stripe_enabled && (
                                        <div className="grid gap-4 md:grid-cols-2">
                                            <div className="space-y-2">
                                                <Label htmlFor="stripe_public_key">Public Key</Label>
                                                <Input
                                                    id="stripe_public_key"
                                                    value={data.stripe_public_key || ''}
                                                    onChange={e => setData('stripe_public_key', e.target.value)}
                                                />
                                                <InputError message={errors?.stripe_public_key} />
                                            </div>
                                            <div className="space-y-2">
                                                <Label htmlFor="stripe_secret_key">Secret Key</Label>
                                                <Input
                                                    id="stripe_secret_key"
                                                    type="password"
                                                    value={data.stripe_secret_key || ''}
                                                    onChange={e => setData('stripe_secret_key', e.target.value)}
                                                />
                                                <InputError message={errors?.stripe_secret_key} />
                                            </div>
                                        </div>
                                    )}
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader>
                                    <CardTitle>PayU Integration</CardTitle>
                                    <CardDescription>Configure PayU Money for accepting payments.</CardDescription>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                    <div className="flex items-center gap-2 mb-4">
                                        <Checkbox
                                            id="payu_enabled"
                                            checked={data.payu_enabled}
                                            onCheckedChange={checked => setData('payu_enabled', !!checked)}
                                        />
                                        <Label htmlFor="payu_enabled" className="cursor-pointer font-medium">Enable PayU</Label>
                                    </div>
                                    {data.payu_enabled && (
                                        <div className="grid gap-4 md:grid-cols-2">
                                            <div className="space-y-2">
                                                <Label htmlFor="payu_merchant_key">Merchant Key</Label>
                                                <Input
                                                    id="payu_merchant_key"
                                                    value={data.payu_merchant_key || ''}
                                                    onChange={e => setData('payu_merchant_key', e.target.value)}
                                                />
                                                <InputError message={errors?.payu_merchant_key} />
                                            </div>
                                            <div className="space-y-2">
                                                <Label htmlFor="payu_merchant_salt">Merchant Salt</Label>
                                                <Input
                                                    id="payu_merchant_salt"
                                                    type="password"
                                                    value={data.payu_merchant_salt || ''}
                                                    onChange={e => setData('payu_merchant_salt', e.target.value)}
                                                />
                                                <InputError message={errors?.payu_merchant_salt} />
                                            </div>
                                        </div>
                                    )}
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader>
                                    <CardTitle>Cashfree Integration</CardTitle>
                                    <CardDescription>Configure Cashfree Payments.</CardDescription>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                    <div className="flex items-center gap-2 mb-4">
                                        <Checkbox
                                            id="cashfree_enabled"
                                            checked={data.cashfree_enabled}
                                            onCheckedChange={checked => setData('cashfree_enabled', !!checked)}
                                        />
                                        <Label htmlFor="cashfree_enabled" className="cursor-pointer font-medium">Enable Cashfree</Label>
                                    </div>
                                    {data.cashfree_enabled && (
                                        <div className="grid gap-4 md:grid-cols-2">
                                            <div className="space-y-2">
                                                <Label htmlFor="cashfree_app_id">App ID</Label>
                                                <Input
                                                    id="cashfree_app_id"
                                                    value={data.cashfree_app_id || ''}
                                                    onChange={e => setData('cashfree_app_id', e.target.value)}
                                                />
                                                <InputError message={errors?.cashfree_app_id} />
                                            </div>
                                            <div className="space-y-2">
                                                <Label htmlFor="cashfree_secret_key">Secret Key</Label>
                                                <Input
                                                    id="cashfree_secret_key"
                                                    type="password"
                                                    value={data.cashfree_secret_key || ''}
                                                    onChange={e => setData('cashfree_secret_key', e.target.value)}
                                                />
                                                <InputError message={errors?.cashfree_secret_key} />
                                            </div>
                                        </div>
                                    )}
                                </CardContent>
                            </Card>

                            {/* Interest Settings */}
                            <Card className="mt-6">
                                <CardHeader>
                                    <CardTitle>Late Fee / Interest Settings</CardTitle>
                                    <CardDescription>Configure penalties for overdue bill payments.</CardDescription>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                     <div className="grid gap-4 md:grid-cols-2">
                                        <div className="space-y-2">
                                            <Label htmlFor="late_fee_calculation_type">Calculation Method</Label>
                                            <Select 
                                                value={data.late_fee_calculation_type} 
                                                onValueChange={(val: any) => setData('late_fee_calculation_type', val)}
                                            >
                                                <SelectTrigger id="late_fee_calculation_type"><SelectValue /></SelectTrigger>
                                                <SelectContent>
                                                    <SelectItem value="none">No Late Fee</SelectItem>
                                                    <SelectItem value="fixed_daily">Fixed Amount per Day</SelectItem>
                                                    <SelectItem value="percentage_daily">Percentage of Due Amount per Day</SelectItem>
                                                </SelectContent>
                                            </Select>
                                            <InputError message={errors?.late_fee_calculation_type} />
                                        </div>
                                        {data.late_fee_calculation_type !== 'none' && (
                                            <div className="space-y-2">
                                                <Label htmlFor="late_fee_value">
                                                    {data.late_fee_calculation_type === 'fixed_daily' ? 'Amount (Daily)' : 'Percentage (Daily %)'}
                                                </Label>
                                                <Input
                                                    id="late_fee_value"
                                                    type="number"
                                                    min="0"
                                                    step="0.01"
                                                    value={data.late_fee_value}
                                                    onChange={e => setData('late_fee_value', parseFloat(e.target.value))}
                                                />
                                                <InputError message={errors?.late_fee_value} />
                                            </div>
                                        )}
                                     </div>
                                </CardContent>
                            </Card>
                        </TabsContent>

                        {/* Email Tab */}
                        <TabsContent value="email" className="space-y-6 mt-6">
                            <Card>
                                <CardHeader>
                                    <CardTitle>Email Settings</CardTitle>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                    <div className="flex items-center gap-2 mb-4">
                                        <Checkbox
                                            id="email_enabled"
                                            checked={data.email_enabled}
                                            onCheckedChange={checked => setData('email_enabled', !!checked)}
                                        />
                                        <Label htmlFor="email_enabled" className="cursor-pointer font-medium">Enable Email Notifications</Label>
                                    </div>
                                    <div className="grid gap-4 md:grid-cols-2">
                                        <div className="space-y-2">
                                            <Label htmlFor="email_provider">Provider</Label>
                                            <Select value={data.email_provider} onValueChange={val => setData('email_provider', val)}>
                                                <SelectTrigger id="email_provider"><SelectValue /></SelectTrigger>
                                                <SelectContent>
                                                    <SelectItem value="smtp">SMTP</SelectItem>
                                                    <SelectItem value="sendgrid">SendGrid</SelectItem>
                                                    <SelectItem value="mailgun">Mailgun</SelectItem>
                                                </SelectContent>
                                            </Select>
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="from_email">From Email</Label>
                                            <Input
                                                id="from_email"
                                                value={data.from_email || ''}
                                                onChange={e => setData('from_email', e.target.value)}
                                            />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="from_name">From Name</Label>
                                            <Input
                                                id="from_name"
                                                value={data.from_name || ''}
                                                onChange={e => setData('from_name', e.target.value)}
                                            />
                                        </div>
                                    </div>

                                    {data.email_provider === 'smtp' && (
                                        <div className="space-y-4 pt-4 border-t">
                                            <div className="grid gap-4 md:grid-cols-2">
                                                <div className="space-y-2">
                                                    <Label htmlFor="smtp_host">SMTP Host</Label>
                                                    <Input
                                                        id="smtp_host"
                                                        value={data.smtp_host || ''}
                                                        onChange={e => setData('smtp_host', e.target.value)}
                                                    />
                                                </div>
                                                <div className="space-y-2">
                                                    <Label htmlFor="smtp_port">SMTP Port</Label>
                                                    <Input
                                                        id="smtp_port"
                                                        type="number"
                                                        value={data.smtp_port || ''}
                                                        onChange={e => setData('smtp_port', e.target.value ? parseInt(e.target.value) : null)}
                                                    />
                                                </div>
                                                <div className="space-y-2">
                                                    <Label htmlFor="smtp_username">Username</Label>
                                                    <Input
                                                        id="smtp_username"
                                                        value={data.smtp_username || ''}
                                                        onChange={e => setData('smtp_username', e.target.value)}
                                                    />
                                                </div>
                                                <div className="space-y-2">
                                                    <Label htmlFor="smtp_password">Password</Label>
                                                    <Input
                                                        id="smtp_password"
                                                        type="password"
                                                        value={data.smtp_password || ''}
                                                        onChange={e => setData('smtp_password', e.target.value)}
                                                    />
                                                </div>
                                                <div className="space-y-2">
                                                    <Label htmlFor="smtp_encryption">Encryption</Label>
                                                    <Select value={data.smtp_encryption || 'none'} onValueChange={val => setData('smtp_encryption', val)}>
                                                        <SelectTrigger id="smtp_encryption"><SelectValue /></SelectTrigger>
                                                        <SelectContent>
                                                            <SelectItem value="none">None</SelectItem>
                                                            <SelectItem value="tls">TLS</SelectItem>
                                                            <SelectItem value="ssl">SSL</SelectItem>
                                                        </SelectContent>
                                                    </Select>
                                                </div>
                                            </div>
                                        </div>
                                    )}
                                </CardContent>
                            </Card>
                        </TabsContent>
                        
                        {/* Notify Tab */}
                        <TabsContent value="notifications" className="space-y-6 mt-6">
                            <Card>
                                <CardHeader>
                                    <CardTitle>Notification Triggers</CardTitle>
                                    <CardDescription>Decide when to send notifications to residents.</CardDescription>
                                </CardHeader>
                                <CardContent className="grid gap-6">
                                    <div className="grid gap-4 md:grid-cols-2">
                                        {[
                                            { id: 'notify_bill_generated', label: 'When Bill is Generated' },
                                            { id: 'notify_payment_received', label: 'When Payment is Received' },
                                            { id: 'notify_visitor_arrival', label: 'On Visitor Arrival' },
                                            { id: 'notify_maintenance_request', label: 'On Maintenance Request' },
                                            { id: 'notify_announcement', label: 'When Announcement is Posted' },
                                        ].map((trigger) => (
                                            <div key={trigger.id} className="flex items-center gap-2">
                                                <Checkbox
                                                    id={trigger.id}
                                                    checked={Boolean(data[trigger.id as keyof typeof data])}
                                                    onCheckedChange={(checked) => setData(trigger.id as any, checked)}
                                                />
                                                <Label htmlFor={trigger.id} className="cursor-pointer">{trigger.label}</Label>
                                            </div>
                                        ))}
                                    </div>
                                </CardContent>
                            </Card>
                        </TabsContent>

                        {/* Security Tab */}
                         <TabsContent value="security" className="space-y-6 mt-6">
                            <Card>
                                <CardHeader>
                                    <CardTitle>Security Settings</CardTitle>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                    <div className="flex items-center gap-2">
                                        <Checkbox
                                            id="enable_2fa"
                                            checked={data.enable_2fa}
                                            onCheckedChange={checked => setData('enable_2fa', !!checked)}
                                        />
                                        <Label htmlFor="enable_2fa">Enable Two-Factor Authentication (2FA)</Label>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <Checkbox
                                            id="require_otp_for_login"
                                            checked={data.require_otp_for_login}
                                            onCheckedChange={checked => setData('require_otp_for_login', !!checked)}
                                        />
                                        <Label htmlFor="require_otp_for_login">Require OTP for Login</Label>
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="session_timeout_minutes">Session Timeout (Minutes)</Label>
                                        <Input
                                            id="session_timeout_minutes"
                                            type="number"
                                            value={data.session_timeout_minutes || ''}
                                            onChange={e => setData('session_timeout_minutes', parseInt(e.target.value))}
                                        />
                                    </div>
                                </CardContent>
                            </Card>
                         </TabsContent>

                         {/* SMS Tab Placeholder */}
                        <TabsContent value="sms" className="space-y-6 mt-6">
                            <Card>
                                <CardHeader>
                                    <CardTitle>SMS Settings</CardTitle>
                                    <CardDescription>Configure your SMS provider details.</CardDescription>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                    <div className="flex items-center gap-2 mb-4">
                                        <Checkbox
                                            id="sms_enabled"
                                            checked={data.sms_enabled}
                                            onCheckedChange={checked => setData('sms_enabled', !!checked)}
                                        />
                                        <Label htmlFor="sms_enabled" className="cursor-pointer font-medium">Enable SMS</Label>
                                    </div>
                                    {data.sms_enabled && (
                                        <div className="grid gap-4 md:grid-cols-2">
                                            <div className="space-y-2">
                                                <Label htmlFor="sms_provider">Provider Name</Label>
                                                <Input
                                                    id="sms_provider"
                                                    value={data.sms_provider || ''}
                                                    onChange={e => setData('sms_provider', e.target.value)}
                                                    placeholder="e.g., Twilio, Msg91"
                                                />
                                                <InputError message={errors?.sms_provider} />
                                            </div>
                                            <div className="space-y-2">
                                                <Label htmlFor="sms_sender_id">Sender ID</Label>
                                                <Input
                                                    id="sms_sender_id"
                                                    value={data.sms_sender_id || ''}
                                                    onChange={e => setData('sms_sender_id', e.target.value)}
                                                />
                                                <InputError message={errors?.sms_sender_id} />
                                            </div>
                                            <div className="space-y-2">
                                                <Label htmlFor="sms_api_key">API Key</Label>
                                                <Input
                                                    id="sms_api_key"
                                                    type="password"
                                                    value={data.sms_api_key || ''}
                                                    onChange={e => setData('sms_api_key', e.target.value)}
                                                />
                                                <InputError message={errors?.sms_api_key} />
                                            </div>
                                            <div className="space-y-2">
                                                <Label htmlFor="sms_username">Username</Label>
                                                <Input
                                                    id="sms_username"
                                                    value={data.sms_username || ''}
                                                    onChange={e => setData('sms_username', e.target.value)}
                                                />
                                                <InputError message={errors?.sms_username} />
                                            </div>
                                            <div className="space-y-2">
                                                <Label htmlFor="sms_password">Password</Label>
                                                <Input
                                                    id="sms_password"
                                                    type="password"
                                                    value={data.sms_password || ''}
                                                    onChange={e => setData('sms_password', e.target.value)}
                                                />
                                                <InputError message={errors?.sms_password} />
                                            </div>
                                        </div>
                                    )}
                                </CardContent>
                            </Card>

                            <Card>
                                <CardHeader>
                                    <CardTitle>WhatsApp Settings</CardTitle>
                                    <CardDescription>Configure WhatsApp API integration.</CardDescription>
                                </CardHeader>
                                <CardContent className="space-y-4">
                                    <div className="flex items-center gap-2 mb-4">
                                        <Checkbox
                                            id="whatsapp_enabled"
                                            checked={data.whatsapp_enabled}
                                            onCheckedChange={checked => setData('whatsapp_enabled', !!checked)}
                                        />
                                        <Label htmlFor="whatsapp_enabled" className="cursor-pointer font-medium">Enable WhatsApp</Label>
                                    </div>
                                    {data.whatsapp_enabled && (
                                        <div className="grid gap-4 md:grid-cols-2">
                                            <div className="space-y-2">
                                                <Label htmlFor="whatsapp_provider">Provider Name</Label>
                                                <Input
                                                    id="whatsapp_provider"
                                                    value={data.whatsapp_provider || ''}
                                                    onChange={e => setData('whatsapp_provider', e.target.value)}
                                                    placeholder="e.g., Twilio, Meta"
                                                />
                                                <InputError message={errors?.whatsapp_provider} />
                                            </div>
                                            <div className="space-y-2">
                                                <Label htmlFor="whatsapp_api_key">API Key</Label>
                                                <Input
                                                    id="whatsapp_api_key"
                                                    type="password"
                                                    value={data.whatsapp_api_key || ''}
                                                    onChange={e => setData('whatsapp_api_key', e.target.value)}
                                                />
                                                <InputError message={errors?.whatsapp_api_key} />
                                            </div>
                                            <div className="space-y-2">
                                                <Label htmlFor="whatsapp_sender_id">Sender ID</Label>
                                                <Input
                                                    id="whatsapp_sender_id"
                                                    value={data.whatsapp_sender_id || ''}
                                                    onChange={e => setData('whatsapp_sender_id', e.target.value)}
                                                />
                                                <InputError message={errors?.whatsapp_sender_id} />
                                            </div>
                                            <div className="space-y-2">
                                                <Label htmlFor="whatsapp_business_id">Business ID</Label>
                                                <Input
                                                    id="whatsapp_business_id"
                                                    value={data.whatsapp_business_id || ''}
                                                    onChange={e => setData('whatsapp_business_id', e.target.value)}
                                                />
                                                <InputError message={errors?.whatsapp_business_id} />
                                            </div>
                                        </div>
                                    )}
                                </CardContent>
                            </Card>


                        </TabsContent>



                    </Tabs>
                </form>
            </div>
        </SocietyAdminLayout>
    );
}
