import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link, router } from '@inertiajs/react';
import { ArrowUpRight, BarChart3, CreditCard, DollarSign, FileText, TrendingDown, TrendingUp, Calendar } from 'lucide-react';
import { useState } from 'react';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Revenue Dashboard',
        href: '/societies/{society}/revenue',
    },
];

interface RecentTransaction {
    id: number;
    type: 'credit' | 'debit';
    amount: number;
    date: string;
    description: string;
    reference: string;
    account_name: string;
    balance_after: number;
}

interface IndexProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    filters: {
        date_from: string;
        date_to: string;
    };
    stats: {
        total_income: number;
        total_expense: number;
        net_revenue: number;
    };
    charts: {
        income_by_category: Record<string, number>;
        expense_by_category: Record<string, number>;
    };
    recent_transactions: RecentTransaction[];
}

export default function Index({ society, filters, stats, charts, recent_transactions }: IndexProps) {
    const [dateFrom, setDateFrom] = useState(filters.date_from);
    const [dateTo, setDateTo] = useState(filters.date_to);

    const handleFilter = () => {
        router.get(`/societies/${society.id}/revenue`, {
            date_from: dateFrom,
            date_to: dateTo,
        }, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 0,
        }).format(amount);
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Revenue Dashboard" />
            <div className="flex h-full flex-1 flex-col gap-6 overflow-x-auto rounded-xl p-6">
                <div className="flex items-center justify-between">
                    <div>
                        <h1 className="text-2xl font-bold">Revenue & Expenses</h1>
                        <p className="text-muted-foreground">
                            Overview of financial performance
                        </p>
                    </div>
                    <div className="flex items-end gap-4">
                        <div className="grid gap-1.5">
                            <Label htmlFor="date_from">From</Label>
                            <Input
                                id="date_from"
                                type="date"
                                value={dateFrom}
                                onChange={(e) => setDateFrom(e.target.value)}
                                className="w-40"
                            />
                        </div>
                        <div className="grid gap-1.5">
                            <Label htmlFor="date_to">To</Label>
                            <Input
                                id="date_to"
                                type="date"
                                value={dateTo}
                                onChange={(e) => setDateTo(e.target.value)}
                                className="w-40"
                            />
                        </div>
                        <Button onClick={handleFilter}>
                            <Calendar className="mr-2 h-4 w-4" />
                            Filter
                        </Button>
                        <Link href={`/societies/${society.id}/revenue/history`}>
                            <Button variant="outline">
                                <FileText className="mr-2 h-4 w-4" />
                                View History
                            </Button>
                        </Link>
                    </div>
                </div>

                {/* KPI Cards */}
                <div className="grid gap-4 md:grid-cols-3">
                    <Card>
                        <CardContent className="p-6">
                            <div className="flex items-center justify-between">
                                <div className="space-y-1">
                                    <p className="text-sm text-muted-foreground">Total Income</p>
                                    <p className="text-2xl font-bold text-green-600 dark:text-green-400">
                                        {formatCurrency(stats.total_income)}
                                    </p>
                                </div>
                                <div className="rounded-full bg-green-100 dark:bg-green-950 p-3">
                                    <TrendingUp className="h-6 w-6 text-green-600 dark:text-green-400" />
                                </div>
                            </div>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardContent className="p-6">
                            <div className="flex items-center justify-between">
                                <div className="space-y-1">
                                    <p className="text-sm text-muted-foreground">Total Expense</p>
                                    <p className="text-2xl font-bold text-red-600 dark:text-red-400">
                                        {formatCurrency(stats.total_expense)}
                                    </p>
                                </div>
                                <div className="rounded-full bg-red-100 dark:bg-red-950 p-3">
                                    <TrendingDown className="h-6 w-6 text-red-600 dark:text-red-400" />
                                </div>
                            </div>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardContent className="p-6">
                            <div className="flex items-center justify-between">
                                <div className="space-y-1">
                                    <p className="text-sm text-muted-foreground">Net Revenue</p>
                                    <p className={`text-2xl font-bold ${stats.net_revenue >= 0 ? 'text-blue-600 dark:text-blue-400' : 'text-red-600 dark:text-red-400'}`}>
                                        {formatCurrency(stats.net_revenue)}
                                    </p>
                                </div>
                                <div className="rounded-full bg-blue-100 dark:bg-blue-950 p-3">
                                    <DollarSign className="h-6 w-6 text-blue-600 dark:text-blue-400" />
                                </div>
                            </div>
                        </CardContent>
                    </Card>
                </div>

                {/* Category Breakdown */}
                <div className="grid gap-6 md:grid-cols-2">
                    <Card>
                        <CardHeader>
                            <CardTitle className="text-base font-semibold">Income Sources</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="space-y-4">
                                {Object.entries(charts.income_by_category).length > 0 ? (
                                    Object.entries(charts.income_by_category).map(([category, amount]) => (
                                        <div key={category} className="flex items-center justify-between">
                                            <div className="flex items-center gap-2">
                                                <div className="h-2 w-2 rounded-full bg-green-500" />
                                                <span className="text-sm font-medium capitalize">
                                                    {category.replace(/_/g, ' ')}
                                                </span>
                                            </div>
                                            <span className="text-sm font-bold">{formatCurrency(amount)}</span>
                                        </div>
                                    ))
                                ) : (
                                    <p className="text-sm text-muted-foreground text-center py-4">No income recorded</p>
                                )}
                            </div>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardHeader>
                            <CardTitle className="text-base font-semibold">Expense Breakdown</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <div className="space-y-4">
                                {Object.entries(charts.expense_by_category).length > 0 ? (
                                    Object.entries(charts.expense_by_category).map(([category, amount]) => (
                                        <div key={category} className="flex items-center justify-between">
                                            <div className="flex items-center gap-2">
                                                <div className="h-2 w-2 rounded-full bg-red-500" />
                                                <span className="text-sm font-medium capitalize">
                                                    {category.replace(/_/g, ' ')}
                                                </span>
                                            </div>
                                            <span className="text-sm font-bold">{formatCurrency(amount)}</span>
                                        </div>
                                    ))
                                ) : (
                                    <p className="text-sm text-muted-foreground text-center py-4">No expenses recorded</p>
                                )}
                            </div>
                        </CardContent>
                    </Card>
                </div>

                {/* Recent Transactions */}
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between">
                        <CardTitle>Recent Transactions</CardTitle>
                        <Link href={`/societies/${society.id}/revenue/history`}>
                            <Button variant="ghost" size="sm">
                                View All <ArrowUpRight className="ml-2 h-4 w-4" />
                            </Button>
                        </Link>
                    </CardHeader>
                    <CardContent>
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead>
                                    <tr className="border-b text-xs uppercase text-muted-foreground">
                                        <th className="px-4 py-3 text-left font-medium">Date</th>
                                        <th className="px-4 py-3 text-left font-medium">Description</th>
                                        <th className="px-4 py-3 text-left font-medium">Reference</th>
                                        <th className="px-4 py-3 text-left font-medium">Account</th>
                                        <th className="px-4 py-3 text-right font-medium">Amount</th>
                                    </tr>
                                </thead>
                                <tbody className="text-sm">
                                    {recent_transactions.length > 0 ? (
                                        recent_transactions.map((tx) => (
                                            <tr key={tx.id} className="border-b hover:bg-muted/50">
                                                <td className="px-4 py-3 whitespace-nowrap">{tx.date}</td>
                                                <td className="px-4 py-3 font-medium">{tx.description}</td>
                                                <td className="px-4 py-3 text-muted-foreground">{tx.reference}</td>
                                                <td className="px-4 py-3 text-muted-foreground">{tx.account_name}</td>
                                                <td className={`px-4 py-3 text-right font-bold ${
                                                    tx.type === 'credit' ? 'text-green-600' : 'text-red-600'
                                                }`}>
                                                    {tx.type === 'credit' ? '+' : '-'}{formatCurrency(tx.amount)}
                                                </td>
                                            </tr>
                                        ))
                                    ) : (
                                        <tr>
                                            <td colSpan={5} className="px-4 py-8 text-center text-muted-foreground">
                                                No recent transactions
                                            </td>
                                        </tr>
                                    )}
                                </tbody>
                            </table>
                        </div>
                    </CardContent>
                </Card>
            </div>
        </SocietyAdminLayout>
    );
}
