import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link, router } from '@inertiajs/react';
import { ArrowLeft, FileText, Search, TrendingDown, TrendingUp } from 'lucide-react';
import { useState } from 'react';

const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Revenue Dashboard', href: '/societies/{society}/revenue' },
    { title: 'History', href: '#' },
];

interface Transaction {
    id: number;
    type: 'credit' | 'debit';
    date: string;
    description: string;
    reference: string;
    payment_method: string;
    debit: number;
    credit: number;
    balance: number;
    bank_account: string;
    category: string;
    created_by_name: string;
}

interface HistoryProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    transactions: {
        data: Transaction[];
        current_page: number;
        per_page: number;
        total: number;
        last_page: number;
        from: number;
        to: number;
    };
    bankAccounts: {
        id: number;
        account_name: string;
    }[];
    filters: {
        search?: string;
        type?: string;
        date_from?: string;
        date_to?: string;
        bank_account_id?: string;
    };
}

export default function History({ society, transactions, bankAccounts, filters }: HistoryProps) {
    const [search, setSearch] = useState(filters.search || '');
    const [type, setType] = useState(filters.type || 'all');
    const [dateFrom, setDateFrom] = useState(filters.date_from || '');
    const [dateTo, setDateTo] = useState(filters.date_to || '');
    const [bankAccount, setBankAccount] = useState(filters.bank_account_id || 'all');

    const handleFilter = () => {
        const filterParams: any = {};
        if (search) filterParams.search = search;
        if (type && type !== 'all') filterParams.type = type;
        if (dateFrom) filterParams.date_from = dateFrom;
        if (dateTo) filterParams.date_to = dateTo;
        if (bankAccount && bankAccount !== 'all') filterParams.bank_account_id = bankAccount;

        router.get(`/societies/${society.id}/revenue/history`, filterParams, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const handlePageChange = (page: number) => {
        const url = new URL(window.location.href);
        url.searchParams.set('page', page.toString());
        router.visit(url.toString(), {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 2,
        }).format(amount);
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Transaction History" />
            <div className="flex h-full flex-1 flex-col gap-6 overflow-x-auto rounded-xl p-6">
                <div className="flex items-center gap-4">
                    <Link href={`/societies/${society.id}/revenue`}>
                        <Button variant="ghost" size="sm">
                            <ArrowLeft className="mr-2 h-4 w-4" />
                            Back to Dashboard
                        </Button>
                    </Link>
                    <div>
                        <h1 className="text-2xl font-bold">Transaction History</h1>
                        <p className="text-muted-foreground">
                            All financial records
                        </p>
                    </div>
                </div>

                {/* Filters */}
                <Card>
                    <CardContent className="pt-6">
                        <div className="flex flex-col gap-4">
                            <div className="flex gap-4">
                                <div className="flex-1">
                                    <Label htmlFor="search">Search</Label>
                                    <div className="relative mt-1">
                                        <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                        <Input
                                            id="search"
                                            placeholder="Search description, reference..."
                                            value={search}
                                            onChange={(e) => setSearch(e.target.value)}
                                            onKeyDown={(e) => e.key === 'Enter' && handleFilter()}
                                            className="pl-9"
                                        />
                                    </div>
                                </div>
                                <div className="w-48">
                                    <Label htmlFor="type">Type</Label>
                                    <Select value={type} onValueChange={setType}>
                                        <SelectTrigger id="type" className="mt-1">
                                            <SelectValue placeholder="All" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="all">All</SelectItem>
                                            <SelectItem value="credit">Credit (Income)</SelectItem>
                                            <SelectItem value="debit">Debit (Expense)</SelectItem>
                                        </SelectContent>
                                    </Select>
                                </div>
                                <div className="w-48">
                                    <Label htmlFor="bank_account">Bank Account</Label>
                                    <Select value={bankAccount} onValueChange={setBankAccount}>
                                        <SelectTrigger id="bank_account" className="mt-1">
                                            <SelectValue placeholder="All Accounts" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="all">All Accounts</SelectItem>
                                            {bankAccounts.map((acc) => (
                                                <SelectItem key={acc.id} value={acc.id.toString()}>
                                                    {acc.account_name}
                                                </SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                </div>
                            </div>
                            <div className="flex gap-4 items-end">
                                <div className="grid gap-1.5">
                                    <Label htmlFor="date_from">From Date</Label>
                                    <Input
                                        id="date_from"
                                        type="date"
                                        value={dateFrom}
                                        onChange={(e) => setDateFrom(e.target.value)}
                                        className="w-40"
                                    />
                                </div>
                                <div className="grid gap-1.5">
                                    <Label htmlFor="date_to">To Date</Label>
                                    <Input
                                        id="date_to"
                                        type="date"
                                        value={dateTo}
                                        onChange={(e) => setDateTo(e.target.value)}
                                        className="w-40"
                                    />
                                </div>
                                <Button onClick={handleFilter} className="ml-auto">
                                    Apply Filters
                                </Button>
                            </div>
                        </div>
                    </CardContent>
                </Card>

                {/* Transactions Table */}
                <Card>
                    <CardHeader>
                        <CardTitle className="flex items-center gap-2">
                            <FileText className="h-5 w-5" />
                            Transactions
                        </CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="overflow-x-auto">
                            <table className="w-full border-collapse">
                                <thead>
                                    <tr className="border-b-2 border-primary">
                                        <th className="px-4 py-3 text-left text-sm font-semibold">Date</th>
                                        <th className="px-4 py-3 text-left text-sm font-semibold">Description</th>
                                        <th className="px-4 py-3 text-left text-sm font-semibold">Category</th>
                                        <th className="px-4 py-3 text-left text-sm font-semibold">Reference</th>
                                        <th className="px-4 py-3 text-left text-sm font-semibold">Issued By</th>
                                        <th className="px-4 py-3 text-left text-sm font-semibold">Account</th>
                                        <th className="px-4 py-3 text-right text-sm font-semibold text-red-600 dark:text-red-400">Debit</th>
                                        <th className="px-4 py-3 text-right text-sm font-semibold text-green-600 dark:text-green-400">Credit</th>
                                        <th className="px-4 py-3 text-right text-sm font-semibold">Balance</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {transactions.data.length > 0 ? (
                                        transactions.data.map((transaction) => (
                                            <tr 
                                                key={transaction.id} 
                                                className={`border-b hover:bg-muted/30 ${
                                                    transaction.type === 'debit' ? 'bg-red-50/30 dark:bg-red-950/10' : 'bg-green-50/30 dark:bg-green-950/10'
                                                }`}
                                            >
                                                <td className="px-4 py-3 text-sm whitespace-nowrap">
                                                    {transaction.date}
                                                </td>
                                                <td className="px-4 py-3 text-sm">
                                                    <div className="flex items-start gap-2">
                                                        {transaction.type === 'credit' ? (
                                                            <TrendingUp className="h-4 w-4 text-green-600 dark:text-green-400 mt-0.5 flex-shrink-0" />
                                                        ) : (
                                                            <TrendingDown className="h-4 w-4 text-red-600 dark:text-red-400 mt-0.5 flex-shrink-0" />
                                                        )}
                                                        <div>
                                                            <div className="font-medium">{transaction.description}</div>
                                                            <div className="text-xs text-muted-foreground capitalize">{transaction.payment_method.replace('_', ' ')}</div>
                                                        </div>
                                                    </div>
                                                </td>
                                                <td className="px-4 py-3 text-sm text-muted-foreground capitalize">
                                                    {transaction.category}
                                                </td>
                                                <td className="px-4 py-3 text-sm text-muted-foreground">
                                                    {transaction.reference}
                                                </td>
                                                <td className="px-4 py-3 text-sm text-muted-foreground">
                                                    {transaction.created_by_name}
                                                </td>
                                                <td className="px-4 py-3 text-sm text-muted-foreground">
                                                    {transaction.bank_account}
                                                </td>
                                                <td className="px-4 py-3 text-sm text-right font-medium">
                                                    {transaction.debit > 0 ? (
                                                        <span className="text-red-600 dark:text-red-400">
                                                            {formatCurrency(transaction.debit)}
                                                        </span>
                                                    ) : '-'}
                                                </td>
                                                <td className="px-4 py-3 text-sm text-right font-medium">
                                                    {transaction.credit > 0 ? (
                                                        <span className="text-green-600 dark:text-green-400">
                                                            {formatCurrency(transaction.credit)}
                                                        </span>
                                                    ) : '-'}
                                                </td>
                                                <td className="px-4 py-3 text-sm text-right font-semibold">
                                                    {formatCurrency(transaction.balance)}
                                                </td>
                                            </tr>
                                        ))
                                    ) : (
                                        <tr>
                                            <td colSpan={8} className="px-4 py-8 text-center text-muted-foreground">
                                                No transactions found matching your filters.
                                            </td>
                                        </tr>
                                    )}
                                </tbody>
                            </table>
                        </div>

                        {/* Pagination */}
                        {transactions.total > transactions.per_page && (
                            <div className="mt-4 flex items-center justify-between border-t pt-4">
                                <div className="text-sm text-muted-foreground">
                                    Showing {transactions.from} to {transactions.to} of {transactions.total} transactions
                                </div>
                                <div className="flex gap-2">
                                    <Button
                                        variant="outline"
                                        size="sm"
                                        onClick={() => handlePageChange(transactions.current_page - 1)}
                                        disabled={transactions.current_page === 1}
                                    >
                                        Previous
                                    </Button>
                                    <Button
                                        variant="outline"
                                        size="sm"
                                        onClick={() => handlePageChange(transactions.current_page + 1)}
                                        disabled={transactions.current_page === transactions.last_page}
                                    >
                                        Next
                                    </Button>
                                </div>
                            </div>
                        )}
                    </CardContent>
                </Card>
            </div>
        </SocietyAdminLayout>
    );
}
