import { Button } from '@/components/ui/button';
import { CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { GlassCard } from '@/components/ui/glass-card';
import { PageHeader } from '@/components/ui/page-header';
import { ModernTable, ModernTableHeader, ModernTableBody, ModernTableFooter, ModernTableHead, ModernTableRow, ModernTableCell } from '@/components/ui/modern-table';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link, router, useForm } from '@inertiajs/react';
import { Users, Plus, Search, Edit, Trash2, Eye, UserCheck, UserX, Building as BuildingIcon, Home, UserPlus, UsersRound } from 'lucide-react';
import { useState, useEffect, useMemo } from 'react';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Residents',
        href: '/societies/{society}/residents',
    },
];

interface Building {
    id: number;
    name: string;
}

interface Unit {
    id: number;
    unit_no: string;
    building_id: number | null;
    building: {
        id: number;
        name: string;
    } | null;
    status: string;
}

interface Resident {
    id: number;
    name: string;
    email: string | null;
    phone: string;
    status: 'active' | 'inactive' | 'suspended';
    directory_visible: boolean;
    unit_id: number | null;
    type: 'owner' | 'tenant' | 'family_member' | null;
    parent_id: number | null;
    unit: {
        id: number;
        unit_no: string;
        building: {
            id: number;
            name: string;
        } | null;
    } | null;
    created_at: string;
}

interface IndexProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    residents: {
        data: Resident[];
        links: any;
        meta: any;
    };
    units: Unit[];
    buildings: Building[];
    filters: {
        search?: string;
        status?: string;
        unit_id?: string;
        directory_visible?: string;
        type?: string;
    };
    stats: {
        total_members: number;
        active_members: number;
        owners: number;
        tenants: number;
        family_members: number;
    };
}

export default function Index({ society, residents, units, buildings, filters, stats }: IndexProps) {
    const [search, setSearch] = useState(filters.search || '');
    const [status, setStatus] = useState(filters.status || 'all');
    const [unitId, setUnitId] = useState(filters.unit_id || 'all');
    const [directoryVisible, setDirectoryVisible] = useState(filters.directory_visible || 'all');
    const [typeFilter, setTypeFilter] = useState(filters.type || 'all');

    // Modal States
    const [isCreateOpen, setIsCreateOpen] = useState(false);
    const [isEditOpen, setIsEditOpen] = useState(false);
    const [isViewOpen, setIsViewOpen] = useState(false);
    const [editingResident, setEditingResident] = useState<Resident | null>(null);
    const [viewingResident, setViewingResident] = useState<Resident | null>(null);

    // Filtered Units for Modals
    const [selectedBuildingId, setSelectedBuildingId] = useState<string>('all');
    
    // Create Form
    const createForm = useForm({
        name: '',
        email: '',
        phone: '',
        password: '',
        password_confirmation: '',
        unit_id: 'none',
        status: 'active',
        directory_visible: true,
    });

    // Edit Form
    const editForm = useForm({
        name: '',
        email: '',
        phone: '',
        password: '',
        password_confirmation: '',
        unit_id: 'none',
        status: 'active',
        directory_visible: true,
    });

    // Reset forms when modals close
    useEffect(() => {
        if (!isCreateOpen) {
            createForm.reset();
            setSelectedBuildingId('all');
        }
    }, [isCreateOpen]);

    // Reset unit_id when building selection changes
    useEffect(() => {
        if (isCreateOpen) {
            createForm.setData('unit_id', 'none');
        }
    }, [selectedBuildingId]);

    useEffect(() => {
        if (!isEditOpen) {
            editForm.reset();
            setEditingResident(null);
            setSelectedBuildingId('all');
        }
    }, [isEditOpen]);

    const handleSearch = () => {
        const filterParams: any = {};
        if (search) filterParams.search = search;
        if (status && status !== 'all') filterParams.status = status;
        if (unitId && unitId !== 'all') filterParams.unit_id = unitId;
        if (directoryVisible && directoryVisible !== 'all') filterParams.directory_visible = directoryVisible;
        if (typeFilter && typeFilter !== 'all') filterParams.type = typeFilter;

        router.get(`/societies/${society.id}/residents`, filterParams, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const handleCreateSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        createForm.post(`/societies/${society.id}/residents`, {
            onSuccess: () => setIsCreateOpen(false),
        });
    };

    const openEditModal = (resident: Resident) => {
        setEditingResident(resident);
        editForm.setData({
            name: resident.name,
            email: resident.email || '',
            phone: resident.phone,
            password: '',
            password_confirmation: '',
            unit_id: resident.unit_id ? resident.unit_id.toString() : 'none',
            status: resident.status,
            directory_visible: resident.directory_visible,
        });
        
        // Set initial building selection if resident has a unit
        if (resident.unit && resident.unit.building) {
            setSelectedBuildingId(resident.unit.building.id.toString());
        } else {
            setSelectedBuildingId('all');
        }
        
        setIsEditOpen(true);
    };

    const handleEditSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        if (!editingResident) return;
        
        editForm.put(`/societies/${society.id}/residents/${editingResident.id}`, {
            onSuccess: () => setIsEditOpen(false),
        });
    };

    const openViewModal = (resident: Resident) => {
        setViewingResident(resident);
        setIsViewOpen(true);
    };

    const handleDelete = (id: number) => {
        if (confirm('Are you sure you want to delete this resident?')) {
            router.delete(`/societies/${society.id}/residents/${id}`, {
                preserveScroll: true,
            });
        }
    };

    // Base filter for building
    const unitsByBuilding = useMemo(() => {
        if (selectedBuildingId === 'all') {
            return units;
        }
        return units.filter(unit => String(unit.building_id) === String(selectedBuildingId));
    }, [units, selectedBuildingId]);

    // Filter for Create Modal (Vacant only)
    const availableUnitsForCreate = useMemo(() => {
        return unitsByBuilding.filter(unit => unit.status === 'vacant');
    }, [unitsByBuilding]);

    // Filter for Edit Modal (Vacant + Current User's Unit)
    const availableUnitsForEdit = useMemo(() => {
        return unitsByBuilding.filter(unit => 
            unit.status === 'vacant' || 
            (editingResident?.unit_id && unit.id === editingResident.unit_id)
        );
    }, [unitsByBuilding, editingResident]);

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Residents" />
            <div className="flex h-full flex-1 flex-col gap-8 p-6 md:p-8 overflow-x-hidden">
                <PageHeader 
                    title="Residents"
                    description="Manage all residents in your society, their status, and unit assignments."
                    breadcrumbs={breadcrumbs}
                    actions={
                        <Dialog open={isCreateOpen} onOpenChange={setIsCreateOpen}>
                            <DialogTrigger asChild>
                                <Button className="bg-gradient-to-r from-sky-600 to-cyan-600 hover:from-sky-700 hover:to-cyan-700 text-white shadow-lg shadow-cyan-500/20 transition-all duration-300 hover:scale-[1.02]">
                                    <Plus className="mr-2 h-4 w-4" />
                                    Add Resident
                                </Button>
                            </DialogTrigger>
                            <DialogContent className="sm:max-w-[600px]">
                                <DialogHeader>
                                    <DialogTitle>Add New Resident</DialogTitle>
                                    <DialogDescription>
                                        Create a new resident account and assign them to a unit.
                                    </DialogDescription>
                                </DialogHeader>
                                <form onSubmit={handleCreateSubmit} className="space-y-4 py-4">
                                    <div className="grid grid-cols-2 gap-4">
                                        <div className="space-y-2">
                                            <Label htmlFor="name">Full Name</Label>
                                            <Input
                                                id="name"
                                                value={createForm.data.name}
                                                onChange={(e) => createForm.setData('name', e.target.value)}
                                                placeholder="John Doe"
                                                required
                                            />
                                            {createForm.errors.name && <p className="text-sm text-red-500">{createForm.errors.name}</p>}
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="phone">Phone Number</Label>
                                            <Input
                                                id="phone"
                                                value={createForm.data.phone}
                                                onChange={(e) => createForm.setData('phone', e.target.value)}
                                                placeholder="9876543210"
                                                required
                                            />
                                            {createForm.errors.phone && <p className="text-sm text-red-500">{createForm.errors.phone}</p>}
                                        </div>
                                    </div>

                                    <div className="space-y-2">
                                        <Label htmlFor="email">Email Address (Optional)</Label>
                                        <Input
                                            id="email"
                                            type="email"
                                            value={createForm.data.email}
                                            onChange={(e) => createForm.setData('email', e.target.value)}
                                            placeholder="john@example.com"
                                        />
                                        {createForm.errors.email && <p className="text-sm text-red-500">{createForm.errors.email}</p>}
                                    </div>

                                    <div className="grid grid-cols-2 gap-4">
                                        <div className="space-y-2">
                                            <Label htmlFor="password">Password</Label>
                                            <Input
                                                id="password"
                                                type="password"
                                                value={createForm.data.password}
                                                onChange={(e) => createForm.setData('password', e.target.value)}
                                                required
                                            />
                                            {createForm.errors.password && <p className="text-sm text-red-500">{createForm.errors.password}</p>}
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="password_confirmation">Confirm Password</Label>
                                            <Input
                                                id="password_confirmation"
                                                type="password"
                                                value={createForm.data.password_confirmation}
                                                onChange={(e) => createForm.setData('password_confirmation', e.target.value)}
                                                required
                                            />
                                        </div>
                                    </div>

                                    <div className="grid grid-cols-2 gap-4">
                                        <div className="space-y-2">
                                            <Label htmlFor="building_select">Building (Filter)</Label>
                                            <Select value={selectedBuildingId} onValueChange={setSelectedBuildingId}>
                                                <SelectTrigger id="building_select">
                                                    <SelectValue placeholder="Select Building" />
                                                </SelectTrigger>
                                                <SelectContent>
                                                    <SelectItem value="all">All Buildings</SelectItem>
                                                    {buildings.map((building) => (
                                                        <SelectItem key={building.id} value={building.id.toString()}>
                                                            {building.name}
                                                        </SelectItem>
                                                    ))}
                                                </SelectContent>
                                            </Select>
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="unit_id">Assign Unit</Label>
                                            <Select 
                                                value={createForm.data.unit_id} 
                                                onValueChange={(value) => createForm.setData('unit_id', value)}
                                                disabled={selectedBuildingId === 'all'}
                                            >
                                                <SelectTrigger id="unit_id">
                                                    <SelectValue placeholder={selectedBuildingId === 'all' ? "Select Building First" : "Select Unit"} />
                                                </SelectTrigger>
                                                <SelectContent>
                                                    <SelectItem value="none">No Unit Assigned</SelectItem>
                                                    {availableUnitsForCreate.map((unit) => (
                                                        <SelectItem key={unit.id} value={unit.id.toString()}>
                                                            {unit.unit_no} {unit.building ? `(${unit.building.name})` : ''}
                                                        </SelectItem>
                                                    ))}
                                                </SelectContent>
                                            </Select>
                                            {createForm.errors.unit_id && <p className="text-sm text-red-500">{createForm.errors.unit_id}</p>}
                                        </div>
                                    </div>

                                    <div className="flex items-center space-x-2">
                                        <input
                                            type="checkbox"
                                            id="directory_visible"
                                            className="h-4 w-4 rounded border-gray-300 text-primary focus:ring-primary"
                                            checked={createForm.data.directory_visible}
                                            onChange={(e) => createForm.setData('directory_visible', e.target.checked)}
                                        />
                                        <Label htmlFor="directory_visible">Show in Resident Directory</Label>
                                    </div>

                                    <DialogFooter>
                                        <Button type="button" variant="outline" onClick={() => setIsCreateOpen(false)}>Cancel</Button>
                                        <Button type="submit" disabled={createForm.processing}>
                                            {createForm.processing ? 'Creating...' : 'Create Resident'}
                                        </Button>
                                    </DialogFooter>
                                </form>
                            </DialogContent>
                        </Dialog>
                    }
                />

                {/* KPI Cards */}
                <div className="grid gap-4 md:grid-cols-5">
                    <GlassCard className="border-l-4 border-l-blue-500 py-6 flex flex-col gap-4" variant="hover">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Total Members</CardTitle>
                            <div className="h-8 w-8 rounded-lg bg-blue-100 flex items-center justify-center text-blue-600">
                                <Users className="h-4 w-4" />
                            </div>
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.total_members}</div>
                            <p className="text-xs text-muted-foreground mt-1">All registered</p>
                        </CardContent>
                    </GlassCard>
                    <GlassCard className="border-l-4 border-l-green-500 py-6 flex flex-col gap-4" variant="hover">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Active</CardTitle>
                            <div className="h-8 w-8 rounded-lg bg-green-100 flex items-center justify-center text-green-600">
                                <UserCheck className="h-4 w-4" />
                            </div>
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.active_members}</div>
                            <p className="text-xs text-muted-foreground mt-1">Active accounts</p>
                        </CardContent>
                    </GlassCard>
                    <GlassCard className="border-l-4 border-l-cyan-500 py-6 flex flex-col gap-4" variant="hover">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Owners</CardTitle>
                            <div className="h-8 w-8 rounded-lg bg-cyan-100 flex items-center justify-center text-cyan-600">
                                <Home className="h-4 w-4" />
                            </div>
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.owners}</div>
                            <p className="text-xs text-muted-foreground mt-1">Primary residents</p>
                        </CardContent>
                    </GlassCard>
                    <GlassCard className="border-l-4 border-l-orange-500 py-6 flex flex-col gap-4" variant="hover">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Tenants</CardTitle>
                            <div className="h-8 w-8 rounded-lg bg-orange-100 flex items-center justify-center text-orange-600">
                                <UserPlus className="h-4 w-4" />
                            </div>
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.tenants}</div>
                            <p className="text-xs text-muted-foreground mt-1">Renting units</p>
                        </CardContent>
                    </GlassCard>
                    <GlassCard className="border-l-4 border-l-purple-500 py-6 flex flex-col gap-4" variant="hover">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Family</CardTitle>
                            <div className="h-8 w-8 rounded-lg bg-purple-100 flex items-center justify-center text-purple-600">
                                <UsersRound className="h-4 w-4" />
                            </div>
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.family_members}</div>
                            <p className="text-xs text-muted-foreground mt-1">Family members</p>
                        </CardContent>
                    </GlassCard>
                </div>

                {/* Filters */}
                <GlassCard>
                    <CardContent className="py-6">
                        <div className="grid gap-4 md:grid-cols-6 items-end">
                            <div className="md:col-span-2">
                                <Label htmlFor="search">Search</Label>
                                <div className="relative mt-1">
                                    <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                    <Input
                                        id="search"
                                        placeholder="Search by name, phone, or email..."
                                        value={search}
                                        onChange={(e) => setSearch(e.target.value)}
                                        onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                        className="pl-9 bg-background/50"
                                    />
                                </div>
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="status">Status</Label>
                                <Select value={status} onValueChange={setStatus}>
                                    <SelectTrigger id="status" className="bg-background/50">
                                        <SelectValue />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Status</SelectItem>
                                        <SelectItem value="active">Active</SelectItem>
                                        <SelectItem value="inactive">Inactive</SelectItem>
                                        <SelectItem value="suspended">Suspended</SelectItem>
                                    </SelectContent>
                                </Select>
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="unit_id">Unit</Label>
                                <Select value={unitId} onValueChange={setUnitId}>
                                    <SelectTrigger id="unit_id" className="bg-background/50">
                                        <SelectValue />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Units</SelectItem>
                                        {units.map((unit) => (
                                            <SelectItem key={unit.id} value={unit.id.toString()}>
                                                {unit.unit_no}
                                                {unit.building && ` - ${unit.building.name}`}
                                            </SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="directory_visible">Directory Visible</Label>
                                <Select value={directoryVisible} onValueChange={setDirectoryVisible}>
                                    <SelectTrigger id="directory_visible" className="bg-background/50">
                                        <SelectValue />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All</SelectItem>
                                        <SelectItem value="1">Visible</SelectItem>
                                        <SelectItem value="0">Hidden</SelectItem>
                                    </SelectContent>
                                </Select>
                            </div>

                            <Button onClick={handleSearch} className="bg-primary hover:bg-primary/90">Filter</Button>
                        </div>
                    </CardContent>
                </GlassCard>

                {/* Residents Table */}
                {/* Residents Table */}
                <div className="space-y-4">
                    <div className="flex items-center justify-between px-1">
                        <h2 className="text-lg font-semibold tracking-tight">All Residents</h2>
                    </div>
                    
                    <ModernTable>
                        <ModernTableHeader>
                            <ModernTableRow>
                                <ModernTableHead>Name</ModernTableHead>
                                <ModernTableHead>Phone</ModernTableHead>
                                <ModernTableHead>Email</ModernTableHead>
                                <ModernTableHead>Unit</ModernTableHead>
                                <ModernTableHead>Status</ModernTableHead>
                                <ModernTableHead>Directory</ModernTableHead>
                                <ModernTableHead className="text-right">Actions</ModernTableHead>
                            </ModernTableRow>
                        </ModernTableHeader>
                        <ModernTableBody>
                            {residents.data.length > 0 ? (
                                residents.data.map((resident) => (
                                    <ModernTableRow key={resident.id}>
                                        <ModernTableCell>
                                            <div className="flex items-center gap-3">
                                                <div className="h-9 w-9 rounded-full bg-gradient-to-br from-indigo-400 to-purple-600 flex items-center justify-center text-white font-bold text-xs shadow-md shadow-indigo-500/20">
                                                    {resident.name.charAt(0)}
                                                </div>
                                                <div className="flex flex-col">
                                                    <span className="font-semibold text-sm">{resident.name}</span>
                                                    {resident.type && (
                                                        <span className={`inline-flex w-fit rounded-full px-1.5 py-0.5 text-[10px] font-semibold mt-0.5 ${
                                                            resident.type === 'owner' 
                                                                ? 'bg-cyan-100 text-cyan-800 dark:bg-cyan-900/30 dark:text-cyan-400'
                                                                : resident.type === 'tenant'
                                                                ? 'bg-orange-100 text-orange-800 dark:bg-orange-900/30 dark:text-orange-400'
                                                                : 'bg-purple-100 text-purple-800 dark:bg-purple-900/30 dark:text-purple-400'
                                                        }`}>
                                                            {resident.type === 'owner' ? 'Owner' : resident.type === 'tenant' ? 'Tenant' : 'Family'}
                                                        </span>
                                                    )}
                                                </div>
                                            </div>
                                        </ModernTableCell>
                                        <ModernTableCell>{resident.phone}</ModernTableCell>
                                        <ModernTableCell className="text-muted-foreground">
                                            {resident.email || '-'}
                                        </ModernTableCell>
                                        <ModernTableCell>
                                            {resident.unit ? (
                                                <span className="inline-flex items-center gap-1.5 px-2 py-1 rounded-md bg-muted/50 text-xs font-medium">
                                                    <BuildingIcon className="h-3 w-3 text-muted-foreground" />
                                                    {resident.unit.unit_no}
                                                    {resident.unit.building && <span className="text-muted-foreground opacity-70">({resident.unit.building.name})</span>}
                                                </span>
                                            ) : (
                                                <span className="text-muted-foreground text-xs italic">Unassigned</span>
                                            )}
                                        </ModernTableCell>
                                        <ModernTableCell>
                                            <span
                                                className={`inline-flex rounded-full px-2 py-1 text-xs font-semibold ${
                                                    resident.status === 'active'
                                                        ? 'bg-green-100 text-green-700 dark:bg-green-500/20 dark:text-green-400'
                                                        : resident.status === 'inactive'
                                                        ? 'bg-gray-100 text-gray-700 dark:bg-gray-800 dark:text-gray-400'
                                                        : 'bg-red-100 text-red-700 dark:bg-red-500/20 dark:text-red-400'
                                                }`}
                                            >
                                                {resident.status.charAt(0).toUpperCase() + resident.status.slice(1)}
                                            </span>
                                        </ModernTableCell>
                                        <ModernTableCell>
                                            {resident.directory_visible ? (
                                                <span className="text-green-600 dark:text-green-400 flex items-center gap-1.5 text-xs font-medium">
                                                    <Eye className="h-3 w-3" /> Visible
                                                </span>
                                            ) : (
                                                <span className="text-muted-foreground flex items-center gap-1.5 text-xs font-medium opacity-70">
                                                    <Eye className="h-3 w-3" /> Hidden
                                                </span>
                                            )}
                                        </ModernTableCell>
                                        <ModernTableCell className="text-right">
                                            <div className="flex items-center justify-end gap-1">
                                                <Button 
                                                    variant="ghost" 
                                                    size="icon" 
                                                    className="h-8 w-8 text-blue-600 hover:text-blue-700 hover:bg-blue-50 dark:hover:bg-blue-900/20"
                                                    onClick={() => openViewModal(resident)}
                                                >
                                                    <Eye className="h-4 w-4" />
                                                </Button>
                                                <Button 
                                                    variant="ghost" 
                                                    size="icon" 
                                                    className="h-8 w-8 text-amber-600 hover:text-amber-700 hover:bg-amber-50 dark:hover:bg-amber-900/20"
                                                    onClick={() => openEditModal(resident)}
                                                >
                                                    <Edit className="h-4 w-4" />
                                                </Button>
                                                <Button
                                                    variant="ghost"
                                                    size="icon"
                                                    className="h-8 w-8 text-red-600 hover:text-red-700 hover:bg-red-50 dark:hover:bg-red-900/20"
                                                    onClick={() => handleDelete(resident.id)}
                                                >
                                                    <Trash2 className="h-4 w-4" />
                                                </Button>
                                            </div>
                                        </ModernTableCell>
                                    </ModernTableRow>
                                ))
                            ) : (
                                <ModernTableRow>
                                    <ModernTableCell colSpan={7} className="h-32 text-center text-muted-foreground">
                                        <div className="flex flex-col items-center gap-2">
                                            <Users className="h-10 w-10 text-muted-foreground/30" />
                                            <p>No residents found matching your criteria</p>
                                        </div>
                                    </ModernTableCell>
                                </ModernTableRow>
                            )}
                        </ModernTableBody>
                    </ModernTable>
                </div>

                        {/* Pagination */}
                        {residents.links && residents.links.length > 3 && (
                            <div className="mt-4 flex items-center justify-between p-4 border-t border-border/10">
                                <div className="text-sm text-muted-foreground">
                                    Showing <span className="font-medium text-foreground">{residents.meta.from}</span> to <span className="font-medium text-foreground">{residents.meta.to}</span> of <span className="font-medium text-foreground">{residents.meta.total}</span> results
                                </div>
                                <div className="flex gap-2">
                                    {residents.links.map((link: any, index: number) => (
                                        <Link
                                            key={index}
                                            href={link.url || '#'}
                                            className={`px-3 py-1 rounded-md text-sm font-medium transition-colors ${
                                                link.active
                                                    ? 'bg-primary text-primary-foreground shadow-sm'
                                                    : 'bg-muted/50 hover:bg-muted text-muted-foreground'
                                            }`}
                                            dangerouslySetInnerHTML={{ __html: link.label }}
                                        />
                                    ))}
                                </div>
                            </div>
                        )}


                {/* Edit Dialog */}
                <Dialog open={isEditOpen} onOpenChange={setIsEditOpen}>
                    <DialogContent className="sm:max-w-[600px]">
                        <DialogHeader>
                            <DialogTitle>Edit Resident</DialogTitle>
                            <DialogDescription>
                                Update resident details and unit assignment.
                            </DialogDescription>
                        </DialogHeader>
                        <form onSubmit={handleEditSubmit} className="space-y-4 py-4">
                            <div className="grid grid-cols-2 gap-4">
                                <div className="space-y-2">
                                    <Label htmlFor="edit_name">Full Name</Label>
                                    <Input
                                        id="edit_name"
                                        value={editForm.data.name}
                                        onChange={(e) => editForm.setData('name', e.target.value)}
                                        required
                                    />
                                    {editForm.errors.name && <p className="text-sm text-red-500">{editForm.errors.name}</p>}
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_phone">Phone Number</Label>
                                    <Input
                                        id="edit_phone"
                                        value={editForm.data.phone}
                                        onChange={(e) => editForm.setData('phone', e.target.value)}
                                        required
                                    />
                                    {editForm.errors.phone && <p className="text-sm text-red-500">{editForm.errors.phone}</p>}
                                </div>
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="edit_email">Email Address (Optional)</Label>
                                <Input
                                    id="edit_email"
                                    type="email"
                                    value={editForm.data.email}
                                    onChange={(e) => editForm.setData('email', e.target.value)}
                                />
                                {editForm.errors.email && <p className="text-sm text-red-500">{editForm.errors.email}</p>}
                            </div>

                            <div className="grid grid-cols-2 gap-4">
                                <div className="space-y-2">
                                    <Label htmlFor="edit_password">New Password (Optional)</Label>
                                    <Input
                                        id="edit_password"
                                        type="password"
                                        value={editForm.data.password}
                                        onChange={(e) => editForm.setData('password', e.target.value)}
                                        placeholder="Leave blank to keep current"
                                    />
                                    {editForm.errors.password && <p className="text-sm text-red-500">{editForm.errors.password}</p>}
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_password_confirmation">Confirm New Password</Label>
                                    <Input
                                        id="edit_password_confirmation"
                                        type="password"
                                        value={editForm.data.password_confirmation}
                                        onChange={(e) => editForm.setData('password_confirmation', e.target.value)}
                                    />
                                </div>
                            </div>

                            <div className="grid grid-cols-2 gap-4">
                                <div className="space-y-2">
                                    <Label htmlFor="edit_building_select">Building (Filter)</Label>
                                    <Select value={selectedBuildingId} onValueChange={setSelectedBuildingId}>
                                        <SelectTrigger id="edit_building_select">
                                            <SelectValue placeholder="Select Building" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="all">All Buildings</SelectItem>
                                            {buildings.map((building) => (
                                                <SelectItem key={building.id} value={building.id.toString()}>
                                                    {building.name}
                                                </SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_unit_id">Assign Unit</Label>
                                    <Select 
                                        value={editForm.data.unit_id} 
                                        onValueChange={(value) => editForm.setData('unit_id', value)}
                                        disabled={selectedBuildingId === 'all'}
                                    >
                                        <SelectTrigger id="edit_unit_id">
                                            <SelectValue placeholder={selectedBuildingId === 'all' ? "Select Building First" : "Select Unit"} />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="none">No Unit Assigned</SelectItem>
                                            {availableUnitsForEdit.map((unit) => (
                                                <SelectItem key={unit.id} value={unit.id.toString()}>
                                                    {unit.unit_no} {unit.building ? `(${unit.building.name})` : ''}
                                                </SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                    {editForm.errors.unit_id && <p className="text-sm text-red-500">{editForm.errors.unit_id}</p>}
                                </div>
                            </div>

                            <div className="grid grid-cols-2 gap-4">
                                <div className="space-y-2">
                                    <Label htmlFor="edit_status">Status</Label>
                                    <Select 
                                        value={editForm.data.status} 
                                        onValueChange={(value) => editForm.setData('status', value)}
                                    >
                                        <SelectTrigger id="edit_status">
                                            <SelectValue />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="active">Active</SelectItem>
                                            <SelectItem value="inactive">Inactive</SelectItem>
                                            <SelectItem value="suspended">Suspended</SelectItem>
                                        </SelectContent>
                                    </Select>
                                </div>
                                <div className="flex items-center space-x-2 mt-8">
                                    <input
                                        type="checkbox"
                                        id="edit_directory_visible"
                                        className="h-4 w-4 rounded border-gray-300 text-primary focus:ring-primary"
                                        checked={editForm.data.directory_visible}
                                        onChange={(e) => editForm.setData('directory_visible', e.target.checked)}
                                    />
                                    <Label htmlFor="edit_directory_visible">Show in Directory</Label>
                                </div>
                            </div>

                            <DialogFooter>
                                <Button type="button" variant="outline" onClick={() => setIsEditOpen(false)}>Cancel</Button>
                                <Button type="submit" disabled={editForm.processing}>
                                    {editForm.processing ? 'Saving...' : 'Save Changes'}
                                </Button>
                            </DialogFooter>
                        </form>
                    </DialogContent>
                </Dialog>

                {/* View Dialog */}
                <Dialog open={isViewOpen} onOpenChange={setIsViewOpen}>
                    <DialogContent className="sm:max-w-[600px]">
                        <DialogHeader>
                            <DialogTitle>Resident Details</DialogTitle>
                            <DialogDescription>
                                View detailed information about this resident.
                            </DialogDescription>
                        </DialogHeader>
                        {viewingResident && (
                            <div className="space-y-6 py-4">
                                <div className="flex items-center gap-4">
                                    <div className="h-16 w-16 rounded-full bg-indigo-100 flex items-center justify-center text-indigo-600 font-bold text-2xl">
                                        {viewingResident.name.charAt(0)}
                                    </div>
                                    <div>
                                        <h3 className="text-xl font-bold">{viewingResident.name}</h3>
                                        <p className="text-muted-foreground">{viewingResident.email || 'No email provided'}</p>
                                    </div>
                                </div>

                                <div className="grid grid-cols-2 gap-4">
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground">Phone</h4>
                                        <p className="text-lg font-semibold">{viewingResident.phone}</p>
                                    </div>
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground">Status</h4>
                                        <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium mt-1 ${
                                            viewingResident.status === 'active' ? 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400' :
                                            viewingResident.status === 'inactive' ? 'bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-400' :
                                            'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400'
                                        }`}>
                                            {viewingResident.status.charAt(0).toUpperCase() + viewingResident.status.slice(1)}
                                        </span>
                                    </div>
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground">Unit</h4>
                                        <p className="text-base">
                                            {viewingResident.unit ? (
                                                <span>{viewingResident.unit.unit_no} {viewingResident.unit.building && `(${viewingResident.unit.building.name})`}</span>
                                            ) : (
                                                '-'
                                            )}
                                        </p>
                                    </div>
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground">Directory Visibility</h4>
                                        <p className="text-base">{viewingResident.directory_visible ? 'Visible' : 'Hidden'}</p>
                                    </div>
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground">Joined</h4>
                                        <p className="text-base">{new Date(viewingResident.created_at).toLocaleDateString()}</p>
                                    </div>
                                </div>
                            </div>
                        )}
                        <DialogFooter>
                            <Button onClick={() => setIsViewOpen(false)}>Close</Button>
                        </DialogFooter>
                    </DialogContent>
                </Dialog>
            </div>
        </SocietyAdminLayout>
    );
}
