import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head } from '@inertiajs/react';
import {
    BarChart3,
    Download,
    FileText,
    TrendingDown,
    TrendingUp,
    DollarSign,
    Calendar,
    Wallet,
    CreditCard,
    ArrowUpRight,
    ArrowDownRight,
    PieChart as PieChartIcon,
    Activity,
    FileInput
} from 'lucide-react';
import { useState, useEffect } from 'react';
import {
    AreaChart,
    Area,
    XAxis,
    YAxis,
    CartesianGrid,
    Tooltip,
    ResponsiveContainer,
    PieChart,
    Pie,
    Cell,
    BarChart,
    Bar,
    Legend
} from 'recharts';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Reports',
        href: '/societies/{society}/reports',
    },
];

interface IndexProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    reportType?: string;
}

const COLORS = ['#8b5cf6', '#10b981', '#f59e0b', '#ef4444', '#3b82f6', '#ec4899', '#6366f1'];

export default function Index({ society, reportType = 'collection' }: IndexProps) {
    const [activeTab, setActiveTab] = useState(reportType);
    const [dateFrom, setDateFrom] = useState(new Date(new Date().getFullYear(), new Date().getMonth(), 1).toISOString().split('T')[0]);
    const [dateTo, setDateTo] = useState(new Date().toISOString().split('T')[0]);
    const [loading, setLoading] = useState(false);
    const [reportData, setReportData] = useState<any>(null);
    const [billStatus, setBillStatus] = useState<string>('all');

    const fetchReport = async (type: string) => {
        setLoading(true);
        try {
            const params = new URLSearchParams({ date_from: dateFrom, date_to: dateTo });
            let url = '';

            switch (type) {
                case 'collection':
                    url = `/societies/${society.id}/reports/collection-summary?${params}`;
                    break;
                case 'arrears':
                    // Pass status filter for bills report
                    params.append('status', billStatus);
                    url = `/societies/${society.id}/reports/arrears?${params}`;
                    break;
                case 'expense':
                    url = `/societies/${society.id}/reports/expense-breakdown?${params}`;
                    break;
                case 'financial':
                    url = `/societies/${society.id}/reports/financial-summary?${params}`;
                    break;
            }

            if (url) {
                const response = await fetch(url);
                const data = await response.json();
                setReportData(data);
            }
        } catch (error) {
            console.error('Error fetching report:', error);
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchReport(activeTab);
    }, [activeTab]); 

    // Re-fetch when bill status changes if on the bills tab
    useEffect(() => {
        if (activeTab === 'arrears') {
            fetchReport('arrears');
        }
    }, [billStatus]);

    const handleApplyFilter = () => {
        fetchReport(activeTab);
    };

    const handleExport = (format: 'excel' | 'pdf') => {
        alert(`Export to ${format.toUpperCase()} feature coming soon!`);
    };

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 0,
        }).format(amount);
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Reports" />
            <div className="flex flex-1 flex-col gap-8 p-6 md:p-8 overflow-x-hidden bg-muted/10">
                {/* Header Banner */}
                <div className="relative overflow-hidden rounded-2xl bg-gradient-to-br from-indigo-600 via-sky-600 to-pink-600 p-8 text-white shadow-lg">
                    <div className="absolute top-0 right-0 -mt-10 -mr-10 h-64 w-64 rounded-full bg-white/10 blur-3xl" />
                    <div className="absolute bottom-0 left-0 -mb-10 -ml-10 h-64 w-64 rounded-full bg-white/10 blur-3xl" />
                    
                    <div className="relative z-10 flex flex-col md:flex-row justify-between items-start md:items-center gap-6">
                        <div>
                            <h1 className="text-3xl font-bold tracking-tight text-white">
                                Analytics & Reports
                            </h1>
                            <p className="mt-2 text-indigo-100 text-lg">
                                Comprehensive insights into your society's financial health.
                            </p>
                        </div>
                        <div className="flex gap-3">
                            <Button 
                                variant="secondary" 
                                className="bg-white/10 hover:bg-white/20 text-white border-0 backdrop-blur-md transition-all shadow-inner"
                                onClick={() => handleExport('excel')}
                            >
                                <Download className="mr-2 h-4 w-4" />
                                Export Excel
                            </Button>
                            <Button 
                                variant="secondary" 
                                className="bg-white text-indigo-900 hover:bg-white/90 border-0 shadow-lg hover:shadow-xl transition-all"
                                onClick={() => handleExport('pdf')}
                            >
                                <FileText className="mr-2 h-4 w-4" />
                                Export PDF
                            </Button>
                        </div>
                    </div>
                </div>

                {/* Date Filter Bar */}
                <div className="flex flex-col md:flex-row items-center justify-between gap-4 bg-background p-4 rounded-xl border shadow-sm">
                    <div className="flex items-center gap-2 text-muted-foreground">
                        <Calendar className="h-5 w-5 text-indigo-600" />
                        <span className="font-medium">Date Range Filter</span>
                    </div>
                    <div className="flex flex-wrap items-center gap-4">
                        <div className="flex items-center gap-2">
                            <Label htmlFor="date_from" className="whitespace-nowrap">From</Label>
                            <Input
                                id="date_from"
                                type="date"
                                className="w-[160px]"
                                value={dateFrom}
                                onChange={(e) => setDateFrom(e.target.value)}
                            />
                        </div>
                        <div className="flex items-center gap-2">
                            <Label htmlFor="date_to" className="whitespace-nowrap">To</Label>
                            <Input
                                id="date_to"
                                type="date"
                                className="w-[160px]"
                                value={dateTo}
                                onChange={(e) => setDateTo(e.target.value)}
                            />
                        </div>
                        <Button onClick={handleApplyFilter} disabled={loading} className="bg-indigo-600 hover:bg-indigo-700">
                            {loading ? 'Refreshing...' : 'Apply Filter'}
                        </Button>
                    </div>
                </div>

                <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full space-y-8">
                    <TabsList className="grid w-full grid-cols-2 lg:grid-cols-4 bg-muted/50 p-1 rounded-xl">
                        <TabsTrigger value="collection" className="rounded-lg data-[state=active]:bg-white data-[state=active]:text-indigo-600 data-[state=active]:shadow-sm py-3 font-medium">
                            <Wallet className="mr-2 h-4 w-4" />
                            Collection Summary
                        </TabsTrigger>
                        <TabsTrigger value="arrears" className="rounded-lg data-[state=active]:bg-white data-[state=active]:text-red-600 data-[state=active]:shadow-sm py-3 font-medium">
                            <FileInput className="mr-2 h-4 w-4" />
                            Bills Report
                        </TabsTrigger>
                        <TabsTrigger value="expense" className="rounded-lg data-[state=active]:bg-white data-[state=active]:text-orange-600 data-[state=active]:shadow-sm py-3 font-medium">
                            <BarChart3 className="mr-2 h-4 w-4" />
                            Expense Breakdown
                        </TabsTrigger>
                        <TabsTrigger value="financial" className="rounded-lg data-[state=active]:bg-white data-[state=active]:text-green-600 data-[state=active]:shadow-sm py-3 font-medium">
                            <TrendingUp className="mr-2 h-4 w-4" />
                            Financial Summary
                        </TabsTrigger>
                    </TabsList>

                    {/* Collection Summary */}
                    <TabsContent value="collection" className="space-y-6 animate-in fade-in-50 duration-300">
                        {loading ? (
                            <div className="h-64 flex items-center justify-center text-muted-foreground">Loading breakdown...</div>
                        ) : reportData ? (
                            <>
                                <div className="grid gap-6 md:grid-cols-3">
                                    <Card className="col-span-2 border-t-4 border-t-indigo-500 shadow-md">
                                        <CardHeader>
                                            <CardTitle className="flex items-center gap-2">
                                                <Activity className="h-5 w-5 text-indigo-500" />
                                                Daily Collection Trend
                                            </CardTitle>
                                            <CardDescription>Daily payment inflows over the selected period</CardDescription>
                                        </CardHeader>
                                        <CardContent>
                                            <div className="h-[300px] w-full">
                                                <ResponsiveContainer width="100%" height="100%">
                                                    <AreaChart data={reportData.daily_breakdown || []}>
                                                        <defs>
                                                            <linearGradient id="colorAmount" x1="0" y1="0" x2="0" y2="1">
                                                                <stop offset="5%" stopColor="#6366f1" stopOpacity={0.1}/>
                                                                <stop offset="95%" stopColor="#6366f1" stopOpacity={0}/>
                                                            </linearGradient>
                                                        </defs>
                                                        <XAxis dataKey="date" tickLine={false} axisLine={false} tick={{ fontSize: 12 }} tickFormatter={(val) => new Date(val).toLocaleDateString(undefined, { day: '2-digit', month: 'short' })} />
                                                        <YAxis tickLine={false} axisLine={false} tick={{ fontSize: 12 }} tickFormatter={(val) => `₹${val/1000}k`} />
                                                        <CartesianGrid vertical={false} strokeDasharray="3 3" stroke="#e5e7eb" />
                                                        <Tooltip 
                                                            contentStyle={{ borderRadius: '12px', border: 'none', boxShadow: '0 4px 12px rgba(0,0,0,0.1)' }}
                                                            formatter={(value: number) => [formatCurrency(value), 'Amount']}
                                                            labelFormatter={(label) => new Date(label).toDateString()}    
                                                        />
                                                        <Area type="monotone" dataKey="amount" stroke="#6366f1" strokeWidth={3} fillOpacity={1} fill="url(#colorAmount)" />
                                                    </AreaChart>
                                                </ResponsiveContainer>
                                            </div>
                                        </CardContent>
                                    </Card>

                                    <div className="space-y-6">
                                        <Card className="border-t-4 border-t-green-500 shadow-md">
                                            <CardHeader className="pb-2">
                                                <CardTitle className="text-sm font-medium text-muted-foreground">Total Collected</CardTitle>
                                            </CardHeader>
                                            <CardContent>
                                                <div className="text-3xl font-bold text-green-600">{formatCurrency(reportData.summary?.total_collected || 0)}</div>
                                                <p className="text-xs text-muted-foreground mt-1 flex items-center gap-1">
                                                    <ArrowUpRight className="h-3 w-3 text-green-500" />
                                                    Across {reportData.summary?.total_transactions || 0} transactions
                                                </p>
                                            </CardContent>
                                        </Card>

                                        {reportData.summary?.by_method && (
                                            <Card className="shadow-md flex-1">
                                                <CardHeader className="pb-0">
                                                    <CardTitle className="text-base">Payment Methods</CardTitle>
                                                </CardHeader>
                                                <CardContent className="h-[200px] relative">
                                                    <ResponsiveContainer width="100%" height="100%">
                                                        <PieChart>
                                                            <Pie
                                                                data={reportData.summary.by_method}
                                                                cx="50%"
                                                                cy="50%"
                                                                innerRadius={60}
                                                                outerRadius={80}
                                                                paddingAngle={5}
                                                                dataKey="amount"
                                                            >
                                                                {reportData.summary.by_method.map((entry: any, index: number) => (
                                                                    <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                                                                ))}
                                                            </Pie>
                                                            <Tooltip formatter={(value: number) => formatCurrency(value)} />
                                                        </PieChart>
                                                    </ResponsiveContainer>
                                                    <div className="absolute inset-0 flex items-center justify-center pointer-events-none">
                                                        <div className="text-center">
                                                            <span className="text-xs text-muted-foreground block">Methods</span>
                                                            <span className="text-xl font-bold">{reportData.summary.by_method.length}</span>
                                                        </div>
                                                    </div>
                                                </CardContent>
                                            </Card>
                                        )}
                                    </div>
                                </div>
                            </>
                        ) : null}
                    </TabsContent>

                    {/* Bills Report (formerly Arrears) */}
                    <TabsContent value="arrears" className="space-y-6 animate-in fade-in-50 duration-300">
                        {loading ? (
                             <div className="h-64 flex items-center justify-center text-muted-foreground">Loading bills...</div>
                        ) : reportData ? (
                            <>
                                <div className="flex justify-end mb-4">
                                     <div className="w-[200px]">
                                         <Select value={billStatus} onValueChange={setBillStatus}>
                                            <SelectTrigger>
                                                <SelectValue placeholder="Filter Status" />
                                            </SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="all">All Bills</SelectItem>
                                                <SelectItem value="due">Due / Unpaid</SelectItem>
                                                <SelectItem value="overdue">Overdue</SelectItem>
                                                <SelectItem value="paid">Paid</SelectItem>
                                            </SelectContent>
                                         </Select>
                                     </div>
                                </div>

                                <div className="grid gap-4 md:grid-cols-2">
                                    <Card className="bg-red-50 dark:bg-red-900/10 border-red-100 dark:border-red-900/20">
                                        <CardHeader className="pb-2">
                                            <CardTitle className="text-sm font-medium text-red-600 dark:text-red-400">Total Bill Amount</CardTitle>
                                        </CardHeader>
                                        <CardContent>
                                            <div className="text-3xl font-bold text-red-700 dark:text-red-300">{formatCurrency(reportData.total_amount || 0)}</div>
                                            <p className="text-sm text-red-600/80 mt-1">
                                                {billStatus === 'all' ? 'Total amount of bills in period' : 'Total filtered amount'}
                                            </p>
                                        </CardContent>
                                    </Card>
                                    <Card>
                                        <CardHeader className="pb-2">
                                            <CardTitle className="text-sm font-medium text-muted-foreground">Bill Count</CardTitle>
                                        </CardHeader>
                                        <CardContent>
                                            <div className="text-3xl font-bold text-foreground">{reportData.total_bills || 0}</div>
                                            <p className="text-sm text-muted-foreground mt-1">Number of bills found</p>
                                        </CardContent>
                                    </Card>
                                </div>

                                <Card className="shadow-md border-none ring-1 ring-border/50">
                                    <CardHeader>
                                        <CardTitle>Bills List</CardTitle>
                                        <CardDescription>
                                            {billStatus === 'all' 
                                                ? 'All bills generated in this period' 
                                                : `Bills with status: ${billStatus}`}
                                        </CardDescription>
                                    </CardHeader>
                                    <CardContent>
                                        <div className="rounded-lg border overflow-hidden">
                                            <table className="w-full text-sm">
                                                <thead className="bg-muted/50">
                                                    <tr>
                                                        <th className="px-4 py-3 text-left font-medium text-muted-foreground">Bill No</th>
                                                        <th className="px-4 py-3 text-left font-medium text-muted-foreground">Date</th>
                                                        <th className="px-4 py-3 text-left font-medium text-muted-foreground">Unit</th>
                                                        <th className="px-4 py-3 text-left font-medium text-muted-foreground">Resident</th>
                                                        <th className="px-4 py-3 text-right font-medium text-muted-foreground">Amount</th>
                                                        <th className="px-4 py-3 text-left font-medium text-muted-foreground pl-8">Status</th>
                                                    </tr>
                                                </thead>
                                                <tbody className="divide-y">
                                                    {reportData.data?.map((bill: any) => (
                                                        <tr key={bill.id} className="hover:bg-muted/50 transition-colors">
                                                            <td className="px-4 py-3 font-medium">{bill.bill_no}</td>
                                                            <td className="px-4 py-3">{bill.created_at}</td>
                                                            <td className="px-4 py-3">{bill.unit_no}</td>
                                                            <td className="px-4 py-3">
                                                                <div>
                                                                    <div className="font-medium text-foreground">{bill.resident_name}</div>
                                                                    {bill.days_overdue > 0 && <div className="text-xs text-red-500 font-medium">{bill.days_overdue} days overdue</div>}
                                                                </div>
                                                            </td>
                                                            <td className="px-4 py-3 text-right font-bold text-foreground/80">
                                                                {formatCurrency(bill.amount)}
                                                            </td>
                                                            <td className="px-4 py-3 pl-8">
                                                                <span className={`inline-flex rounded-full px-2.5 py-0.5 text-xs font-semibold ${
                                                                    bill.status === 'overdue' 
                                                                        ? 'bg-red-100 text-red-800' 
                                                                        : bill.status === 'paid'
                                                                        ? 'bg-green-100 text-green-800'
                                                                        : 'bg-yellow-100 text-yellow-800'
                                                                }`}>
                                                                    {bill.status.toUpperCase()}
                                                                </span>
                                                            </td>
                                                        </tr>
                                                    ))}
                                                    {(!reportData.data || reportData.data.length === 0) && (
                                                        <tr>
                                                            <td colSpan={6} className="px-4 py-8 text-center text-muted-foreground">
                                                                No bills found for the selected criteria.
                                                            </td>
                                                        </tr>
                                                    )}
                                                </tbody>
                                            </table>
                                        </div>
                                    </CardContent>
                                </Card>
                            </>
                        ) : null}
                    </TabsContent>

                    {/* Expense Breakdown */}
                    <TabsContent value="expense" className="space-y-6 animate-in fade-in-50 duration-300">
                        {loading ? (
                            <div className="h-64 flex items-center justify-center text-muted-foreground">Loading expenses...</div>
                        ) : reportData ? (
                            <>
                                <div className="grid gap-6 md:grid-cols-2">
                                    <div className="space-y-6">
                                        <Card className="border-t-4 border-t-orange-500 shadow-md">
                                            <CardHeader className="pb-2">
                                                <CardTitle className="text-sm font-medium text-muted-foreground">Total Expenses</CardTitle>
                                            </CardHeader>
                                            <CardContent>
                                                <div className="text-3xl font-bold text-orange-600">{formatCurrency(reportData.summary?.total_expenses || 0)}</div>
                                                <p className="text-xs text-muted-foreground mt-1 flex items-center gap-1">
                                                    <ArrowDownRight className="h-3 w-3 text-orange-500" />
                                                    Outflow this period
                                                </p>
                                            </CardContent>
                                        </Card>

                                        {reportData.summary?.by_category && (
                                            <Card className="shadow-md">
                                                <CardHeader>
                                                    <CardTitle className="text-base flex items-center gap-2">
                                                        <PieChartIcon className="h-4 w-4 text-orange-500" />
                                                        Expenses by Category
                                                    </CardTitle>
                                                </CardHeader>
                                                <CardContent>
                                                    <div className="h-[250px]">
                                                        <ResponsiveContainer width="100%" height="100%">
                                                            <PieChart>
                                                                <Pie
                                                                    data={reportData.summary.by_category}
                                                                    cx="50%"
                                                                    cy="50%"
                                                                    innerRadius={60}
                                                                    outerRadius={90}
                                                                    paddingAngle={5}
                                                                    dataKey="amount"
                                                                >
                                                                    {reportData.summary.by_category.map((entry: any, index: number) => (
                                                                        <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                                                                    ))}
                                                                </Pie>
                                                                <Tooltip formatter={(value: number) => formatCurrency(value)} />
                                                                <Legend verticalAlign="bottom" height={36}/>
                                                            </PieChart>
                                                        </ResponsiveContainer>
                                                    </div>
                                                </CardContent>
                                            </Card>
                                        )}
                                    </div>

                                    {reportData.summary?.by_vendor && (
                                        <Card className="shadow-md">
                                            <CardHeader>
                                                <CardTitle className="text-base flex items-center gap-2">
                                                    <BarChart3 className="h-4 w-4 text-orange-500" />
                                                    Top Vendors via Spend
                                                </CardTitle>
                                            </CardHeader>
                                            <CardContent>
                                                <div className="h-[400px]">
                                                    <ResponsiveContainer width="100%" height="100%">
                                                        <BarChart
                                                            layout="vertical"
                                                            data={reportData.summary.by_vendor}
                                                            margin={{ top: 5, right: 30, left: 40, bottom: 5 }}
                                                        >
                                                            <CartesianGrid strokeDasharray="3 3" horizontal={true} vertical={false} />
                                                            <XAxis type="number" hide />
                                                            <YAxis type="category" dataKey="vendor" width={100} tick={{fontSize: 12}} />
                                                            <Tooltip cursor={{fill: 'transparent'}} formatter={(value: number) => formatCurrency(value)} />
                                                            <Bar dataKey="amount" fill="#f97316" radius={[0, 4, 4, 0]} barSize={20} />
                                                        </BarChart>
                                                    </ResponsiveContainer>
                                                </div>
                                            </CardContent>
                                        </Card>
                                    )}
                                </div>
                            </>
                        ) : null}
                    </TabsContent>

                    {/* Financial Summary */}
                    <TabsContent value="financial" className="space-y-6 animate-in fade-in-50 duration-300">
                         {loading ? (
                            <div className="h-64 flex items-center justify-center text-muted-foreground">Loading financial summary...</div>
                        ) : reportData ? (
                            <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3">
                                <Card className="bg-green-50/50 dark:bg-green-900/10 border-green-100 shadow-sm">
                                    <CardHeader className="flex flex-row items-center justify-between pb-2">
                                        <CardTitle className="text-sm font-medium text-green-700 dark:text-green-300">Total Collections</CardTitle>
                                        <DollarSign className="h-4 w-4 text-green-600" />
                                    </CardHeader>
                                    <CardContent>
                                        <div className="text-2xl font-bold text-green-700">{formatCurrency(reportData.summary?.collections || 0)}</div>
                                    </CardContent>
                                </Card>

                                <Card className="bg-red-50/50 dark:bg-red-900/10 border-red-100 shadow-sm">
                                    <CardHeader className="flex flex-row items-center justify-between pb-2">
                                        <CardTitle className="text-sm font-medium text-red-700 dark:text-red-300">Total Expenses</CardTitle>
                                        <Wallet className="h-4 w-4 text-red-600" />
                                    </CardHeader>
                                    <CardContent>
                                        <div className="text-2xl font-bold text-red-700">{formatCurrency(reportData.summary?.expenses || 0)}</div>
                                    </CardContent>
                                </Card>

                                <Card className="bg-indigo-50/50 dark:bg-indigo-900/10 border-indigo-100 shadow-sm">
                                    <CardHeader className="flex flex-row items-center justify-between pb-2">
                                        <CardTitle className="text-sm font-medium text-indigo-700 dark:text-indigo-300">Net Income</CardTitle>
                                        <TrendingUp className="h-4 w-4 text-indigo-600" />
                                    </CardHeader>
                                    <CardContent>
                                        <div className={`text-2xl font-bold ${
                                            (reportData.summary?.net_income || 0) >= 0 ? 'text-indigo-700' : 'text-red-600'
                                        }`}>
                                            {formatCurrency(reportData.summary?.net_income || 0)}
                                        </div>
                                    </CardContent>
                                </Card>

                                <Card className="col-span-full border-t-4 border-t-purple-500 shadow-md">
                                    <CardHeader>
                                        <CardTitle>Collection Health (Period)</CardTitle>
                                        <CardDescription>Percentage of bills generated in this period that are fully collected</CardDescription>
                                    </CardHeader>
                                    <CardContent>
                                        <div className="mt-4">
                                            <div className="flex items-center justify-between mb-2">
                                                <span className="text-sm font-medium text-muted-foreground">Collection Rate</span>
                                                <span className="text-xl font-bold text-cyan-600">{reportData.summary?.collection_rate || 0}%</span>
                                            </div>
                                            <div className="h-4 w-full bg-secondary rounded-full overflow-hidden">
                                                <div 
                                                    className="h-full bg-gradient-to-r from-cyan-500 to-pink-500 transition-all duration-1000 ease-out"
                                                    style={{ width: `${reportData.summary?.collection_rate || 0}%` }}
                                                />
                                            </div>
                                            <div className="mt-6 grid grid-cols-2 gap-4">
                                                <div className="p-4 rounded-xl bg-muted/30 border">
                                                    <p className="text-sm text-muted-foreground">Current Outstanding Dues</p>
                                                    <p className="text-xl font-bold text-red-600 mt-1">{formatCurrency(reportData.summary?.outstanding_dues || 0)}</p>
                                                </div>
                                                <div className="p-4 rounded-xl bg-muted/30 border">
                                                    <p className="text-sm text-muted-foreground">Net Cash Flow</p>
                                                    <p className="text-xl font-bold text-green-600 mt-1">{formatCurrency(reportData.summary?.net_income || 0)}</p>
                                                </div>
                                            </div>
                                        </div>
                                    </CardContent>
                                </Card>
                            </div>
                        ) : null}
                    </TabsContent>
                </Tabs>
            </div>
        </SocietyAdminLayout>
    );
}
