import { Button } from '@/components/ui/button';
import { Card, CardContent } from '@/components/ui/card';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Checkbox } from '@/components/ui/checkbox';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link, router, useForm } from '@inertiajs/react';
import { DollarSign, Plus, Search, Eye, CreditCard, Banknote, TrendingUp, AlertCircle, FileText, RefreshCw, CheckCircle2, XCircle, Clock } from 'lucide-react';
import { useState, useEffect, useRef } from 'react';
import axios from 'axios';
import Swal from 'sweetalert2';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Payments',
        href: '/societies/{society}/payments',
    },
];

interface Unit {
    id: number;
    unit_no: string;
}

interface Resident {
    id: number;
    name: string;
    phone: string;
    unit?: Unit;
}

interface BankAccount {
    id: number;
    account_name: string;
    bank_name: string;
    account_number: string;
    is_primary?: boolean;
}

interface DueBill {
    id: number;
    bill_no: string;
    amount: number;
    status: string;
    due_date: string;
    maintenance: number;
    water: number;
    electricity: number;
    parking: number;
    other_charges: number;
    gst: number;
    discount: number;
}

interface Payment {
    id: number;
    payment_no: string;
    amount: number;
    payment_method: string;
    status: string;
    transaction_id: string | null;
    payment_request_id: string | null;
    created_at: string;
    user: {
        id: number;
        name: string;
        phone: string;
        unit?: Unit;
    };
    bill: {
        id: number;
        bill_no: string;
        amount: number;
    } | null;
    bank_account: {
        id: number;
        account_name: string;
        bank_name: string;
    } | null;
}

interface IndexProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    payments: {
        data: Payment[];
        links: any;
        meta: any;
    };
    residents: Resident[];
    bankAccounts: BankAccount[];
    stats: {
        total_collected: number;
        pending_amount: number;
        total_payments: number;
        pending_reconciliation: number;
    };
    filters: {
        search?: string;
        status?: string;
        payment_method?: string;
        user_id?: string;
    };
}

export default function Index({ society, payments, residents, bankAccounts, stats, filters }: IndexProps) {
    const [search, setSearch] = useState(filters.search || '');
    const [status, setStatus] = useState(filters.status || 'all');
    const [paymentMethod, setPaymentMethod] = useState(filters.payment_method || 'all');
    const [userId, setUserId] = useState(filters.user_id || 'all');
    const [isCreateOpen, setIsCreateOpen] = useState(false);
    const [dueBills, setDueBills] = useState<DueBill[]>([]);
    const [selectedBills, setSelectedBills] = useState<number[]>([]);
    const [loadingBills, setLoadingBills] = useState(false);
    const [showQRCode, setShowQRCode] = useState(false);
    const [qrCodeUrl, setQrCodeUrl] = useState('');
    const [razorpayOrderId, setRazorpayOrderId] = useState('');
    const [onlinePaymentMode, setOnlinePaymentMode] = useState<'razorpay' | 'manual'>('manual');
    const [selectedPayment, setSelectedPayment] = useState<any>(null);
    const [isDetailsOpen, setIsDetailsOpen] = useState(false);
    
    // Payment Status Modal States
    const [isStatusModalOpen, setIsStatusModalOpen] = useState(false);
    const [pendingPaymentLink, setPendingPaymentLink] = useState<string | null>(null);
    const [pendingPaymentId, setPendingPaymentId] = useState<string | null>(null);
    const [paymentStatusInfo, setPaymentStatusInfo] = useState<{status: string; message: string} | null>(null);
    const [checkingStatus, setCheckingStatus] = useState(false);
    const statusPollingRef = useRef<NodeJS.Timeout | null>(null);

    const createForm = useForm({
        user_id: '',
        bill_ids: [] as number[],
        amount: '',
        payment_method: 'cash',
        transaction_id: '',
        bank_account_id: '',
        payment_date: new Date().toISOString().split('T')[0],
        notes: '',
        status: 'completed',
    });

    const handleSearch = () => {
        const filterParams: any = {};
        if (search) filterParams.search = search;
        if (status && status !== 'all') filterParams.status = status;
        if (paymentMethod && paymentMethod !== 'all') filterParams.payment_method = paymentMethod;
        if (userId && userId !== 'all') filterParams.user_id = userId;

        router.get(`/societies/${society.id}/payments`, filterParams, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    // Fetch due bills when resident is selected
    useEffect(() => {
        if (createForm.data.user_id && createForm.data.user_id !== '') {
            setLoadingBills(true);
            axios.get(`/societies/${society.id}/payments/due-bills/${createForm.data.user_id}`)
                .then(response => {
                    setDueBills(response.data.dueBills || []);
                    setSelectedBills([]);
                    createForm.setData('bill_ids', []);
                    createForm.setData('amount', '');
                    setLoadingBills(false);
                })
                .catch(error => {
                    console.error('Error fetching due bills:', error);
                    setDueBills([]);
                    setLoadingBills(false);
                });
        } else {
            setDueBills([]);
            setSelectedBills([]);
            createForm.setData('bill_ids', []);
            createForm.setData('amount', '');
        }
    }, [createForm.data.user_id]);

    // Auto-set payment status based on payment method
    useEffect(() => {
        let newStatus = 'completed'; // Default
        
        if (createForm.data.payment_method === 'cash') {
            newStatus = 'completed'; // Cash is immediate
        } else if (createForm.data.payment_method === 'cheque') {
            newStatus = 'pending'; // Cheque needs to clear
        } else if (createForm.data.payment_method === 'online') {
            if (onlinePaymentMode === 'manual') {
                newStatus = 'completed'; // Already received manually
            } else {
                newStatus = 'pending'; // Awaiting Razorpay confirmation
            }
        } else if (createForm.data.payment_method === 'bank_transfer') {
            newStatus = 'pending_reconciliation'; // Needs bank reconciliation
        }
        
        if (createForm.data.status !== newStatus) {
            createForm.setData('status', newStatus);
        }
    }, [createForm.data.payment_method, onlinePaymentMode]);

    // Auto-select primary bank account when dialog opens
    useEffect(() => {
        if (isCreateOpen && !createForm.data.bank_account_id) {
            const primaryAccount = bankAccounts.find(acc => acc.is_primary);
            const defaultAccount = primaryAccount || bankAccounts[0];
            if (defaultAccount) {
                createForm.setData('bank_account_id', defaultAccount.id.toString());
            }
        }
    }, [isCreateOpen, bankAccounts]);

    // Update amount when bills are selected
    useEffect(() => {
        const totalAmount = dueBills
            .filter(bill => selectedBills.includes(bill.id))
            .reduce((sum, bill) => sum + bill.amount, 0);
        createForm.setData('amount', totalAmount.toString());
        createForm.setData('bill_ids', selectedBills);
    }, [selectedBills]);

    const handleBillSelect = (billId: number, checked: boolean) => {
        if (checked) {
            setSelectedBills([...selectedBills, billId]);
        } else {
            setSelectedBills(selectedBills.filter(id => id !== billId));
        }
    };

    const [sendingPaymentRequest, setSendingPaymentRequest] = useState(false);

    const handleCreateSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        
        // Handle "Send Payment Request" mode
        if (createForm.data.payment_method === 'online' && onlinePaymentMode === 'razorpay') {
            if (!createForm.data.user_id || !createForm.data.amount || parseFloat(createForm.data.amount) <= 0) {
                Swal.fire({
                    icon: 'warning',
                    title: 'Missing Information',
                    text: 'Please select a resident and enter a valid amount',
                });
                return;
            }

            setSendingPaymentRequest(true);
            try {
                const response = await axios.post(`/societies/${society.id}/payments/send-request`, {
                    user_id: createForm.data.user_id,
                    amount: parseFloat(createForm.data.amount),
                    bill_id: selectedBills.length > 0 ? selectedBills[0] : null,
                    description: createForm.data.notes || 'Payment Request',
                });

                if (response.data.success) {
                    // Close the create dialog
                    setIsCreateOpen(false);
                    createForm.reset();
                    setDueBills([]);
                    setSelectedBills([]);
                    
                    // Store payment info and open status modal
                    setPendingPaymentLink(response.data.payment_link);
                    setPendingPaymentId(response.data.link_id);
                    setPaymentStatusInfo({ status: 'pending', message: 'Waiting for resident to complete payment...' });
                    setIsStatusModalOpen(true);
                    
                    // Show success toast
                    Swal.fire({
                        icon: 'success',
                        title: 'Payment Request Sent!',
                        html: `<p>Payment link sent to resident via SMS</p><p class="text-sm text-gray-500 mt-2">You can track the status below</p>`,
                        timer: 3000,
                        timerProgressBar: true,
                        showConfirmButton: false,
                    });
                }
            } catch (error: any) {
                console.error('Error sending payment request:', error);
                const errorMessage = error.response?.data?.error || 'Failed to send payment request. Please try again.';
                Swal.fire({
                    icon: 'error',
                    title: 'Request Failed',
                    text: errorMessage,
                });
            } finally {
                setSendingPaymentRequest(false);
            }
            return;
        }
        
        // Convert 'none' to null for bank_account_id before submission
        if (createForm.data.bank_account_id === 'none') {
            createForm.setData('bank_account_id', '');
        }
        
        createForm.post(`/societies/${society.id}/payments`, {
            onSuccess: () => {
                setIsCreateOpen(false);
                createForm.reset();
                setDueBills([]);
                setSelectedBills([]);
            },
        });
    };

    // Check Payment Status from Razorpay
    const checkPaymentStatus = async () => {
        if (!pendingPaymentId) return;
        
        setCheckingStatus(true);
        try {
            const response = await axios.get(`/societies/${society.id}/payments/check-status/${pendingPaymentId}`);
            
            if (response.data.status === 'paid' || response.data.status === 'completed') {
                setPaymentStatusInfo({ status: 'completed', message: 'Payment completed successfully!' });
                Swal.fire({
                    icon: 'success',
                    title: 'Payment Completed!',
                    text: 'The resident has completed the payment.',
                    timer: 3000,
                    timerProgressBar: true,
                });
                // Refresh the page to show updated payment list
                setTimeout(() => {
                    router.reload();
                }, 1500);
            } else if (response.data.status === 'expired' || response.data.status === 'cancelled') {
                setPaymentStatusInfo({ status: 'failed', message: 'Payment link expired or cancelled' });
            } else {
                setPaymentStatusInfo({ status: 'pending', message: 'Waiting for resident to complete payment...' });
            }
        } catch (error: any) {
            console.error('Error checking payment status:', error);
            setPaymentStatusInfo({ status: 'pending', message: 'Unable to check status. Try again.' });
        } finally {
            setCheckingStatus(false);
        }
    };

    // Cleanup status polling when modal closes
    useEffect(() => {
        return () => {
            if (statusPollingRef.current) {
                clearInterval(statusPollingRef.current);
            }
        };
    }, []);

    // Close status modal handler
    const handleCloseStatusModal = () => {
        setIsStatusModalOpen(false);
        setPendingPaymentLink(null);
        setPendingPaymentId(null);
        setPaymentStatusInfo(null);
        router.reload(); // Refresh the page when closing
    };

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 0,
        }).format(amount);
    };

    const formatDate = (dateString: string) => {
        return new Date(dateString).toLocaleDateString('en-IN', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
        });
    };

    const getDaysOverdue = (dueDate: string) => {
        const today = new Date();
        const due = new Date(dueDate);
        const diffTime = today.getTime() - due.getTime();
        const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
        return diffDays;
    };

    const getBillHighlightClass = (bill: DueBill) => {
        const daysOverdue = getDaysOverdue(bill.due_date);
        if (bill.status === 'overdue' || daysOverdue > 0) {
            if (daysOverdue > 30) {
                return 'border-red-500 bg-red-50 dark:bg-red-950';
            } else if (daysOverdue > 7) {
                return 'border-orange-500 bg-orange-50 dark:bg-orange-950';
            } else {
                return 'border-yellow-500 bg-yellow-50 dark:bg-yellow-950';
            }
        }
        return 'border-gray-200';
    };

    const getBillStatusBadge = (bill: DueBill) => {
        const daysOverdue = getDaysOverdue(bill.due_date);
        if (daysOverdue > 0) {
            return (
                <span className="text-xs font-medium text-red-600 dark:text-red-400">
                    {daysOverdue} days overdue
                </span>
            );
        }
        return (
            <span className="text-xs capitalize text-muted-foreground">
                {bill.status}
            </span>
        );
    };

    // Generate Razorpay QR Code
    const generateQRCode = async () => {
        if (!createForm.data.amount || parseFloat(createForm.data.amount) <= 0) {
            alert('Please select bills or enter an amount');
            return;
        }

        try {
            const response = await axios.post(`/societies/${society.id}/payments/generate-qr`, {
                amount: parseFloat(createForm.data.amount),
                user_id: createForm.data.user_id,
            });

            setQrCodeUrl(response.data.qr_code_url);
            setRazorpayOrderId(response.data.order_id);
            setShowQRCode(true);
        } catch (error: any) {
            console.error('Error generating QR code:', error);
            const errorMessage = error.response?.data?.error || 'Failed to generate QR code. Please try again.';
            alert(errorMessage);
        }
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Payments" />
            <div className="flex h-full flex-1 flex-col gap-6 overflow-x-auto rounded-xl p-6">
                <div className="flex items-center justify-between">
                    <div>
                        <h1 className="text-2xl font-bold">Payments</h1>
                        <p className="text-muted-foreground">
                            Collect and manage all payments in your society
                        </p>
                    </div>
                    <div className="flex gap-2">
                        <Dialog open={isCreateOpen} onOpenChange={setIsCreateOpen}>
                            <DialogTrigger asChild>
                                <Button>
                                    <Plus className="mr-2 h-4 w-4" />
                                    Quick Collect
                                </Button>
                            </DialogTrigger>
                        <DialogContent className="max-w-6xl max-h-[90vh] overflow-y-auto">
                            <DialogHeader>
                                <DialogTitle>Collect Payment</DialogTitle>
                                <DialogDescription>
                                    Select resident and their due bills to collect payment
                                </DialogDescription>
                            </DialogHeader>
                            <form onSubmit={handleCreateSubmit} className="grid gap-4 py-4">
                                {/* Resident Selection */}
                                <div className="space-y-2">
                                    <Label htmlFor="user_id">Resident *</Label>
                                    <Select
                                        value={createForm.data.user_id}
                                        onValueChange={(value) => createForm.setData('user_id', value)}
                                    >
                                        <SelectTrigger id="user_id">
                                            <SelectValue placeholder="Select Resident" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            {residents.map((resident) => (
                                                <SelectItem key={resident.id} value={resident.id.toString()}>
                                                    {resident.name} - {resident.unit?.unit_no || 'No Unit'} ({resident.phone})
                                                </SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                    {createForm.errors.user_id && <p className="text-sm text-red-500">{createForm.errors.user_id}</p>}
                                </div>

                                {/* Due Bills - Only show if resident selected */}
                                {createForm.data.user_id && (
                                    <div className="space-y-2">
                                        <Label>Due Bills</Label>
                                        {loadingBills ? (
                                            <p className="text-sm text-muted-foreground">Loading bills...</p>
                                        ) : dueBills.length > 0 ? (
                                            <Card>
                                                <CardContent className="p-4">
                                                    <div className="space-y-3">
                                                        {dueBills.map((bill) => (
                                                            <div key={bill.id} className={`flex items-start space-x-3 p-3 rounded-lg border ${getBillHighlightClass(bill)}`}>
                                                                <Checkbox
                                                                    id={`bill-${bill.id}`}
                                                                    checked={selectedBills.includes(bill.id)}
                                                                    onCheckedChange={(checked) => handleBillSelect(bill.id, checked as boolean)}
                                                                />
                                                                <div className="flex-1">
                                                                    <label htmlFor={`bill-${bill.id}`} className="flex justify-between items-start cursor-pointer">
                                                                        <div>
                                                                            <p className="font-medium">Bill #{bill.bill_no}</p>
                                                                            <p className="text-sm flex items-center gap-2">
                                                                                Due: {formatDate(bill.due_date)} • {getBillStatusBadge(bill)}
                                                                            </p>
                                                                            <div className="text-xs text-muted-foreground mt-1">
                                                                                {bill.maintenance > 0 && <span>Maintenance: {formatCurrency(bill.maintenance)} </span>}
                                                                                {bill.water > 0 && <span>Water: {formatCurrency(bill.water)} </span>}
                                                                                {bill.electricity > 0 && <span>Electricity: {formatCurrency(bill.electricity)} </span>}
                                                                                {bill.parking > 0 && <span>Parking: {formatCurrency(bill.parking)} </span>}
                                                                            </div>
                                                                        </div>
                                                                        <span className="font-bold text-lg">{formatCurrency(bill.amount)}</span>
                                                                    </label>
                                                                </div>
                                                            </div>
                                                        ))}
                                                        {selectedBills.length > 0 && (
                                                            <div className="pt-3 border-t">
                                                                <div className="flex justify-between items-center font-bold text-lg">
                                                                    <span>Total Selected:</span>
                                                                    <span className="text-primary">{formatCurrency(parseFloat(createForm.data.amount) || 0)}</span>
                                                                </div>
                                                            </div>
                                                        )}
                                                    </div>
                                                </CardContent>
                                            </Card>
                                        ) : (
                                            <Card>
                                                <CardContent className="p-4 text-center text-muted-foreground">
                                                    <AlertCircle className="h-8 w-8 mx-auto mb-2" />
                                                    <p>No due bills for this resident</p>
                                                </CardContent>
                                            </Card>
                                        )}
                                    </div>
                                )}

                                <div className="grid grid-cols-2 gap-4">
                                    {/* Payment Method */}
                                    <div className="space-y-2">
                                        <Label htmlFor="payment_method">Payment Method *</Label>
                                        <Select
                                            value={createForm.data.payment_method}
                                            onValueChange={(value) => createForm.setData('payment_method', value)}
                                        >
                                            <SelectTrigger id="payment_method">
                                                <SelectValue />
                                            </SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="cash">Cash</SelectItem>
                                                <SelectItem value="cheque">Cheque</SelectItem>
                                                <SelectItem value="online">Online/UPI</SelectItem>
                                                <SelectItem value="bank_transfer">Bank Transfer</SelectItem>
                                            </SelectContent>
                                        </Select>
                                        {createForm.errors.payment_method && <p className="text-sm text-red-500">{createForm.errors.payment_method}</p>}
                                    </div>

                                    {/* Payment Date */}
                                    <div className="space-y-2">
                                        <Label htmlFor="payment_date">Payment Date</Label>
                                        <Input
                                            id="payment_date"
                                            type="date"
                                            value={createForm.data.payment_date}
                                            onChange={(e) => createForm.setData('payment_date', e.target.value)}
                                        />
                                    </div>
                                </div>

                                {/* Payment Mode Selection for Online/UPI */}
                                {createForm.data.payment_method === 'online' && (
                                    <div className="space-y-2">
                                        <Label>Online Payment Mode</Label>
                                        <div className="flex gap-4">
                                            <label className="flex items-center space-x-2 cursor-pointer">
                                                <input
                                                    type="radio"
                                                    value="manual"
                                                    checked={onlinePaymentMode === 'manual'}
                                                    onChange={() => setOnlinePaymentMode('manual')}
                                                    className="w-4 h-4"
                                                />
                                                <span>Manual Entry</span>
                                            </label>
                                            <label className="flex items-center space-x-2 cursor-pointer">
                                                <input
                                                    type="radio"
                                                    value="razorpay"
                                                    checked={onlinePaymentMode === 'razorpay'}
                                                    onChange={() => setOnlinePaymentMode('razorpay')}
                                                    className="w-4 h-4"
                                                />
                                                <span>Send Payment Request</span>
                                            </label>
                                        </div>
                                        <p className="text-xs text-muted-foreground">
                                            {onlinePaymentMode === 'manual' 
                                                ? 'Manually record payment received via UPI/Online transfer'
                                                : 'Payment request will be sent to resident\'s registered mobile number'}
                                        </p>
                                    </div>
                                )}

                                {/* Payment Request Info for Razorpay Mode */}
                                {createForm.data.payment_method === 'online' && onlinePaymentMode === 'razorpay' && (
                                    <Card>
                                        <CardContent className="p-4">
                                            <div className="space-y-2">
                                                <p className="font-medium text-sm">💡 Payment Request Details</p>
                                                <p className="text-sm text-muted-foreground">
                                                    A payment request for <strong>{formatCurrency(parseFloat(createForm.data.amount) || 0)}</strong> will be sent to the resident's mobile number via SMS/WhatsApp.
                                                </p>
                                                <p className="text-xs text-muted-foreground">
                                                    Payment status will be automatically updated once resident completes the payment.
                                                </p>
                                            </div>
                                        </CardContent>
                                    </Card>
                                )}

                                {/* Transaction ID - Show for manual online payments and bank transfers */}
                                {((createForm.data.payment_method === 'online' && onlinePaymentMode === 'manual') || createForm.data.payment_method === 'bank_transfer') && (
                                    <div className="space-y-2">
                                        <Label htmlFor="transaction_id">Transaction ID / Reference Number</Label>
                                        <Input
                                            id="transaction_id"
                                            placeholder="Enter transaction ID or reference number"
                                            value={createForm.data.transaction_id}
                                            onChange={(e) => createForm.setData('transaction_id', e.target.value)}
                                        />
                                    </div>
                                )}

                                {/* Bank Account */}
                                <div className="space-y-2">
                                    <Label htmlFor="bank_account_id">Deposit to Bank Account</Label>
                                    <Select
                                        value={createForm.data.bank_account_id}
                                        onValueChange={(value) => createForm.setData('bank_account_id', value)}
                                    >
                                        <SelectTrigger id="bank_account_id">
                                            <SelectValue placeholder="Select Bank Account (Optional)" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="none">None</SelectItem>
                                            {bankAccounts.map((account) => (
                                                <SelectItem key={account.id} value={account.id.toString()}>
                                                    {account.account_name} - {account.bank_name} ({account.account_number})
                                                </SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                </div>

                                {/* Amount - Auto-calculated from selected bills */}
                                <div className="space-y-2">
                                    <Label htmlFor="amount">Amount *</Label>
                                    <Input
                                        id="amount"
                                        type="number"
                                        step="0.01"
                                        placeholder="0.00"
                                        value={createForm.data.amount}
                                        onChange={(e) => createForm.setData('amount', e.target.value)}
                                        disabled={selectedBills.length > 0}
                                    />
                                    {selectedBills.length > 0 && (
                                        <p className="text-xs text-muted-foreground">Amount is auto-calculated from selected bills</p>
                                    )}
                                    {createForm.errors.amount && <p className="text-sm text-red-500">{createForm.errors.amount}</p>}
                                </div>

                                {/* Notes */}
                                <div className="space-y-2">
                                    <Label htmlFor="notes">Notes</Label>
                                    <Input
                                        id="notes"
                                        placeholder="Additional notes (optional)"
                                        value={createForm.data.notes}
                                        onChange={(e) => createForm.setData('notes', e.target.value)}
                                    />
                                </div>

                                <DialogFooter>
                                    <Button type="button" variant="outline" onClick={() => setIsCreateOpen(false)}>
                                        Cancel
                                    </Button>
                                    <Button 
                                        type="submit" 
                                        disabled={createForm.processing || sendingPaymentRequest}
                                    >
                                        {sendingPaymentRequest 
                                            ? 'Sending Request...' 
                                            : createForm.processing 
                                                ? 'Processing...' 
                                                : (createForm.data.payment_method === 'online' && onlinePaymentMode === 'razorpay')
                                                    ? '📱 Send Payment Request'
                                                    : 'Record Payment'
                                        }
                                    </Button>
                                </DialogFooter>
                            </form>
                        </DialogContent>
                    </Dialog>

                    {/* Payment Status Modal */}
                    <Dialog open={isStatusModalOpen} onOpenChange={(open) => !open && handleCloseStatusModal()}>
                        <DialogContent className="max-w-md">
                            <DialogHeader>
                                <DialogTitle className="flex items-center gap-2">
                                    {paymentStatusInfo?.status === 'completed' ? (
                                        <CheckCircle2 className="h-5 w-5 text-green-500" />
                                    ) : paymentStatusInfo?.status === 'failed' ? (
                                        <XCircle className="h-5 w-5 text-red-500" />
                                    ) : (
                                        <Clock className="h-5 w-5 text-yellow-500" />
                                    )}
                                    Payment Status
                                </DialogTitle>
                                <DialogDescription>
                                    Track the status of the payment request
                                </DialogDescription>
                            </DialogHeader>
                            
                            <div className="space-y-4 py-4">
                                {/* Status Display */}
                                <div className={`p-4 rounded-lg border ${
                                    paymentStatusInfo?.status === 'completed' 
                                        ? 'bg-green-50 border-green-200 dark:bg-green-950 dark:border-green-800' 
                                        : paymentStatusInfo?.status === 'failed'
                                            ? 'bg-red-50 border-red-200 dark:bg-red-950 dark:border-red-800'
                                            : 'bg-yellow-50 border-yellow-200 dark:bg-yellow-950 dark:border-yellow-800'
                                }`}>
                                    <div className="flex items-center gap-3">
                                        {paymentStatusInfo?.status === 'completed' ? (
                                            <CheckCircle2 className="h-8 w-8 text-green-500" />
                                        ) : paymentStatusInfo?.status === 'failed' ? (
                                            <XCircle className="h-8 w-8 text-red-500" />
                                        ) : (
                                            <Clock className="h-8 w-8 text-yellow-500 animate-pulse" />
                                        )}
                                        <div>
                                            <p className="font-semibold capitalize">{paymentStatusInfo?.status || 'Pending'}</p>
                                            <p className="text-sm text-muted-foreground">{paymentStatusInfo?.message}</p>
                                        </div>
                                    </div>
                                </div>

                                {/* Payment Link Display */}
                                {pendingPaymentLink && (
                                    <div className="space-y-2">
                                        <Label>Payment Link</Label>
                                        <div className="flex gap-2">
                                            <Input 
                                                readOnly 
                                                value={pendingPaymentLink} 
                                                className="text-sm"
                                            />
                                            <Button 
                                                variant="outline" 
                                                size="sm"
                                                onClick={() => {
                                                    navigator.clipboard.writeText(pendingPaymentLink);
                                                    Swal.fire({
                                                        icon: 'success',
                                                        title: 'Copied!',
                                                        text: 'Payment link copied to clipboard',
                                                        timer: 1500,
                                                        showConfirmButton: false,
                                                    });
                                                }}
                                            >
                                                Copy
                                            </Button>
                                        </div>
                                    </div>
                                )}
                            </div>

                            <DialogFooter className="flex gap-2">
                                <Button 
                                    variant="outline" 
                                    onClick={checkPaymentStatus}
                                    disabled={checkingStatus || paymentStatusInfo?.status === 'completed'}
                                >
                                    {checkingStatus ? (
                                        <>
                                            <RefreshCw className="mr-2 h-4 w-4 animate-spin" />
                                            Checking...
                                        </>
                                    ) : (
                                        <>
                                            <RefreshCw className="mr-2 h-4 w-4" />
                                            Check Status
                                        </>
                                    )}
                                </Button>
                                <Button onClick={handleCloseStatusModal}>
                                    {paymentStatusInfo?.status === 'completed' ? 'Done' : 'Close'}
                                </Button>
                            </DialogFooter>
                        </DialogContent>
                    </Dialog>
                </div>
            </div>

                {/* KPI Cards */}
                <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
                    <Card>
                        <CardContent className="p-6">
                            <div className="flex items-center justify-between">
                                <div>
                                    <p className="text-sm font-medium text-muted-foreground">Total Collected</p>
                                    <p className="text-2xl font-bold">{formatCurrency(stats.total_collected)}</p>
                                </div>
                                <div className="h-12 w-12 rounded-full bg-green-100 dark:bg-green-900 flex items-center justify-center">
                                    <TrendingUp className="h-6 w-6 text-green-600 dark:text-green-400" />
                                </div>
                            </div>
                        </CardContent>
                    </Card>

                    <Card>
                        <CardContent className="p-6">
                            <div className="flex items-center justify-between">
                                <div>
                                    <p className="text-sm font-medium text-muted-foreground">Pending Amount</p>
                                    <p className="text-2xl font-bold">{formatCurrency(stats.pending_amount)}</p>
                                </div>
                                <div className="h-12 w-12 rounded-full bg-yellow-100 dark:bg-yellow-900 flex items-center justify-center">
                                    <DollarSign className="h-6 w-6 text-yellow-600 dark:text-yellow-400" />
                                </div>
                            </div>
                        </CardContent>
                    </Card>

                    <Card>
                        <CardContent className="p-6">
                            <div className="flex items-center justify-between">
                                <div>
                                    <p className="text-sm font-medium text-muted-foreground">Total Payments</p>
                                    <p className="text-2xl font-bold">{stats.total_payments}</p>
                                </div>
                                <div className="h-12 w-12 rounded-full bg-blue-100 dark:bg-blue-900 flex items-center justify-center">
                                    <CreditCard className="h-6 w-6 text-blue-600 dark:text-blue-400" />
                                </div>
                            </div>
                        </CardContent>
                    </Card>

                    <Card>
                        <CardContent className="p-6">
                            <div className="flex items-center justify-between">
                                <div>
                                    <p className="text-sm font-medium text-muted-foreground">Pending Reconciliation</p>
                                    <p className="text-2xl font-bold">{stats.pending_reconciliation}</p>
                                </div>
                                <div className="h-12 w-12 rounded-full bg-orange-100 dark:bg-orange-900 flex items-center justify-center">
                                    <Banknote className="h-6 w-6 text-orange-600 dark:text-orange-400" />
                                </div>
                            </div>
                        </CardContent>
                    </Card>
                </div>

                {/* Filters */}
                <Card>
                    <CardContent className="pt-6">
                        <div className="flex gap-4 flex-wrap">
                            <div className="flex-1 min-w-[200px]">
                                <div className="relative">
                                    <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                    <Input
                                        placeholder="Search by payment no, transaction ID, resident..."
                                        value={search}
                                        onChange={(e) => setSearch(e.target.value)}
                                        onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                        className="pl-9"
                                    />
                                </div>
                            </div>
                            <Select value={status} onValueChange={setStatus}>
                                <SelectTrigger className="w-48">
                                    <SelectValue placeholder="All Status" />
                                </SelectTrigger>
                                <SelectContent>
                                    <SelectItem value="all">All Status</SelectItem>
                                    <SelectItem value="pending">Pending</SelectItem>
                                    <SelectItem value="completed">Completed</SelectItem>
                                    <SelectItem value="failed">Failed</SelectItem>
                                    <SelectItem value="pending_reconciliation">Pending Reconciliation</SelectItem>
                                </SelectContent>
                            </Select>
                            <Select value={paymentMethod} onValueChange={setPaymentMethod}>
                                <SelectTrigger className="w-48">
                                    <SelectValue placeholder="All Methods" />
                                </SelectTrigger>
                                <SelectContent>
                                    <SelectItem value="all">All Methods</SelectItem>
                                    <SelectItem value="cash">Cash</SelectItem>
                                    <SelectItem value="cheque">Cheque</SelectItem>
                                    <SelectItem value="online">Online</SelectItem>
                                    <SelectItem value="bank_transfer">Bank Transfer</SelectItem>
                                </SelectContent>
                            </Select>
                            <Select value={userId} onValueChange={setUserId}>
                                <SelectTrigger className="w-48">
                                    <SelectValue placeholder="All Residents" />
                                </SelectTrigger>
                                <SelectContent>
                                    <SelectItem value="all">All Residents</SelectItem>
                                    {residents.map((resident) => (
                                        <SelectItem key={resident.id} value={resident.id.toString()}>
                                            {resident.name}
                                        </SelectItem>
                                    ))}
                                </SelectContent>
                            </Select>
                            <Button onClick={handleSearch}>Filter</Button>
                        </div>
                    </CardContent>
                </Card>

                {/* Payments Table */}
                <Card>
                    <CardContent className="p-0">
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead>
                                    <tr className="border-b bg-muted/50">
                                        <th className="px-4 py-3 text-left text-sm font-medium">Payment No</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Resident</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Bill</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Amount</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Method</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Transaction ID</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Status</th>
                                        <th className="px-4 py-3 text-right text-sm font-medium">Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {payments.data.length > 0 ? (
                                        payments.data.map((payment) => (
                                            <tr key={payment.id} className="border-b hover:bg-muted/50">
                                                <td className="px-4 py-3">
                                                    <div className="flex items-center gap-2">
                                                        <DollarSign className="h-4 w-4 text-muted-foreground" />
                                                        <span className="font-medium">{payment.payment_no}</span>
                                                    </div>
                                                </td>
                                                <td className="px-4 py-3">
                                                    <div>
                                                        <div className="font-medium">{payment.user.name}</div>
                                                        <div className="text-xs text-muted-foreground">
                                                            {payment.user.unit?.unit_no || '-'} • {payment.user.phone}
                                                        </div>
                                                    </div>
                                                </td>
                                                <td className="px-4 py-3 text-sm text-muted-foreground">
                                                    {payment.bill ? (
                                                        <div>
                                                            <div className="font-medium">{payment.bill.bill_no}</div>
                                                            <div className="text-xs">
                                                                {formatCurrency(payment.bill.amount)}
                                                            </div>
                                                        </div>
                                                    ) : (
                                                        '-'
                                                    )}
                                                </td>
                                                <td className="px-4 py-3 text-sm font-medium">
                                                    {formatCurrency(payment.amount)}
                                                </td>
                                                <td className="px-4 py-3 text-sm capitalize">
                                                    {payment.payment_method.replace('_', ' ')}
                                                </td>
                                                <td className="px-4 py-3 text-sm text-muted-foreground">
                                                    {payment.transaction_id || '-'}
                                                </td>
                                                <td className="px-4 py-3">
                                                    <Select
                                                        value={payment.status}
                                                        onValueChange={(value) => {
                                                            router.put(`/societies/${society.id}/payments/${payment.id}`, {
                                                                status: value
                                                            }, {
                                                                preserveState: true,
                                                                preserveScroll: true,
                                                            });
                                                        }}
                                                    >
                                                        <SelectTrigger className="w-[180px] h-8">
                                                            <SelectValue />
                                                        </SelectTrigger>
                                                        <SelectContent>
                                                            <SelectItem value="pending">
                                                                <span className="flex items-center gap-2">
                                                                    <span className="h-2 w-2 rounded-full bg-yellow-500"></span>
                                                                    Pending
                                                                </span>
                                                            </SelectItem>
                                                            <SelectItem value="completed">
                                                                <span className="flex items-center gap-2">
                                                                    <span className="h-2 w-2 rounded-full bg-green-500"></span>
                                                                    Completed
                                                                </span>
                                                            </SelectItem>
                                                            <SelectItem value="failed">
                                                                <span className="flex items-center gap-2">
                                                                    <span className="h-2 w-2 rounded-full bg-red-500"></span>
                                                                    Failed
                                                                </span>
                                                            </SelectItem>
                                                            <SelectItem value="pending_reconciliation">
                                                                <span className="flex items-center gap-2">
                                                                    <span className="h-2 w-2 rounded-full bg-blue-500"></span>
                                                                    Pending Reconciliation
                                                                </span>
                                                            </SelectItem>
                                                        </SelectContent>
                                                    </Select>
                                                </td>
                                                <td className="px-4 py-3">
                                                    <div className="flex items-center justify-end gap-2">
                                                        {/* Check Status button for pending payments */}
                                                        {payment.status === 'pending' && payment.payment_method === 'upi_request' && (
                                                            <Button
                                                                size="sm"
                                                                variant="outline"
                                                                className="text-yellow-600 border-yellow-300 hover:bg-yellow-50"
                                                                onClick={async () => {
                                                                    // Set the payment info and open modal
                                                                    setPendingPaymentId((payment as any).payment_request_id || null);
                                                                    setPendingPaymentLink(null);
                                                                    setPaymentStatusInfo({ status: 'pending', message: 'Checking status...' });
                                                                    setIsStatusModalOpen(true);
                                                                    
                                                                    // Check status
                                                                    try {
                                                                        const response = await axios.get(`/societies/${society.id}/payments/check-status/${(payment as any).payment_request_id}`);
                                                                        if (response.data.status === 'paid' || response.data.status === 'completed') {
                                                                            setPaymentStatusInfo({ status: 'completed', message: 'Payment completed!' });
                                                                            Swal.fire({
                                                                                icon: 'success',
                                                                                title: 'Payment Completed!',
                                                                                timer: 2000,
                                                                            });
                                                                            setTimeout(() => router.reload(), 1500);
                                                                        } else if (response.data.status === 'expired' || response.data.status === 'cancelled') {
                                                                            setPaymentStatusInfo({ status: 'failed', message: 'Payment link expired or cancelled' });
                                                                        } else {
                                                                            setPaymentStatusInfo({ status: 'pending', message: 'Waiting for payment...' });
                                                                            if (response.data.short_url) {
                                                                                setPendingPaymentLink(response.data.short_url);
                                                                            }
                                                                        }
                                                                    } catch (error) {
                                                                        setPaymentStatusInfo({ status: 'pending', message: 'Unable to check status' });
                                                                    }
                                                                }}
                                                            >
                                                                <RefreshCw className="h-4 w-4 mr-1" />
                                                                Check
                                                            </Button>
                                                        )}
                                                        <Button
                                                            size="sm"
                                                            variant="ghost"
                                                            onClick={() => {
                                                                setSelectedPayment(payment);
                                                                setIsDetailsOpen(true);
                                                            }}
                                                        >
                                                            <Eye className="h-4 w-4" />
                                                        </Button>
                                                    </div>
                                                </td>
                                            </tr>
                                        ))
                                    ) : (
                                        <tr>
                                            <td colSpan={8} className="px-4 py-8 text-center text-muted-foreground">
                                                No payments found
                                            </td>
                                        </tr>
                                    )}
                                </tbody>
                            </table>
                        </div>

                        {/* Pagination */}
                        {payments.links && payments.links.length > 3 && (
                            <div className="p-4 flex items-center justify-between border-t">
                                <div className="text-sm text-muted-foreground">
                                    Showing {payments.meta.from} to {payments.meta.to} of {payments.meta.total} results
                                </div>
                                <div className="flex gap-2">
                                    {payments.links.map((link: any, index: number) => (
                                        <Link
                                            key={index}
                                            href={link.url || '#'}
                                            className={`px-3 py-1 rounded ${
                                                link.active
                                                    ? 'bg-primary text-primary-foreground'
                                                    : 'bg-muted hover:bg-muted/80'
                                            }`}
                                            dangerouslySetInnerHTML={{ __html: link.label }}
                                        />
                                    ))}
                                </div>
                            </div>
                        )}
                    </CardContent>
                </Card>

                {/* Payment Details Modal */}
                <Dialog open={isDetailsOpen} onOpenChange={setIsDetailsOpen}>
                    <DialogContent className="max-w-2xl">
                        <DialogHeader>
                            <DialogTitle>Payment Details</DialogTitle>
                            <DialogDescription>
                                View complete payment information
                            </DialogDescription>
                        </DialogHeader>
                        {selectedPayment && (
                            <div className="space-y-4">
                                <div className="grid grid-cols-2 gap-4">
                                    <div>
                                        <p className="text-sm font-medium">Payment No</p>
                                        <p className="text-sm text-muted-foreground">{selectedPayment.payment_no}</p>
                                    </div>
                                    <div>
                                        <p className="text-sm font-medium">Amount</p>
                                        <p className="text-sm text-muted-foreground">{formatCurrency(selectedPayment.amount)}</p>
                                    </div>
                                    <div>
                                        <p className="text-sm font-medium">Resident</p>
                                        <p className="text-sm text-muted-foreground">
                                            {selectedPayment.user?.name} - {selectedPayment.user?.unit?.unit_number}
                                        </p>
                                    </div>
                                    <div>
                                        <p className="text-sm font-medium">Payment Method</p>
                                        <p className="text-sm text-muted-foreground capitalize">{selectedPayment.payment_method}</p>
                                    </div>
                                    <div>
                                        <p className="text-sm font-medium">Status</p>
                                        <p className="text-sm">
                                            <span className={`inline-flex items-center rounded-full px-2 py-1 text-xs font-medium ${
                                                selectedPayment.status === 'completed' ? 'bg-green-50 text-green-700 dark:bg-green-950 dark:text-green-400' :
                                                selectedPayment.status === 'pending' ? 'bg-yellow-50 text-yellow-700 dark:bg-yellow-950 dark:text-yellow-400' :
                                                selectedPayment.status === 'failed' ? 'bg-red-50 text-red-700 dark:bg-red-950 dark:text-red-400' :
                                                'bg-blue-50 text-blue-700 dark:bg-blue-950 dark:text-blue-400'
                                            }`}>
                                                {selectedPayment.status}
                                            </span>
                                        </p>
                                    </div>
                                    <div>
                                        <p className="text-sm font-medium">Payment Date</p>
                                        <p className="text-sm text-muted-foreground">{formatDate(selectedPayment.payment_date)}</p>
                                    </div>
                                    {selectedPayment.transaction_id && (
                                        <div>
                                            <p className="text-sm font-medium">Transaction ID</p>
                                            <p className="text-sm text-muted-foreground">{selectedPayment.transaction_id}</p>
                                        </div>
                                    )}
                                    {selectedPayment.bank_account && (
                                        <div>
                                            <p className="text-sm font-medium">Bank Account</p>
                                            <p className="text-sm text-muted-foreground">
                                                {selectedPayment.bank_account.account_name} - {selectedPayment.bank_account.bank_name}
                                            </p>
                                        </div>
                                    )}
                                </div>
                                {selectedPayment.bills && selectedPayment.bills.length > 0 && (
                                    <div>
                                        <p className="text-sm font-medium mb-2">Related Bills</p>
                                        <div className="space-y-2">
                                            {selectedPayment.bills.map((bill: any) => (
                                                <div key={bill.id} className="flex justify-between items-center p-2 bg-muted rounded">
                                                    <span className="text-sm">Bill #{bill.bill_no}</span>
                                                    <span className="text-sm font-medium">{formatCurrency(bill.amount)}</span>
                                                </div>
                                            ))}
                                        </div>
                                    </div>
                                )}
                                {selectedPayment.notes && (
                                    <div>
                                        <p className="text-sm font-medium">Notes</p>
                                        <p className="text-sm text-muted-foreground">{selectedPayment.notes}</p>
                                    </div>
                                )}
                            </div>
                        )}
                    </DialogContent>
                </Dialog>
            </div>
        </SocietyAdminLayout>
    );
}
