import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Switch } from '@/components/ui/switch';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { Head, useForm } from '@inertiajs/react';
import { Loader2, IndianRupee, Layers } from 'lucide-react';
import Swal from 'sweetalert2';

export default function Settings({ society, slots }: any) {
    const { data, setData, post, processing } = useForm({
        slots: slots || [],
    });

    const handleUpdate = (index: number, field: string, value: any) => {
        const newSlots = [...data.slots];
        newSlots[index] = { ...newSlots[index], [field]: value };
        setData('slots', newSlots);
    };

    const handleSubmit = (e: any) => {
        e.preventDefault();
        post(`/societies/${society.id}/monetization/settings`, {
            preserveScroll: true,
            onSuccess: () => {
                 const Toast = Swal.mixin({
                    toast: true,
                    position: 'top-end',
                    showConfirmButton: false,
                    timer: 3000,
                    timerProgressBar: true,
                });
                Toast.fire({ icon: 'success', title: 'Settings updated' });
            }
        });
    };

    return (
        <SocietyAdminLayout society={society} breadcrumbs={[{ title: 'Ad Settings', href: '#' }]}>
            <Head title="Ad Settings" />
            <form onSubmit={handleSubmit} className="p-6 space-y-6">
                <div className="flex items-center justify-between">
                     <div>
                        <h1 className="text-2xl font-bold">Slot Configuration</h1>
                        <p className="text-muted-foreground">Manage pricing and rotations for ad slots</p>
                     </div>
                     <Button type="submit" disabled={processing}>
                        {processing && <Loader2 className="mr-2 h-4 w-4 animate-spin" />} Save Changes
                     </Button>
                </div>
                
                <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {data.slots.map((slot: any, index: number) => (
                        <Card key={slot.id} className={!slot.status ? 'opacity-75' : ''}>
                            <CardHeader>
                                <div className="flex justify-between items-center space-x-2">
                                    <div className="font-semibold">{slot.name || slot.position}</div>
                                    <Switch 
                                        checked={Boolean(slot.status)} 
                                        onCheckedChange={(v) => handleUpdate(index, 'status', v)}
                                    />
                                </div>
                            </CardHeader>
                            <CardContent className="space-y-4">
                                <div className="grid gap-2">
                                    <Label>Price Per Day</Label>
                                    <div className="relative">
                                        <div className="absolute left-2 top-2.5 h-4 w-4 text-muted-foreground flex items-center justify-center">₹</div>
                                        <Input 
                                            type="number" 
                                            value={slot.price_per_day} 
                                            onChange={(e) => handleUpdate(index, 'price_per_day', e.target.value)}
                                            className="pl-8"
                                        />
                                    </div>
                                </div>
                                <div className="grid gap-2">
                                    <Label>Max Concurrent Ads</Label>
                                    <div className="relative">
                                         <Layers className="absolute left-2 top-2.5 h-4 w-4 text-muted-foreground" />
                                         <Input 
                                            type="number" 
                                            value={slot.max_concurrent_ads} 
                                            onChange={(e) => handleUpdate(index, 'max_concurrent_ads', e.target.value)} 
                                            className="pl-8"
                                        />
                                    </div>
                                </div>
                            </CardContent>
                        </Card>
                    ))}
                </div>
            </form>
        </SocietyAdminLayout>
    );
}
