import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { Head, Link, router } from '@inertiajs/react';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Check, X, User, Calendar, ExternalLink, Image as ImageIcon, Search } from 'lucide-react';
import Swal from 'sweetalert2';
import { format } from 'date-fns';
import { useState } from 'react';

export default function Requests({ society, ads, filters }: any) {
    const [search, setSearch] = useState(filters?.search || '');
    
    const handleSearch = () => {
        router.get(`/societies/${society.id}/monetization/requests`, { search }, { preserveState: true });
    };
    
    const handleApprove = (id: number) => {
        Swal.fire({
            title: 'Approve Ad?',
            text: "This ad will go live on the start date.",
            icon: 'question',
            showCancelButton: true,
            confirmButtonText: 'Yes, Approve',
            confirmButtonColor: '#10B981', 
        }).then((result) => {
            if (result.isConfirmed) {
                router.post(`/societies/${society.id}/monetization/${id}/approve`, {}, {
                    onSuccess: () => Swal.fire('Approved!', 'Ad is now approved.', 'success')
                });
            }
        });
    };

    const handleReject = (id: number) => {
        Swal.fire({
            title: 'Reject Announcement',
            input: 'textarea',
            inputLabel: 'Reason for rejection',
            inputPlaceholder: 'Type your reason here...',
            inputAttributes: {
                'aria-label': 'Type your reason here'
            },
            showCancelButton: true,
            confirmButtonText: 'Reject',
            confirmButtonColor: '#EF4444',
            inputValidator: (value) => {
                if (!value) {
                    return 'You need to write something!'
                }
            }
        }).then((result) => {
            if (result.isConfirmed) {
                router.post(`/societies/${society.id}/monetization/${id}/reject`, {
                    rejection_reason: result.value
                }, {
                    onSuccess: () => Swal.fire('Rejected', 'Ad has been rejected.', 'info')
                });
            }
        });
    };

    return (
        <SocietyAdminLayout society={society} breadcrumbs={[{ title: 'Ad Requests', href: '#' }]}>
            <Head title="Pending Ad Requests" />
            <div className="flex h-full flex-1 flex-col gap-6 overflow-x-auto rounded-xl p-6">
                <div className="flex justify-between items-center">
                    <div>
                        <h1 className="text-2xl font-bold">Pending Requests</h1>
                        <p className="text-muted-foreground">Review and approve advertisement requests</p>
                    </div>
                </div>

                <Card>
                    <CardContent className="pt-6">
                        <div className="flex gap-4">
                            <div className="flex-1">
                                <Label htmlFor="search">Search</Label>
                                <div className="relative mt-1">
                                    <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                    <Input
                                        id="search"
                                        placeholder="Search by advertiser..."
                                        value={search}
                                        onChange={(e) => setSearch(e.target.value)}
                                        onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                        className="pl-9"
                                    />
                                </div>
                            </div>
                            <div className="flex items-end">
                                <Button onClick={handleSearch}>Filter</Button>
                            </div>
                        </div>
                    </CardContent>
                </Card>

                <Card>
                    <CardHeader>
                        <CardTitle>Ad Requests Queue</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead>
                                    <tr className="border-b">
                                        <th className="px-4 py-3 text-left text-sm font-medium">Date</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Advertiser</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Ad Details</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Slot</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Duration</th>
                                        <th className="px-4 py-3 text-right text-sm font-medium">Actions</th>
                                    </tr>
                                </thead>
                                <tbody className="divide-y divide-gray-200">
                                    {ads.data.length === 0 ? (
                                        <tr>
                                            <td colSpan={6} className="px-4 py-8 text-center text-muted-foreground">
                                                No pending requests found.
                                            </td>
                                        </tr>
                                    ) : (
                                        ads.data.map((ad: any) => (
                                            <tr key={ad.id} className="hover:bg-gray-50/50">
                                                <td className="px-4 py-3 text-sm text-muted-foreground whitespace-nowrap">
                                                    {format(new Date(ad.created_at), 'dd MMM yyyy')}
                                                    <div className="text-xs">{format(new Date(ad.created_at), 'hh:mm a')}</div>
                                                </td>
                                                <td className="px-4 py-3">
                                                    <div className="flex items-center gap-2">
                                                        <div className="h-8 w-8 rounded-full bg-gray-100 flex items-center justify-center">
                                                            <User className="h-4 w-4 text-gray-500" />
                                                        </div>
                                                        <div>
                                                            <div className="font-medium text-sm text-gray-900">{ad.user?.name || 'Unknown'}</div>
                                                            <div className="text-xs text-muted-foreground">
                                                                {ad.user?.unit?.building?.name || ''} - {ad.user?.unit?.unit_no || ''}
                                                            </div>
                                                        </div>
                                                    </div>
                                                </td>
                                                <td className="px-4 py-3">
                                                    <div className="font-medium text-sm">{ad.title}</div>
                                                    <div className="text-xs text-muted-foreground flex items-center gap-1 mt-1">
                                                        {ad.media_url ? (
                                                            <a href={ad.media_url} target="_blank" className="flex items-center text-blue-600 hover:underline">
                                                                <ImageIcon className="h-3 w-3 mr-1" /> View Media
                                                            </a>
                                                        ) : 'No Media'}
                                                        {ad.target_link && (
                                                            <>
                                                                <span className="mx-1">•</span>
                                                                <a href={ad.target_link} target="_blank" className="flex items-center text-blue-600 hover:underline">
                                                                    <ExternalLink className="h-3 w-3 mr-1" /> Link
                                                                </a>
                                                            </>
                                                        )}
                                                    </div>
                                                </td>
                                                <td className="px-4 py-3">
                                                    <Badge variant="outline">{ad.slot?.name || ad.slot?.position || 'Unknown'}</Badge>
                                                </td>
                                                <td className="px-4 py-3">
                                                    <div className="flex items-center gap-1 text-xs">
                                                        <Calendar className="h-3 w-3" />
                                                        <span>{format(new Date(ad.start_date), 'dd MMM')} - {format(new Date(ad.end_date), 'dd MMM')}</span>
                                                    </div>
                                                    <div className="text-xs text-muted-foreground mt-0.5">
                                                         {Math.ceil((new Date(ad.end_date).getTime() - new Date(ad.start_date).getTime()) / (1000 * 60 * 60 * 24))} Days
                                                    </div>
                                                    <div className="text-xs font-mono mt-1 text-green-600">
                                                        ₹{ad.price} Paid
                                                    </div>
                                                </td>
                                                <td className="px-4 py-3 text-right">
                                                    <div className="flex justify-end gap-2">
                                                        <Button size="sm" variant="default" className="bg-green-600 hover:bg-green-700 h-8 px-3" onClick={() => handleApprove(ad.id)}>
                                                            <Check className="h-4 w-4 mr-1" /> Approve
                                                        </Button>
                                                        <Button size="sm" variant="destructive" className="h-8 px-3" onClick={() => handleReject(ad.id)}>
                                                            <X className="h-4 w-4 mr-1" /> Reject
                                                        </Button>
                                                    </div>
                                                </td>
                                            </tr>
                                        ))
                                    )}
                                </tbody>
                            </table>
                        </div>

                         {/* Pagination */}
                         {ads.links && ads.links.length > 3 && (
                            <div className="mt-4 flex items-center justify-between">
                                <div className="text-sm text-muted-foreground">
                                    Page {ads.current_page} of {ads.last_page}
                                </div>
                                <div className="flex gap-2">
                                    {ads.links.map((link: any, index: number) => (
                                        <Link
                                            key={index}
                                            href={link.url || '#'}
                                            className={`px-3 py-1 rounded text-sm ${
                                                link.active
                                                    ? 'bg-primary text-primary-foreground'
                                                    : 'bg-muted hover:bg-muted/80'
                                            } ${!link.url ? 'opacity-50 pointer-events-none' : ''}`}
                                            dangerouslySetInnerHTML={{ __html: link.label }}
                                        />
                                    ))}
                                </div>
                            </div>
                        )}
                    </CardContent>
                </Card>
            </div>
        </SocietyAdminLayout>
    );
}
