import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Head, Link, router } from '@inertiajs/react';
import { Calendar, Trash2, Eye, Filter, MousePointerClick, Activity, Search } from 'lucide-react';
import { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import Swal from 'sweetalert2';
import { format, isPast } from 'date-fns';

export default function Index({ society, ads, filters }: any) {
    const [search, setSearch] = useState(filters.search || '');
    const [status, setStatus] = useState(filters.status || 'all');

    const handleDelete = (id: number) => {
        Swal.fire({
            title: 'Stop/Delete Campaign?',
            text: "This will remove the ad immediately.",
            icon: 'warning',
            showCancelButton: true,
            confirmButtonText: 'Yes, Delete',
            confirmButtonColor: '#EF4444',
        }).then((result) => {
            if (result.isConfirmed) {
                router.delete(`/societies/${society.id}/monetization/${id}`);
            }
        });
    };

    const handleSearch = () => {
        router.get(`/societies/${society.id}/monetization`, { 
            search,
            status: status === 'all' ? '' : status 
        }, { preserveState: true });
    };

    return (
        <SocietyAdminLayout society={society} breadcrumbs={[{ title: 'Active Campaigns', href: '#' }]}>
            <Head title="Ad Campaigns" />
            <div className="flex h-full flex-1 flex-col gap-6 overflow-x-auto rounded-xl p-6">
                <div className="flex justify-between items-center">
                    <div>
                        <h1 className="text-2xl font-bold">Ad Campaigns</h1>
                        <p className="text-muted-foreground">Monitor active and past advertisements</p>
                    </div>
                </div>

                {/* Filters */}
                {/* Filters */}
                <Card>
                    <CardContent className="pt-6">
                        <div className="flex gap-4">
                            <div className="flex-1">
                                <Label htmlFor="search">Search</Label>
                                <div className="relative mt-1">
                                    <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                    <Input
                                        id="search"
                                        placeholder="Search by title, user or building..."
                                        value={search}
                                        onChange={(e) => setSearch(e.target.value)}
                                        onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                        className="pl-9"
                                    />
                                </div>
                            </div>
                            <div className="w-56">
                                <Label htmlFor="status">Status</Label>
                                <Select value={status} onValueChange={setStatus}>
                                    <SelectTrigger id="status" className="mt-1">
                                        <SelectValue placeholder="All Status" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Status</SelectItem>
                                        <SelectItem value="active">Active</SelectItem>
                                        <SelectItem value="expired">Expired</SelectItem>
                                        <SelectItem value="rejected">Rejected</SelectItem>
                                        <SelectItem value="pending_approval">Pending</SelectItem>
                                    </SelectContent>
                                </Select>
                            </div>
                            <div className="flex items-end">
                                <Button onClick={handleSearch}>Filter</Button>
                            </div>
                        </div>
                    </CardContent>
                </Card>

                {/* Campaigns Table */}
                <Card>
                    <CardHeader>
                        <CardTitle>Campaign List</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead>
                                    <tr className="border-b">
                                        <th className="px-4 py-3 text-left text-sm font-medium">#ID</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Details</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Slot/Position</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Stats</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Status</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Expiry</th>
                                        <th className="px-4 py-3 text-right text-sm font-medium">Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {ads.data.length === 0 ? (
                                        <tr>
                                            <td colSpan={7} className="px-4 py-8 text-center text-muted-foreground">
                                                No campaigns found.
                                            </td>
                                        </tr>
                                    ) : (
                                        ads.data.map((ad: any) => {
                                            const isExpired = isPast(new Date(ad.end_date)) && ad.status === 'active';
                                            return (
                                            <tr key={ad.id} className="border-b hover:bg-muted/50">
                                                <td className="px-4 py-3 text-sm text-muted-foreground">#{ad.id}</td>
                                                <td className="px-4 py-3">
                                                    <div className="font-medium text-sm">{ad.title}</div>
                                                    <div className="text-xs text-muted-foreground mt-0.5">By {ad.user?.name || 'Unknown'}</div>
                                                    <div className="text-xs text-muted-foreground">
                                                        {ad.user?.unit ? `${ad.user.unit.building?.name || ''} - ${ad.user.unit.unit_no}` : ''}
                                                    </div>
                                                </td>
                                                <td className="px-4 py-3">
                                                    <Badge variant="outline">{ad.slot?.name || ad.slot?.position || 'N/A'}</Badge>
                                                </td>
                                                <td className="px-4 py-3">
                                                    <div className="flex flex-col gap-1 text-xs">
                                                        <div className="flex items-center gap-1"><Eye className="h-3 w-3 text-muted-foreground" /> {ad.views.toLocaleString()}</div>
                                                        <div className="flex items-center gap-1"><MousePointerClick className="h-3 w-3 text-muted-foreground" /> {ad.clicks.toLocaleString()}</div>
                                                    </div>
                                                </td>
                                                <td className="px-4 py-3">
                                                    {ad.status === 'active' && !isExpired ? (
                                                        <Badge className="bg-green-500 hover:bg-green-600">Active</Badge>
                                                    ) : ad.status === 'rejected' ? (
                                                        <Badge variant="destructive">Rejected</Badge>
                                                    ) : ad.status === 'pending_approval' ? (
                                                        <Badge variant="secondary">Pending</Badge>
                                                    ) : (
                                                        <Badge variant="secondary" className="bg-gray-200 text-gray-700 hover:bg-gray-300">Expired</Badge>
                                                    )}
                                                </td>
                                                <td className={`px-4 py-3 whitespace-nowrap text-sm ${isExpired ? 'text-red-600 font-medium' : ''}`}>
                                                    <div className="flex items-center gap-1">
                                                        <Calendar className="h-3 w-3" />
                                                        {format(new Date(ad.end_date), 'dd MMM yyyy')}
                                                    </div>
                                                </td>
                                                <td className="px-4 py-3 text-right">
                                                     <Button size="icon" variant="ghost" className="text-red-500 hover:text-red-700 hover:bg-red-50 h-8 w-8" onClick={() => handleDelete(ad.id)}>
                                                         <Trash2 className="h-4 w-4" />
                                                     </Button>
                                                </td>
                                            </tr>
                                        )})
                                    )}
                                </tbody>
                            </table>
                        </div>

                         {/* Pagination */}
                         {ads.links && ads.links.length > 3 && (
                            <div className="mt-4 flex items-center justify-between">
                                <div className="text-sm text-muted-foreground">
                                    Page {ads.current_page} of {ads.last_page}
                                </div>
                                <div className="flex gap-2">
                                    {ads.links.map((link: any, index: number) => (
                                        <Link
                                            key={index}
                                            href={link.url || '#'}
                                            className={`px-3 py-1 rounded text-sm ${
                                                link.active
                                                    ? 'bg-primary text-primary-foreground'
                                                    : 'bg-muted hover:bg-muted/80'
                                            } ${!link.url ? 'opacity-50 pointer-events-none' : ''}`}
                                            dangerouslySetInnerHTML={{ __html: link.label }}
                                        />
                                    ))}
                                </div>
                            </div>
                        )}
                    </CardContent>
                </Card>
            </div>
        </SocietyAdminLayout>
    );
}
