import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, useForm } from '@inertiajs/react';
import { Save } from 'lucide-react';
import { useEffect } from 'react';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Maintenance Structure',
        href: '/societies/{society}/maintenance-structures',
    },
];

interface MaintenanceStructure {
    id: number;
    name: string;
    maintenance: number;
    parking: number;
    other_charges: number;
    description: string | null;
}

interface IndexProps {
    society: {
        id: number;
        name: string;
        code?: string;
        address?: string;
        city?: string;
        state?: string;
        pincode?: string;
        phone?: string;
        email?: string;
        gst_no?: string;
        registration_no?: string;
        logo?: string;
    };
    structures: {
        data: MaintenanceStructure[];
    };
}

export default function Index({ society, structures }: IndexProps) {
    // Use the first structure as the default, or empty if none exists
    const existingStructure = structures.data.length > 0 ? structures.data[0] : null;

    const { data, setData, post, put, processing, errors } = useForm({
        name: existingStructure?.name || 'Default Structure',
        maintenance: existingStructure?.maintenance || 0,
        parking: existingStructure?.parking || 0,
        other_charges: existingStructure?.other_charges || 0,
        description: existingStructure?.description || '',
    });

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        if (existingStructure) {
            put(`/societies/${society.id}/maintenance-structures/${existingStructure.id}`);
        } else {
            post(`/societies/${society.id}/maintenance-structures`);
        }
    };

    const formatCurrency = (amount: number | string) => {
        const val = Number(amount) || 0;
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 0,
        }).format(val);
    };

    const totalAmount = Number(data.maintenance) + Number(data.parking) + Number(data.other_charges);

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Maintenance Structure" />
            <div className="flex h-full flex-1 flex-col gap-6 overflow-x-auto rounded-xl p-6">
                <div className="flex items-center justify-between">
                    <div>
                        <h1 className="text-2xl font-bold">Default Bill Structure</h1>
                        <p className="text-muted-foreground">
                            Define the default charges that will appear on your bills.
                        </p>
                    </div>
                    <Button onClick={handleSubmit} disabled={processing}>
                        <Save className="mr-2 h-4 w-4" />
                        {processing ? 'Saving...' : 'Save Structure'}
                    </Button>
                </div>

                {/* Invoice-like Editor */}
                <div className="max-w-4xl mx-auto w-full bg-white text-black shadow-lg rounded-lg overflow-hidden border">
                    <div className="p-8">
                        {/* Header */}
                        <div className="flex justify-between items-start border-b pb-8 mb-8">
                            <div className="flex gap-4">
                                {society.logo && (
                                    <img src={`/storage/${society.logo}`} alt="Logo" className="h-20 w-20 object-contain" />
                                )}
                                <div>
                                    <h2 className="text-3xl font-bold text-gray-900">{society.name}</h2>
                                    <div className="text-sm text-gray-600 mt-2 space-y-1">
                                        {society.address && <p>{society.address}</p>}
                                        <p>
                                            {[society.city, society.state, society.pincode].filter(Boolean).join(', ')}
                                        </p>
                                        {society.phone && <p>Phone: {society.phone}</p>}
                                        {society.email && <p>Email: {society.email}</p>}
                                    </div>
                                </div>
                            </div>
                            <div className="text-right">
                                <h1 className="text-4xl font-bold text-gray-200 uppercase tracking-widest">Structure</h1>
                                <div className="mt-4 text-sm text-gray-600">
                                    <p>Define your standard charges here.</p>
                                    <p>These will be applied to all new bills.</p>
                                </div>
                            </div>
                        </div>

                        {/* Bill To (Placeholder) */}
                        <div className="mb-8 p-4 bg-gray-50 rounded-md border border-gray-100">
                            <h3 className="text-xs font-bold text-gray-500 uppercase tracking-wider mb-2">Structure Name</h3>
                            <Input
                                value={data.name}
                                onChange={(e) => setData('name', e.target.value)}
                                className="max-w-md bg-white border-gray-300 focus:border-primary"
                                placeholder="e.g. Standard Maintenance"
                            />
                            {errors.name && <p className="text-sm text-red-500 mt-1">{errors.name}</p>}
                        </div>

                        {/* Line Items */}
                        <div className="mb-8">
                            <table className="w-full text-sm">
                                <thead>
                                    <tr className="border-b-2 border-gray-200">
                                        <th className="text-left py-3 font-bold text-gray-600 uppercase tracking-wider">Description</th>
                                        <th className="text-right py-3 font-bold text-gray-600 uppercase tracking-wider w-48">Amount (₹)</th>
                                    </tr>
                                </thead>
                                <tbody className="divide-y divide-gray-100">
                                    <tr>
                                        <td className="py-4 align-middle">
                                            <span className="font-medium text-gray-900">Maintenance Charges</span>
                                            <p className="text-xs text-gray-500 mt-0.5">Basic monthly maintenance fee</p>
                                        </td>
                                        <td className="py-4">
                                            <Input
                                                type="number"
                                                value={data.maintenance}
                                                onChange={(e) => setData('maintenance', parseFloat(e.target.value) || 0)}
                                                className="text-right font-mono"
                                            />
                                            {errors.maintenance && <p className="text-xs text-red-500 mt-1 text-right">{errors.maintenance}</p>}
                                        </td>
                                    </tr>
                                    <tr>
                                        <td className="py-4 align-middle">
                                            <span className="font-medium text-gray-900">Parking Charges</span>
                                            <p className="text-xs text-gray-500 mt-0.5">Charges for vehicle parking slots</p>
                                        </td>
                                        <td className="py-4">
                                            <Input
                                                type="number"
                                                value={data.parking}
                                                onChange={(e) => setData('parking', parseFloat(e.target.value) || 0)}
                                                className="text-right font-mono"
                                            />
                                            {errors.parking && <p className="text-xs text-red-500 mt-1 text-right">{errors.parking}</p>}
                                        </td>
                                    </tr>
                                    <tr>
                                        <td className="py-4 align-middle">
                                            <span className="font-medium text-gray-900">Other Charges</span>
                                            <p className="text-xs text-gray-500 mt-0.5">Any additional fixed charges</p>
                                        </td>
                                        <td className="py-4">
                                            <Input
                                                type="number"
                                                value={data.other_charges}
                                                onChange={(e) => setData('other_charges', parseFloat(e.target.value) || 0)}
                                                className="text-right font-mono"
                                            />
                                            {errors.other_charges && <p className="text-xs text-red-500 mt-1 text-right">{errors.other_charges}</p>}
                                        </td>
                                    </tr>
                                </tbody>
                                <tfoot>
                                    <tr className="border-t-2 border-gray-200 bg-gray-50">
                                        <td className="py-4 px-4 font-bold text-gray-900 text-lg">Total Monthly Bill</td>
                                        <td className="py-4 px-4 font-bold text-gray-900 text-lg text-right">
                                            {formatCurrency(totalAmount)}
                                        </td>
                                    </tr>
                                </tfoot>
                            </table>
                        </div>

                        {/* Footer / Notes */}
                        <div className="border-t pt-6">
                            <Label htmlFor="description" className="text-sm font-medium text-gray-700 mb-2 block">Internal Notes (Optional)</Label>
                            <Textarea
                                id="description"
                                value={data.description}
                                onChange={(e) => setData('description', e.target.value)}
                                placeholder="Add any notes about this structure..."
                                className="bg-gray-50 border-gray-200 focus:bg-white transition-colors"
                            />
                        </div>
                    </div>
                    
                    {/* Bottom Action Bar */}
                    <div className="bg-gray-50 px-8 py-4 border-t flex justify-between items-center">
                        <div className="text-sm text-gray-500">
                            {existingStructure ? 'Last updated: ' + new Date().toLocaleDateString() : 'Create your first structure'}
                        </div>
                        <Button onClick={handleSubmit} disabled={processing} size="lg">
                            <Save className="mr-2 h-4 w-4" />
                            {processing ? 'Saving...' : 'Save Structure'}
                        </Button>
                    </div>
                </div>
            </div>
        </SocietyAdminLayout>
    );
}
