import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, router, useForm as useInertiaForm } from '@inertiajs/react';
import { 
    ArrowLeft, Calendar, User, Phone, MapPin, 
    MessageSquare, Clock, CheckCircle2, AlertCircle, 
    MoreVertical, Send, Paperclip
} from 'lucide-react';
import { useState, useRef, useEffect } from 'react';
import axios from 'axios';
import Swal from 'sweetalert2';

interface HistoryItem {
    type: 'timeline' | 'comment';
    id: number;
    action?: string;
    description?: string; // For timeline
    message?: string; // For comment
    images?: string[];
    timestamp: string;
    user: string;
    user_role?: string;
    is_me?: boolean;
}

interface Ticket {
    id: number;
    ticket_no: string;
    title: string;
    description: string;
    priority: string;
    status: string;
    created_at: string;
    resolved_at: string | null;
    resolution_notes: string | null;
    user: {
        id: number;
        name: string;
        phone: string;
        email: string;
        unit?: {
            id: number;
            unit_no: string;
            building: { name: string };
        };
    };
    vendor: {
        id: number;
        name: string;
        phone: string;
    } | null;
    society_id: number;
}

interface ShowProps {
    society: { id: number; name: string; code: string };
    ticket: Ticket;
    history: HistoryItem[];
    vendors: { id: number; name: string }[];
}

const statusConfig = {
    open: { label: 'Open', color: 'bg-blue-100 text-blue-800', icon: AlertCircle },
    assigned: { label: 'Assigned', color: 'bg-yellow-100 text-yellow-800', icon: Clock },
    in_progress: { label: 'In Progress', color: 'bg-purple-100 text-purple-800', icon: Clock }, // Wrench?
    resolved: { label: 'Resolved', color: 'bg-green-100 text-green-800', icon: CheckCircle2 },
    closed: { label: 'Closed', color: 'bg-gray-100 text-gray-800', icon: CheckCircle2 },
};

const Toast = Swal.mixin({
    toast: true,
    position: 'top-end',
    showConfirmButton: false,
    timer: 3000,
    timerProgressBar: true,
});

export default function Show({ society, ticket, history: initialHistory, vendors }: ShowProps) {
    const [history, setHistory] = useState<HistoryItem[]>(initialHistory);
    const [replyText, setReplyText] = useState('');
    const [isSending, setIsSending] = useState(false);
    const [localTicket, setLocalTicket] = useState(ticket);

    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Maintenance', href: `/societies/${society.id}/maintenance` },
        { title: ticket.ticket_no, href: `/societies/${society.id}/maintenance/${ticket.id}` },
    ];

    const handleSendReply = async () => {
        if (!replyText.trim()) return;

        setIsSending(true);
        try {
            // Use the API endpoint we created
            const response = await axios.post(`/api/society-admin/tickets/${ticket.id}/reply`, {
                message: replyText,
                images: [] // TODO: Add file upload support
            });

            if (response.data.success) {
                Toast.fire({ icon: 'success', title: 'Reply sent successfully' });
                setReplyText('');
                // Add new comment to history locally (or reload)
                const newComment: HistoryItem = {
                    type: 'comment',
                    id: response.data.data.id,
                    message: response.data.data.message,
                    timestamp: new Date().toISOString(),
                    user: 'You', // Or Auth user name
                    user_role: 'Admin',
                    is_me: true,
                };
                setHistory([newComment, ...history]);
            }
        } catch (error) {
            console.error(error);
            Toast.fire({ icon: 'error', title: 'Failed to send reply' });
        } finally {
            setIsSending(false);
        }
    };

    const handleStatusUpdate = (newStatus: string) => {
         // Using Swal for confirmation
         Swal.fire({
            title: 'Are you sure?',
            text: `Change status to ${statusConfig[newStatus as keyof typeof statusConfig].label}?`,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#3085d6',
            cancelButtonColor: '#d33',
            confirmButtonText: 'Yes, change it!'
         }).then((result) => {
            if (result.isConfirmed) {
                 router.post(
                    `/societies/${society.id}/maintenance/${ticket.id}/update-status`,
                    { status: newStatus },
                    {
                        preserveScroll: true,
                        onSuccess: () => {
                            Toast.fire({ icon: 'success', title: 'Status updated' });
                            setLocalTicket({...localTicket, status: newStatus});
                        }
                    }
                );
            }
         });
    };

    const handleAssignVendor = (vendorId: string) => {
         router.post(
            `/societies/${society.id}/maintenance/${ticket.id}/assign-vendor`,
            { vendor_id: vendorId },
            { preserveScroll: true }
        );
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title={`Ticket ${ticket.ticket_no}`} />
            
            <div className="flex h-full flex-col gap-6 p-6">
                {/* Header */}
                <div className="flex items-center justify-between">
                    <div className="flex items-center gap-4">
                        <Button variant="ghost" size="icon" onClick={() => window.history.back()}>
                            <ArrowLeft className="h-5 w-5" />
                        </Button>
                        <div>
                            <h1 className="text-xl font-bold">{ticket.title}</h1>
                            <p className="text-sm text-muted-foreground flex items-center gap-2">
                                {ticket.ticket_no} • Created {new Date(ticket.created_at).toLocaleString()}
                            </p>
                        </div>
                    </div>
                     <div className="flex items-center gap-2">
                        <Select value={localTicket.status} onValueChange={handleStatusUpdate}>
                            <SelectTrigger className="w-[150px]">
                                <SelectValue />
                            </SelectTrigger>
                            <SelectContent>
                                {Object.entries(statusConfig).map(([key, config]) => (
                                    <SelectItem key={key} value={key}>{config.label}</SelectItem>
                                ))}
                            </SelectContent>
                        </Select>
                    </div>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 h-full min-h-0">
                    {/* Left Column: Ticket Details */}
                    <div className="space-y-6 lg:col-span-1 overflow-y-auto pr-2">
                        <Card>
                            <CardHeader>
                                <CardTitle className="text-base">Details</CardTitle>
                            </CardHeader>
                            <CardContent className="space-y-4">
                                <div>
                                    <label className="text-xs font-medium text-muted-foreground">Description</label>
                                    <p className="text-sm mt-1 whitespace-pre-wrap">{ticket.description}</p>
                                </div>
                                <div className="grid grid-cols-2 gap-4">
                                     <div>
                                        <label className="text-xs font-medium text-muted-foreground">Priority</label>
                                        <div className="mt-1 capitalize text-sm font-medium">{ticket.priority}</div>
                                     </div>
                                </div>
                            </CardContent>
                        </Card>

                        <Card>
                            <CardHeader>
                                <CardTitle className="text-base">Resident</CardTitle>
                            </CardHeader>
                            <CardContent className="space-y-4">
                                <div className="flex items-center gap-3">
                                    <div className="h-10 w-10 bg-gray-100 rounded-full flex items-center justify-center">
                                        <User className="h-5 w-5 text-gray-500" />
                                    </div>
                                    <div>
                                        <div className="font-medium text-sm">{ticket.user.name}</div>
                                        {ticket.user.unit && (
                                            <div className="text-xs text-muted-foreground">
                                                {ticket.user.unit.building.name} - {ticket.user.unit.unit_no}
                                            </div>
                                        )}
                                    </div>
                                </div>
                                <div className="space-y-2 text-sm">
                                    <div className="flex items-center gap-2">
                                        <Phone className="h-4 w-4 text-muted-foreground" />
                                        <span>{ticket.user.phone}</span>
                                    </div>
                                </div>
                            </CardContent>
                        </Card>

                        <Card>
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-base">Assigned Vendor</CardTitle>
                            </CardHeader>
                            <CardContent className="pt-4">
                                <Select 
                                    value={ticket.vendor?.id.toString() || 'none'} 
                                    onValueChange={handleAssignVendor}
                                >
                                    <SelectTrigger>
                                        <SelectValue placeholder="Assign Vendor" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="none">None</SelectItem>
                                        {vendors.map(v => (
                                            <SelectItem key={v.id} value={v.id.toString()}>{v.name}</SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                                {ticket.vendor && (
                                    <div className="mt-4 flex items-center gap-3">
                                        <div className="h-8 w-8 bg-blue-50 rounded-full flex items-center justify-center">
                                            <User className="h-4 w-4 text-blue-600" />
                                        </div>
                                        <div>
                                            <div className="font-medium text-sm">{ticket.vendor.name}</div>
                                            <div className="text-xs text-muted-foreground">{ticket.vendor.phone}</div>
                                        </div>
                                    </div>
                                )}
                            </CardContent>
                        </Card>
                    </div>

                    {/* Right Column: Activity & Comments */}
                    <div className="lg:col-span-2 flex flex-col h-[600px] lg:h-auto bg-white rounded-xl border shadow-sm">
                        <div className="p-4 border-b">
                            <h3 className="font-medium flex items-center gap-2">
                                <Clock className="h-4 w-4" /> Activity & Comments
                            </h3>
                        </div>
                        
                        {/* Feed */}
                        <div className="flex-1 overflow-y-auto p-4 space-y-6 bg-gray-50/50">
                            {history.length === 0 && (
                                <p className="text-center text-muted-foreground text-sm py-10">No activity yet.</p>
                            )}
                            
                            {history.map((item, index) => {
                                if (item.type === 'timeline') {
                                    return (
                                        <div key={`timeline-${item.id}-${index}`} className="flex gap-4 justify-center">
                                            <div className="bg-gray-100 rounded-full px-3 py-1 text-xs text-muted-foreground flex items-center gap-2 border">
                                                <span className="font-medium">{item.user}</span>
                                                <span>{item.description || item.action}</span>
                                                <span className="text-gray-400">• {new Date(item.timestamp).toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'})}</span>
                                            </div>
                                        </div>
                                    );
                                } else {
                                    // Comment
                                    return (
                                        <div key={`comment-${item.id}-${index}`} className={`flex gap-3 ${item.is_me ? 'flex-row-reverse' : ''}`}>
                                            <div className="h-8 w-8 rounded-full bg-blue-100 flex items-center justify-center flex-shrink-0">
                                                <span className="text-xs font-semibold text-blue-700">
                                                    {item.user.charAt(0)}
                                                </span>
                                            </div>
                                            <div className={`flex flex-col max-w-[80%] ${item.is_me ? 'items-end' : 'items-start'}`}>
                                                <div className="flex items-center gap-2 mb-1">
                                                    <span className="text-xs font-semibold">{item.user}</span>
                                                    <span className="text-xs text-muted-foreground">{new Date(item.timestamp).toLocaleString()}</span>
                                                </div>
                                                <div className={`rounded-2xl px-4 py-2 text-sm ${
                                                    item.is_me 
                                                        ? 'bg-primary text-primary-foreground rounded-tr-sm' 
                                                        : 'bg-white border rounded-tl-sm shadow-sm'
                                                }`}>
                                                    <p>{item.message}</p>
                                                    {item.images && item.images.length > 0 && (
                                                        <div className="mt-2 text-xs opacity-75">[Image Attachment] (TODO)</div>
                                                    )}
                                                </div>
                                            </div>
                                        </div>
                                    );
                                }
                            })}
                        </div>

                        {/* Reply Input */}
                        <div className="p-4 bg-white border-t">
                            <div className="flex gap-4">
                                <Textarea 
                                    placeholder="Type a reply..." 
                                    value={replyText}
                                    onChange={(e) => setReplyText(e.target.value)}
                                    className="resize-none min-h-[80px]"
                                    onKeyDown={(e) => {
                                        if (e.key === 'Enter' && !e.shiftKey) {
                                            e.preventDefault();
                                            handleSendReply();
                                        }
                                    }}
                                />
                                <div className="flex flex-col justify-between">
                                    <Button 
                                        size="icon" 
                                        variant="outline"
                                        title="Attach File (Coming Soon)"
                                        disabled
                                    >
                                        <Paperclip className="h-4 w-4" />
                                    </Button>
                                    <Button 
                                        size="icon" 
                                        onClick={handleSendReply}
                                        disabled={!replyText.trim() || isSending}
                                    >
                                        <Send className="h-4 w-4" />
                                    </Button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </SocietyAdminLayout>
    );
}
