import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Textarea } from '@/components/ui/textarea';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link, router, useForm } from '@inertiajs/react';
import { Wrench, Plus, Search, Eye, AlertCircle, Clock, CheckCircle2, XCircle, User, FileText } from 'lucide-react';
import { useState, useEffect } from 'react';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Maintenance',
        href: '/societies/{society}/maintenance',
    },
];

interface Vendor {
    id: number;
    name: string;
    phone: string;
}

interface Resident {
    id: number;
    name: string;
    phone: string;
    unit_id: number | null;
    unit?: {
        id: number;
        unit_no: string;
    } | null;
}

interface Ticket {
    id: number;
    ticket_no: string;
    title: string;
    description: string;
    priority: string;
    status: string;
    assigned_at: string | null;
    resolved_at: string | null;
    resolution_notes: string | null;
    user: {
        id: number;
        name: string;
        phone: string;
        unit_id: number | null;
        unit: {
            id: number;
            unit_no: string;
        } | null;
    };
    vendor: {
        id: number;
        name: string;
        phone: string;
    } | null;
    created_at: string;
}

interface IndexProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    tickets: {
        data: Ticket[];
        links: any;
        meta: any;
    };
    vendors: Vendor[];
    residents: Resident[];
    filters: {
        search?: string;
        priority?: string;
        vendor_id?: string;
    };
    stats: {
        total: number;
        open: number;
        in_progress: number;
        resolved: number;
    };
}

const statusConfig = {
    open: { label: 'Open', color: 'bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400', icon: AlertCircle },
    assigned: { label: 'Assigned', color: 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400', icon: Clock },
    in_progress: { label: 'In Progress', color: 'bg-purple-100 text-purple-800 dark:bg-purple-900/30 dark:text-purple-400', icon: Wrench },
    resolved: { label: 'Resolved', color: 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400', icon: CheckCircle2 },
    closed: { label: 'Closed', color: 'bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-400', icon: XCircle },
};

const priorityConfig = {
    low: { label: 'Low', color: 'bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-300' },
    medium: { label: 'Medium', color: 'bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400' },
    high: { label: 'High', color: 'bg-orange-100 text-orange-800 dark:bg-orange-900/30 dark:text-orange-400' },
    urgent: { label: 'Urgent', color: 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400' },
};

export default function Index({ society, tickets, vendors, residents, filters, stats }: IndexProps) {
    const [search, setSearch] = useState(filters.search || '');
    const [priority, setPriority] = useState(filters.priority || 'all');
    const [vendorId, setVendorId] = useState(filters.vendor_id || 'all');

    // Modal States
    const [isCreateOpen, setIsCreateOpen] = useState(false);

    // Create Form
    const createForm = useForm({
        title: '',
        description: '',
        priority: 'medium',
        user_id: 'none',
        vendor_id: 'none',
    });

    // Reset form when modal closes
    useEffect(() => {
        if (!isCreateOpen) {
            createForm.reset();
        }
    }, [isCreateOpen]);

    const handleSearch = () => {
        const filterParams: any = {};
        if (search) filterParams.search = search;
        if (priority && priority !== 'all') filterParams.priority = priority;
        if (vendorId && vendorId !== 'all') filterParams.vendor_id = vendorId;

        router.get(`/societies/${society.id}/maintenance`, filterParams, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const handleCreateSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        createForm.transform((data) => ({
            ...data,
            user_id: data.user_id === 'none' ? null : data.user_id,
            vendor_id: data.vendor_id === 'none' ? null : data.vendor_id,
        }));
        createForm.post(`/societies/${society.id}/maintenance`, {
            onSuccess: () => setIsCreateOpen(false),
        });
    };



    const handleStatusUpdate = (ticketId: number, newStatus: string) => {
        if (confirm(`Are you sure you want to change the status to ${statusConfig[newStatus as keyof typeof statusConfig].label}?`)) {
            router.post(
                `/societies/${society.id}/maintenance/${ticketId}/update-status`,
                { status: newStatus },
                {
                    preserveScroll: true,
                    onSuccess: () => {
                        // Status updated
                    }
                }
            );
        }
    };

    const handleAssignVendor = (ticketId: number, vendorId: string) => {
        router.post(
            `/societies/${society.id}/maintenance/${ticketId}/assign-vendor`,
            { vendor_id: vendorId },
            {
                preserveScroll: true,
                onSuccess: () => {
                     // Status updated
                }
            }
        );
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Maintenance Tickets" />
            <div className="flex h-full flex-1 flex-col gap-6 overflow-x-auto rounded-xl p-6">
                <div className="flex items-center justify-between">
                    <div>
                        <h1 className="text-2xl font-bold">Maintenance Tickets</h1>
                        <p className="text-muted-foreground">
                            Manage all maintenance tickets in your society
                        </p>
                    </div>
                    
                    <Dialog open={isCreateOpen} onOpenChange={setIsCreateOpen}>
                        <DialogTrigger asChild>
                            <Button>
                                <Plus className="mr-2 h-4 w-4" />
                                Create Ticket
                            </Button>
                        </DialogTrigger>
                        <DialogContent className="sm:max-w-[600px]">
                            <DialogHeader>
                                <DialogTitle>Create Maintenance Ticket</DialogTitle>
                                <DialogDescription>
                                    Create a new ticket for a resident or common area issue.
                                </DialogDescription>
                            </DialogHeader>
                            <form onSubmit={handleCreateSubmit} className="space-y-4 py-4">
                                <div className="space-y-2">
                                    <Label htmlFor="title">Issue Title</Label>
                                    <Input
                                        id="title"
                                        value={createForm.data.title}
                                        onChange={(e) => createForm.setData('title', e.target.value)}
                                        placeholder="e.g., Leaking tap in kitchen"
                                        required
                                    />
                                    {createForm.errors.title && <p className="text-sm text-red-500">{createForm.errors.title}</p>}
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="description">Description</Label>
                                    <Textarea
                                        id="description"
                                        value={createForm.data.description}
                                        onChange={(e) => createForm.setData('description', e.target.value)}
                                        placeholder="Describe the issue in detail..."
                                        required
                                    />
                                    {createForm.errors.description && <p className="text-sm text-red-500">{createForm.errors.description}</p>}
                                </div>

                                <div className="grid grid-cols-2 gap-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="priority">Priority</Label>
                                        <Select 
                                            value={createForm.data.priority} 
                                            onValueChange={(value) => createForm.setData('priority', value)}
                                        >
                                            <SelectTrigger id="priority">
                                                <SelectValue />
                                            </SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="low">Low</SelectItem>
                                                <SelectItem value="medium">Medium</SelectItem>
                                                <SelectItem value="high">High</SelectItem>
                                                <SelectItem value="urgent">Urgent</SelectItem>
                                            </SelectContent>
                                        </Select>
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="user_id">Resident (Optional)</Label>
                                        <Select 
                                            value={createForm.data.user_id} 
                                            onValueChange={(value) => createForm.setData('user_id', value)}
                                        >
                                            <SelectTrigger id="user_id">
                                                <SelectValue placeholder="Select Resident" />
                                            </SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="none">None (Common Area)</SelectItem>
                                                {residents.map((resident) => (
                                                    <SelectItem key={resident.id} value={resident.id.toString()}>
                                                        {resident.name} {resident.unit ? `(${resident.unit.unit_no})` : ''}
                                                    </SelectItem>
                                                ))}
                                            </SelectContent>
                                        </Select>
                                        {createForm.errors.user_id && <p className="text-sm text-red-500">{createForm.errors.user_id}</p>}
                                    </div>
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="vendor_id">Assign Vendor (Optional)</Label>
                                    <Select 
                                        value={createForm.data.vendor_id} 
                                        onValueChange={(value) => createForm.setData('vendor_id', value)}
                                    >
                                        <SelectTrigger id="vendor_id">
                                            <SelectValue placeholder="Select Vendor" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="none">None</SelectItem>
                                            {vendors.map((vendor) => (
                                                <SelectItem key={vendor.id} value={vendor.id.toString()}>
                                                    {vendor.name}
                                                </SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                    {createForm.errors.vendor_id && <p className="text-sm text-red-500">{createForm.errors.vendor_id}</p>}
                                </div>

                                <DialogFooter>
                                    <Button type="button" variant="outline" onClick={() => setIsCreateOpen(false)}>Cancel</Button>
                                    <Button type="submit" disabled={createForm.processing}>
                                        {createForm.processing ? 'Creating...' : 'Create Ticket'}
                                    </Button>
                                </DialogFooter>
                            </form>
                        </DialogContent>
                    </Dialog>
                </div>

                {/* KPI Cards */}
                <div className="grid gap-4 md:grid-cols-4">
                    <Card className="border-l-4 border-l-gray-500">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Total Tickets</CardTitle>
                            <FileText className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.total}</div>
                            <p className="text-xs text-muted-foreground">
                                All time
                            </p>
                        </CardContent>
                    </Card>
                    <Card className="border-l-4 border-l-blue-500">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Open</CardTitle>
                            <AlertCircle className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.open}</div>
                            <p className="text-xs text-muted-foreground">
                                Needs attention
                            </p>
                        </CardContent>
                    </Card>
                    <Card className="border-l-4 border-l-purple-500">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">In Progress</CardTitle>
                            <Wrench className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.in_progress}</div>
                            <p className="text-xs text-muted-foreground">
                                Being worked on
                            </p>
                        </CardContent>
                    </Card>
                    <Card className="border-l-4 border-l-green-500">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Resolved</CardTitle>
                            <CheckCircle2 className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.resolved}</div>
                            <p className="text-xs text-muted-foreground">
                                Completed
                            </p>
                        </CardContent>
                    </Card>
                </div>

                {/* Filters */}
                <Card>
                    <CardContent className="pt-6">
                        <div className="flex gap-4 flex-wrap">
                            <div className="flex-1 min-w-[200px]">
                                <Label htmlFor="search">Search</Label>
                                <div className="relative mt-1">
                                    <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                    <Input
                                        id="search"
                                        placeholder="Search by ticket no, title, resident..."
                                        value={search}
                                        onChange={(e) => setSearch(e.target.value)}
                                        onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                        className="pl-9"
                                    />
                                </div>
                            </div>
                            <div className="w-48">
                                <Label htmlFor="priority">Priority</Label>
                                <Select value={priority} onValueChange={setPriority}>
                                    <SelectTrigger id="priority" className="mt-1">
                                        <SelectValue placeholder="All Priorities" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Priorities</SelectItem>
                                        <SelectItem value="low">Low</SelectItem>
                                        <SelectItem value="medium">Medium</SelectItem>
                                        <SelectItem value="high">High</SelectItem>
                                        <SelectItem value="urgent">Urgent</SelectItem>
                                    </SelectContent>
                                </Select>
                            </div>
                            <div className="w-48">
                                <Label htmlFor="vendor_id">Vendor</Label>
                                <Select value={vendorId} onValueChange={setVendorId}>
                                    <SelectTrigger id="vendor_id" className="mt-1">
                                        <SelectValue placeholder="All Vendors" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Vendors</SelectItem>
                                        {vendors.map((vendor) => (
                                            <SelectItem key={vendor.id} value={vendor.id.toString()}>
                                                {vendor.name}
                                            </SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                            </div>
                            <div className="flex items-end">
                                <Button onClick={handleSearch}>Filter</Button>
                            </div>
                        </div>
                    </CardContent>
                </Card>

                {/* Tickets Table */}
                <Card>
                    <CardHeader>
                        <CardTitle>All Tickets</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead>
                                    <tr className="border-b">
                                        <th className="px-4 py-3 text-left text-sm font-medium">Ticket</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Priority</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Resident</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Vendor</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Created</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Status</th>
                                        <th className="px-4 py-3 text-right text-sm font-medium">Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {tickets.data.length > 0 ? (
                                        tickets.data.map((ticket) => {
                                            const StatusIcon = statusConfig[ticket.status as keyof typeof statusConfig].icon;
                                            return (
                                                <tr key={ticket.id} className="border-b hover:bg-muted/50">
                                                    <td className="px-4 py-3">
                                                        <div>
                                                            <div className="font-medium">{ticket.title}</div>
                                                            <div className="text-xs text-muted-foreground font-mono">
                                                                {ticket.ticket_no}
                                                            </div>
                                                        </div>
                                                    </td>
                                                    <td className="px-4 py-3">
                                                        <span
                                                            className={`inline-flex rounded-full px-2 py-1 text-xs font-medium ${priorityConfig[ticket.priority as keyof typeof priorityConfig].color}`}
                                                        >
                                                            {priorityConfig[ticket.priority as keyof typeof priorityConfig].label}
                                                        </span>
                                                    </td>
                                                    <td className="px-4 py-3 text-sm">
                                                        {ticket.user ? (
                                                            <div>
                                                                <div className="font-medium">{ticket.user.name}</div>
                                                                {ticket.user.unit && (
                                                                    <div className="text-xs text-muted-foreground">
                                                                        {ticket.user.unit.unit_no}
                                                                    </div>
                                                                )}
                                                            </div>
                                                        ) : (
                                                            <span className="text-muted-foreground italic">Common Area</span>
                                                        )}
                                                    </td>
                                                    <td className="px-4 py-3 text-sm">
                                                        <Select
                                                            value={ticket.vendor?.id.toString() || 'none'}
                                                            onValueChange={(value) => handleAssignVendor(ticket.id, value)}
                                                        >
                                                            <SelectTrigger className="h-8 w-[140px] text-xs">
                                                                <SelectValue placeholder="Assign" />
                                                            </SelectTrigger>
                                                            <SelectContent>
                                                                <SelectItem value="none">None</SelectItem>
                                                                {vendors.map((vendor) => (
                                                                    <SelectItem key={vendor.id} value={vendor.id.toString()}>
                                                                        {vendor.name}
                                                                    </SelectItem>
                                                                ))}
                                                            </SelectContent>
                                                        </Select>
                                                    </td>
                                                    <td className="px-4 py-3 text-sm text-muted-foreground">
                                                        {new Date(ticket.created_at).toLocaleDateString()}
                                                    </td>
                                                    <td className="px-4 py-3">
                                                        <div className="flex items-center gap-2">
                                                            <span
                                                                className={`inline-flex items-center gap-1 rounded-full px-2 py-1 text-xs font-medium ${statusConfig[ticket.status as keyof typeof statusConfig].color}`}
                                                            >
                                                                <StatusIcon className="h-3 w-3" />
                                                                {statusConfig[ticket.status as keyof typeof statusConfig].label}
                                                            </span>
                                                        </div>
                                                    </td>
                                                    <td className="px-4 py-3">
                                                        <div className="flex items-center justify-end gap-2">
                                                            <Button 
                                                                variant="ghost" 
                                                                size="icon" 
                                                                className="h-8 w-8 hover:bg-blue-50 hover:text-blue-600"
                                                                asChild
                                                            >
                                                                <Link href={`/societies/${society.id}/maintenance/${ticket.id}`}>
                                                                    <Eye className="h-4 w-4" />
                                                                </Link>
                                                            </Button>
                                                        </div>
                                                    </td>
                                                </tr>
                                            );
                                        })
                                    ) : (
                                        <tr>
                                            <td colSpan={7} className="px-4 py-8 text-center text-muted-foreground">
                                                No tickets found
                                            </td>
                                        </tr>
                                    )}
                                </tbody>
                            </table>
                        </div>

                        {/* Pagination */}
                        {tickets.links && tickets.links.length > 3 && (
                            <div className="mt-4 flex items-center justify-between">
                                <div className="text-sm text-muted-foreground">
                                    Showing {tickets.meta.from} to {tickets.meta.to} of {tickets.meta.total} results
                                </div>
                                <div className="flex gap-2">
                                    {tickets.links.map((link: any, index: number) => (
                                        <Link
                                            key={index}
                                            href={link.url || '#'}
                                            className={`px-3 py-1 rounded ${
                                                link.active
                                                    ? 'bg-primary text-primary-foreground'
                                                    : 'bg-muted hover:bg-muted/80'
                                            }`}
                                            dangerouslySetInnerHTML={{ __html: link.label }}
                                        />
                                    ))}
                                </div>
                            </div>
                        )}
                    </CardContent>
                </Card>


            </div>
        </SocietyAdminLayout>
    );
}
