import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import {
    Dialog,
    DialogContent,
    DialogDescription,
    DialogFooter,
    DialogHeader,
    DialogTitle,
} from '@/components/ui/dialog';
import { Checkbox } from '@/components/ui/checkbox';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { type Facility, type FacilitySlot, type FacilityBooking } from '@/types/facility';
import { Head, Link, router, useForm } from '@inertiajs/react';
import { ArrowLeft, Edit, Clock, CalendarDays, ListOrdered, Plus, Trash2, IndianRupee, Users, Check, X, ChevronLeft, ChevronRight, Eye } from 'lucide-react';
import { useState, useMemo } from 'react';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';

interface ShowProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    facility: Facility & {
        slots: FacilitySlot[];
        bookings?: FacilityBooking[];
    };
    stats?: {
        total_bookings: number;
        pending_approval: number;
        confirmed: number;
        revenue_collected: string | number;
        revenue_pending: string | number;
    };
}

export default function Show({ society, facility, stats }: ShowProps) {
    const breadcrumbs: BreadcrumbItem[] = [
        {
            title: 'Facilities',
            href: `/societies/${society.id}/facilities`,
        },
        {
            title: facility.name,
            href: `/societies/${society.id}/facilities/${facility.id}`,
        },
    ];

    const [isAddSlotOpen, setIsAddSlotOpen] = useState(false);
    const [isEditSlotOpen, setIsEditSlotOpen] = useState(false);
    const [editingSlot, setEditingSlot] = useState<FacilitySlot | null>(null);
    const [viewBooking, setViewBooking] = useState<FacilityBooking | null>(null);
    const [isBookingModalOpen, setIsBookingModalOpen] = useState(false);

    const statusForm = useForm({
        booking_status: '',
        admin_notes: '',
    });

    const addSlotForm = useForm({
        name: '',
        start_time: '',
        end_time: '',
        price: '',
        max_bookings_per_slot: '1',
        is_active: true,
    });

    const editSlotForm = useForm({
        name: '',
        start_time: '',
        end_time: '',
        price: '',
        max_bookings_per_slot: '1',
        is_active: true,
    });

    const handleAddSlot = (e: React.FormEvent) => {
        e.preventDefault();
        router.post(`/societies/${society.id}/facilities/${facility.id}/slots`, {
            slots: [addSlotForm.data]
        }, {
            onSuccess: () => {
                setIsAddSlotOpen(false);
                addSlotForm.reset();
            },
        });
    };

    const handleEditSlot = (e: React.FormEvent) => {
        e.preventDefault();
        if (!editingSlot) return;
        
        router.post(`/societies/${society.id}/facilities/${facility.id}/slots`, {
            slots: [{ ...editSlotForm.data, id: editingSlot.id }]
        }, {
            onSuccess: () => {
                setIsEditSlotOpen(false);
                setEditingSlot(null);
                editSlotForm.reset();
            },
        });
    };

    const openEditSlotDialog = (slot: FacilitySlot) => {
        setEditingSlot(slot);
        editSlotForm.setData({
            name: slot.name,
            start_time: slot.start_time,
            end_time: slot.end_time,
            price: slot.price.toString(),
            max_bookings_per_slot: slot.max_bookings_per_slot.toString(),
            is_active: slot.is_active,
        });
        setIsEditSlotOpen(true);
    };

    const handleDeleteSlot = (slotId: number) => {
        if (confirm('Are you sure you want to delete this slot? This cannot be undone if there are active bookings.')) {
            router.delete(`/societies/${society.id}/facilities/${facility.id}/slots/${slotId}`);
        }
    };

    const openBookingModal = (booking: FacilityBooking) => {
        setViewBooking(booking);
        statusForm.setData({
            booking_status: booking.booking_status,
            admin_notes: booking.admin_notes || '',
        });
        setIsBookingModalOpen(true);
    };

    const handleStatusUpdate = (e: React.FormEvent, newStatus?: string) => {
        e.preventDefault();
        if (!viewBooking) return;

        const finalStatus = newStatus || statusForm.data.booking_status;
        
        router.post(`/societies/${society.id}/facility-bookings/${viewBooking.id}/status`, {
            ...statusForm.data,
            booking_status: finalStatus
        }, {
            onSuccess: () => {
                setIsBookingModalOpen(false);
                setViewBooking(null);
                statusForm.reset();
            },
        });
    };

    const formatCurrency = (amount: string | number) => {
        return `₹${parseFloat(amount.toString()).toLocaleString('en-IN')}`;
    };

    const recentBookings = facility.bookings?.slice(0, 10) || [];

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title={facility.name} />
            <div className="flex flex-1 flex-col gap-6 p-6 md:p-8 overflow-x-hidden bg-muted/10">
                {/* Header */}
                <div className="flex items-start justify-between">
                    <div className="flex-1">
                        <div className="flex items-center gap-3 mb-2">
                            <h1 className="text-3xl font-bold tracking-tight">{facility.name}</h1>
                            <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${
                                facility.is_active 
                                    ? 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400'
                                    : 'bg-gray-100 text-gray-800 dark:bg-gray-900/30 dark:text-gray-400'
                            }`}>
                                {facility.is_active ? 'Active' : 'Inactive'}
                            </span>
                        </div>
                        {facility.description && (
                            <p className="text-muted-foreground max-w-3xl">{facility.description}</p>
                        )}
                    </div>
                    
                    <div className="flex gap-2">
                        <Link href={`/societies/${society.id}/facilities`}>
                            <Button variant="outline">
                                <ArrowLeft className="mr-2 h-4 w-4" />
                                Back
                            </Button>
                        </Link>
                        <Link href={`/societies/${society.id}/facilities/${facility.id}/edit`}>
                            <Button>
                                <Edit className="mr-2 h-4 w-4" />
                                Edit Facility
                            </Button>
                        </Link>
                    </div>
                </div>

                {/* Facility Details Card */}
                <Card>
                    <CardHeader>
                        <CardTitle>Facility Information</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-4">
                            <div>
                                <h4 className="text-sm font-medium text-muted-foreground mb-1">Capacity</h4>
                                <p className="text-lg font-semibold flex items-center gap-1">
                                    <Users className="h-4 w-4" />
                                    {facility.capacity || 'Not specified'}
                                </p>
                            </div>
                            <div>
                                <h4 className="text-sm font-medium text-muted-foreground mb-1">Total Slots</h4>
                                <p className="text-lg font-semibold">{facility.slots?.length || 0}</p>
                            </div>
                            <div>
                                <h4 className="text-sm font-medium text-muted-foreground mb-1">Total Bookings</h4>
                                <p className="text-lg font-semibold">{stats?.total_bookings || 0}</p>
                            </div>
                            <div>
                                <h4 className="text-sm font-medium text-muted-foreground mb-1">Requires Approval</h4>
                                <p className="text-lg font-semibold">{facility.requires_approval ? 'Yes' : 'No'}</p>
                            </div>
                        </div>

                        {/* Amenities, Rules, and Policy in Row Format */}
                        {(facility.amenities || facility.rules || facility.cancellation_policy) && (
                            <div className="grid gap-6 md:grid-cols-3 mt-6 pt-6 border-t">
                                {facility.amenities && (
                                    <div>
                                        <h4 className="text-sm font-bold text-foreground mb-2 bg-blue-50 dark:bg-blue-950/30 px-3 py-2 rounded-md">Amenities</h4>
                                        <p className="text-sm whitespace-pre-line text-muted-foreground">{facility.amenities}</p>
                                    </div>
                                )}

                                {facility.rules && (
                                    <div>
                                        <h4 className="text-sm font-bold text-foreground mb-2 bg-amber-50 dark:bg-amber-950/30 px-3 py-2 rounded-md">Rules</h4>
                                        <p className="text-sm whitespace-pre-line text-muted-foreground">{facility.rules}</p>
                                    </div>
                                )}

                                {facility.cancellation_policy && (
                                    <div>
                                        <h4 className="text-sm font-bold text-foreground mb-2 bg-red-50 dark:bg-red-950/30 px-3 py-2 rounded-md">Cancellation Policy</h4>
                                        <p className="text-sm whitespace-pre-line text-muted-foreground">{facility.cancellation_policy}</p>
                                    </div>
                                )}
                            </div>
                        )}
                    </CardContent>
                </Card>

                {/* Statistics Cards */}
                {stats && (
                    <div className="grid gap-4 md:grid-cols-5">
                        <Card className="border-t-4 border-t-blue-500">
                            <CardHeader className="pb-2">
                                <CardTitle className="text-sm font-medium text-muted-foreground">Total Bookings</CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="text-2xl font-bold">{stats.total_bookings}</div>
                            </CardContent>
                        </Card>
                        <Card className="border-t-4 border-t-yellow-500">
                            <CardHeader className="pb-2">
                                <CardTitle className="text-sm font-medium text-muted-foreground">Pending Approval</CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="text-2xl font-bold">{stats.pending_approval}</div>
                            </CardContent>
                        </Card>
                        <Card className="border-t-4 border-t-green-500">
                            <CardHeader className="pb-2">
                                <CardTitle className="text-sm font-medium text-muted-foreground">Confirmed</CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="text-2xl font-bold">{stats.confirmed}</div>
                            </CardContent>
                        </Card>
                        <Card className="border-t-4 border-t-emerald-500">
                            <CardHeader className="pb-2">
                                <CardTitle className="text-sm font-medium text-muted-foreground">Revenue Collected</CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="text-2xl font-bold">{formatCurrency(stats.revenue_collected)}</div>
                            </CardContent>
                        </Card>
                        <Card className="border-t-4 border-t-orange-500">
                            <CardHeader className="pb-2">
                                <CardTitle className="text-sm font-medium text-muted-foreground">Revenue Pending</CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="text-2xl font-bold">{formatCurrency(stats.revenue_pending)}</div>
                            </CardContent>
                        </Card>
                    </div>
                )}

                {/* Tabs */}
                <Tabs defaultValue="slots" className="w-full">
                    <TabsList className="grid w-full grid-cols-3 lg:w-[400px]">
                        <TabsTrigger value="slots" className="flex items-center gap-2">
                            <Clock className="h-4 w-4" />
                            Time Slots
                        </TabsTrigger>
                        <TabsTrigger value="calendar" className="flex items-center gap-2">
                            <CalendarDays className="h-4 w-4" />
                            Calendar
                        </TabsTrigger>
                        <TabsTrigger value="bookings" className="flex items-center gap-2">
                            <ListOrdered className="h-4 w-4" />
                            Recent Bookings
                        </TabsTrigger>
                    </TabsList>

                    {/* Slots Tab */}
                    <TabsContent value="slots" className="mt-6">
                        <Card>
                            <CardHeader className="flex flex-row items-center justify-between">
                                <div>
                                    <CardTitle>Time Slots</CardTitle>
                                    <CardDescription>Manage booking time slots and pricing</CardDescription>
                                </div>
                                <Button onClick={() => setIsAddSlotOpen(true)}>
                                    <Plus className="mr-2 h-4 w-4" />
                                    Add Slot
                                </Button>
                            </CardHeader>
                            <CardContent>
                                <div className="space-y-3">
                                    {facility.slots && facility.slots.length > 0 ? (
                                        facility.slots.map((slot) => (
                                            <div key={slot.id} className="flex items-center justify-between p-4 border rounded-lg hover:bg-muted/50 transition-colors">
                                                <div className="flex-1">
                                                    <div className="flex items-center gap-2">
                                                        <h4 className="font-semibold">{slot.name}</h4>
                                                        <span className={`inline-flex items-center px-2 py-0.5 rounded-full text-xs font-medium ${
                                                            slot.is_active
                                                                ? 'bg-green-100 text-green-800'
                                                                : 'bg-gray-100 text-gray-800'
                                                        }`}>
                                                            {slot.is_active ? 'Active' : 'Inactive'}
                                                        </span>
                                                    </div>
                                                    <div className="flex items-center gap-4 mt-2 text-sm text-muted-foreground">
                                                        <span>{slot.time_range || `${slot.start_time} - ${slot.end_time}`}</span>
                                                        <span>•</span>
                                                        <span className="font-medium text-foreground">{formatCurrency(slot.price)}</span>
                                                        <span>•</span>
                                                        <span>Max {slot.max_bookings_per_slot} booking(s)</span>
                                                    </div>
                                                </div>
                                                <div className="flex gap-2">
                                                    <Button
                                                        variant="ghost"
                                                        size="icon"
                                                        onClick={() => openEditSlotDialog(slot)}
                                                    >
                                                        <Edit className="h-4 w-4" />
                                                    </Button>
                                                    <Button
                                                        variant="ghost"
                                                        size="icon"
                                                        onClick={() => handleDeleteSlot(slot.id)}
                                                    >
                                                        <Trash2 className="h-4 w-4" />
                                                    </Button>
                                                </div>
                                            </div>
                                        ))
                                    ) : (
                                        <div className="text-center py-12">
                                            <Clock className="h-12 w-12 text-muted-foreground/30 mx-auto mb-4" />
                                            <p className="text-muted-foreground mb-4">No time slots configured</p>
                                            <Button onClick={() => setIsAddSlotOpen(true)}>
                                                <Plus className="mr-2 h-4 w-4" />
                                                Add First Slot
                                            </Button>
                                        </div>
                                    )}
                                </div>
                            </CardContent>
                        </Card>
                    </TabsContent>

                    {/* Calendar Tab */}
                    <TabsContent value="calendar" className="mt-6">
                        <CalendarView facility={facility} society={society} onBookingClick={openBookingModal} />
                    </TabsContent>

                    {/* Recent Bookings Tab */}
                    <TabsContent value="bookings" className="mt-6">
                        <Card>
                            <CardHeader className="flex flex-row items-center justify-between">
                                <div>
                                    <CardTitle>Recent Bookings</CardTitle>
                                    <CardDescription>Last 10 bookings for this facility</CardDescription>
                                </div>
                                <Link href={`/societies/${society.id}/facility-bookings?facility_id=${facility.id}`}>
                                    <Button variant="outline" size="sm">
                                        View All
                                    </Button>
                                </Link>
                            </CardHeader>
                            <CardContent>
                                {recentBookings.length > 0 ? (
                                    <div className="space-y-3">
                                        {recentBookings.map((booking) => (
                                            <div 
                                                key={booking.id} 
                                                className="flex items-center justify-between p-4 border rounded-lg hover:bg-muted/50 transition-colors cursor-pointer"
                                                onClick={() => openBookingModal(booking)}
                                            >
                                                <div className="flex-1">
                                                    <div className="flex items-center gap-2">
                                                        <span className="font-semibold">{booking.user?.name}</span>
                                                        <span className="text-xs text-muted-foreground">
                                                            ({booking.user?.unit?.unit_no})
                                                        </span>
                                                    </div>
                                                    <div className="flex items-center gap-4 mt-1 text-sm text-muted-foreground">
                                                        <span>
                                                            {new Date(booking.booking_date).toLocaleDateString(undefined, {
                                                                year: 'numeric',
                                                                month: 'short',
                                                                day: 'numeric'
                                                            })}
                                                        </span>
                                                        <span>•</span>
                                                        <span>{booking.slot?.name}</span>
                                                        <span>•</span>
                                                        <span className="font-medium">{formatCurrency(booking.amount)}</span>
                                                    </div>
                                                </div>
                                                <div className="flex gap-2 items-center">
                                                    <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${
                                                        booking.booking_status === 'confirmed' ? 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400' :
                                                        booking.booking_status === 'pending' ? 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400' :
                                                        booking.booking_status === 'cancelled' ? 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400' :
                                                        'bg-gray-100 text-gray-800 dark:bg-gray-900/30 dark:text-gray-400'
                                                    }`}>
                                                        {booking.booking_status}
                                                    </span>
                                                    <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${
                                                        booking.payment_status === 'paid' ? 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400' :
                                                        'bg-orange-100 text-orange-800 dark:bg-orange-900/30 dark:text-orange-400'
                                                    }`}>
                                                        {booking.payment_status}
                                                    </span>
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                ) : (
                                    <div className="text-center py-12">
                                        <ListOrdered className="h-12 w-12 text-muted-foreground/30 mx-auto mb-4" />
                                        <p className="text-muted-foreground">No bookings yet</p>
                                    </div>
                                )}
                            </CardContent>
                        </Card>
                    </TabsContent>
                </Tabs>

                {/* Add Slot Dialog */}
                <Dialog open={isAddSlotOpen} onOpenChange={setIsAddSlotOpen}>
                    <DialogContent>
                        <DialogHeader>
                            <DialogTitle>Add Time Slot</DialogTitle>
                            <DialogDescription>
                                Create a new bookable time slot for this facility
                            </DialogDescription>
                        </DialogHeader>
                        <form onSubmit={handleAddSlot} className="space-y-4">
                            <div className="space-y-2">
                                <Label htmlFor="slot_name">Slot Name</Label>
                                <Input
                                    id="slot_name"
                                    value={addSlotForm.data.name}
                                    onChange={(e) => addSlotForm.setData('name', e.target.value)}
                                    placeholder="e.g. Morning Slot"
                                    required
                                />
                            </div>
                            <div className="grid grid-cols-2 gap-4">
                                <div className="space-y-2">
                                    <Label htmlFor="start_time">Start Time</Label>
                                    <Input
                                        id="start_time"
                                        type="time"
                                        value={addSlotForm.data.start_time}
                                        onChange={(e) => addSlotForm.setData('start_time', e.target.value)}
                                        required
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="end_time">End Time</Label>
                                    <Input
                                        id="end_time"
                                        type="time"
                                        value={addSlotForm.data.end_time}
                                        onChange={(e) => addSlotForm.setData('end_time', e.target.value)}
                                        required
                                    />
                                </div>
                            </div>
                            <div className="grid grid-cols-2 gap-4">
                                <div className="space-y-2">
                                    <Label htmlFor="price">Price (₹)</Label>
                                    <Input
                                        id="price"
                                        type="number"
                                        min="0"
                                        step="0.01"
                                        value={addSlotForm.data.price}
                                        onChange={(e) => addSlotForm.setData('price', e.target.value)}
                                        placeholder="5000"
                                        required
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="max_bookings">Max Bookings</Label>
                                    <Input
                                        id="max_bookings"
                                        type="number"
                                        min="1"
                                        value={addSlotForm.data.max_bookings_per_slot}
                                        onChange={(e) => addSlotForm.setData('max_bookings_per_slot', e.target.value)}
                                        required
                                    />
                                </div>
                            </div>
                            <div className="flex items-center gap-2">
                                <Checkbox
                                    id="is_active"
                                    checked={addSlotForm.data.is_active}
                                    onCheckedChange={(checked) => addSlotForm.setData('is_active', !!checked)}
                                />
                                <Label htmlFor="is_active">Active</Label>
                            </div>
                            <DialogFooter>
                                <Button type="button" variant="outline" onClick={() => setIsAddSlotOpen(false)}>
                                    Cancel
                                </Button>
                                <Button type="submit" disabled={addSlotForm.processing}>
                                    {addSlotForm.processing ? 'Adding...' : 'Add Slot'}
                                </Button>
                            </DialogFooter>
                        </form>
                    </DialogContent>
                </Dialog>

                {/* Edit Slot Dialog */}
                <Dialog open={isEditSlotOpen} onOpenChange={setIsEditSlotOpen}>
                    <DialogContent>
                        <DialogHeader>
                            <DialogTitle>Edit Time Slot</DialogTitle>
                            <DialogDescription>
                                Update slot details and pricing
                            </DialogDescription>
                        </DialogHeader>
                        <form onSubmit={handleEditSlot} className="space-y-4">
                            <div className="space-y-2">
                                <Label htmlFor="edit_slot_name">Slot Name</Label>
                                <Input
                                    id="edit_slot_name"
                                    value={editSlotForm.data.name}
                                    onChange={(e) => editSlotForm.setData('name', e.target.value)}
                                    placeholder="e.g. Morning Slot"
                                    required
                                />
                            </div>
                            <div className="grid grid-cols-2 gap-4">
                                <div className="space-y-2">
                                    <Label htmlFor="edit_start_time">Start Time</Label>
                                    <Input
                                        id="edit_start_time"
                                        type="time"
                                        value={editSlotForm.data.start_time}
                                        onChange={(e) => editSlotForm.setData('start_time', e.target.value)}
                                        required
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_end_time">End Time</Label>
                                    <Input
                                        id="edit_end_time"
                                        type="time"
                                        value={editSlotForm.data.end_time}
                                        onChange={(e) => editSlotForm.setData('end_time', e.target.value)}
                                        required
                                    />
                                </div>
                            </div>
                            <div className="grid grid-cols-2 gap-4">
                                <div className="space-y-2">
                                    <Label htmlFor="edit_price">Price (₹)</Label>
                                    <Input
                                        id="edit_price"
                                        type="number"
                                        min="0"
                                        step="0.01"
                                        value={editSlotForm.data.price}
                                        onChange={(e) => editSlotForm.setData('price', e.target.value)}
                                        placeholder="5000"
                                        required
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_max_bookings">Max Bookings</Label>
                                    <Input
                                        id="edit_max_bookings"
                                        type="number"
                                        min="1"
                                        value={editSlotForm.data.max_bookings_per_slot}
                                        onChange={(e) => editSlotForm.setData('max_bookings_per_slot', e.target.value)}
                                        required
                                    />
                                </div>
                            </div>
                            <div className="flex items-center gap-2">
                                <Checkbox
                                    id="edit_is_active"
                                    checked={editSlotForm.data.is_active}
                                    onCheckedChange={(checked) => editSlotForm.setData('is_active', !!checked)}
                                />
                                <Label htmlFor="edit_is_active">Active</Label>
                            </div>
                            <DialogFooter>
                                <Button type="button" variant="outline" onClick={() => setIsEditSlotOpen(false)}>
                                    Cancel
                                </Button>
                                <Button type="submit" disabled={editSlotForm.processing}>
                                    {editSlotForm.processing ? 'Saving...' : 'Save Changes'}
                                </Button>
                            </DialogFooter>
                        </form>
                    </DialogContent>
                </Dialog>

                {/* Booking View & Approval Modal */}
                <Dialog open={isBookingModalOpen} onOpenChange={setIsBookingModalOpen}>
                    <DialogContent className="max-w-2xl">
                        <DialogHeader>
                            <DialogTitle>Booking Details</DialogTitle>
                            <DialogDescription>
                                Review details and update status
                            </DialogDescription>
                        </DialogHeader>
                        {viewBooking && (
                            <div className="space-y-6">
                                <div className="grid gap-4 md:grid-cols-2 bg-muted/20 p-4 rounded-lg">
                                    <div>
                                        <h4 className="text-xs font-medium text-muted-foreground uppercase mb-1">Resident</h4>
                                        <p className="font-medium">{viewBooking.user?.name}</p>
                                        <p className="text-xs text-muted-foreground">{viewBooking.user?.unit?.unit_no}</p>
                                    </div>
                                    <div>
                                        <h4 className="text-xs font-medium text-muted-foreground uppercase mb-1">Date & Time</h4>
                                        <p className="font-medium">
                                            {new Date(viewBooking.booking_date).toLocaleDateString(undefined, {
                                                year: 'numeric',
                                                month: 'long',
                                                day: 'numeric'
                                            })}
                                        </p>
                                        <p className="text-sm text-muted-foreground">{viewBooking.slot?.name}</p>
                                    </div>
                                    <div>
                                        <h4 className="text-xs font-medium text-muted-foreground uppercase mb-1">Amount</h4>
                                        <p className="font-bold text-lg">{formatCurrency(viewBooking.amount)}</p>
                                        <span className={`inline-flex items-center px-2 py-0.5 rounded text-xs font-medium ${
                                            viewBooking.payment_status === 'paid' ? 'bg-green-100 text-green-800' : 'bg-orange-100 text-orange-800'
                                        }`}>
                                            {viewBooking.payment_status.toUpperCase()} ({viewBooking.payment_method})
                                        </span>
                                    </div>
                                    <div>
                                        <h4 className="text-xs font-medium text-muted-foreground uppercase mb-1">Status</h4>
                                        <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${
                                            viewBooking.booking_status === 'confirmed' ? 'bg-green-100 text-green-800' :
                                            viewBooking.booking_status === 'pending' ? 'bg-yellow-100 text-yellow-800' :
                                            viewBooking.booking_status === 'cancelled' ? 'bg-red-100 text-red-800' : 'bg-gray-100 text-gray-800'
                                        }`}>
                                            {viewBooking.booking_status.toUpperCase()}
                                        </span>
                                    </div>
                                </div>

                                {viewBooking.booking_status === 'pending' ? (
                                    <div className="border-t pt-4">
                                        <h4 className="text-sm font-semibold mb-3">Action Required</h4>
                                        <form onSubmit={(e) => handleStatusUpdate(e)} className="space-y-4">
                                            <div className="flex gap-4">
                                                <Button 
                                                    type="button" 
                                                    className="flex-1 bg-green-600 hover:bg-green-700 text-white"
                                                    onClick={(e) => handleStatusUpdate(e, 'confirmed')}
                                                    disabled={statusForm.processing}
                                                >
                                                    <Check className="mr-2 h-4 w-4" />
                                                    Approve Booking
                                                </Button>
                                                <Button 
                                                    type="button" 
                                                    variant="destructive" 
                                                    className="flex-1"
                                                    onClick={(e) => handleStatusUpdate(e, 'rejected')}
                                                    disabled={statusForm.processing}
                                                >
                                                    <X className="mr-2 h-4 w-4" />
                                                    Reject
                                                </Button>
                                            </div>
                                            <div className="relative">
                                                <div className="absolute inset-0 flex items-center">
                                                    <span className="w-full border-t" />
                                                </div>
                                                <div className="relative flex justify-center text-xs uppercase">
                                                    <span className="bg-background px-2 text-muted-foreground">Or update details</span>
                                                </div>
                                            </div>
                                        </form>
                                    </div>
                                ) : (
                                    <div className="bg-blue-50 dark:bg-blue-950/20 p-3 rounded text-sm text-blue-700 dark:text-blue-300 flex items-center gap-2">
                                        <Check className="h-4 w-4" />
                                        This booking has been processed. Status: <span className="font-bold">{viewBooking.booking_status}</span>
                                    </div>
                                )}

                                <div className="space-y-3 pt-2">
                                    <div className="space-y-2">
                                        <Label htmlFor="status_update">Update Status Manually</Label>
                                        <Select 
                                            value={statusForm.data.booking_status} 
                                            onValueChange={(val) => statusForm.setData('booking_status', val)}
                                        >
                                            <SelectTrigger>
                                                <SelectValue placeholder="Select status" />
                                            </SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="pending">Pending</SelectItem>
                                                <SelectItem value="confirmed">Confirmed</SelectItem>
                                                <SelectItem value="rejected">Rejected</SelectItem>
                                                <SelectItem value="cancelled">Cancelled</SelectItem>
                                                <SelectItem value="completed">Completed</SelectItem>
                                            </SelectContent>
                                        </Select>
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="notes">Admin Notes</Label>
                                        <Textarea
                                            id="notes"
                                            value={statusForm.data.admin_notes}
                                            onChange={(e) => statusForm.setData('admin_notes', e.target.value)}
                                            placeholder="Add notes..."
                                            rows={2}
                                        />
                                    </div>
                                </div>
                            </div>
                        )}
                        <DialogFooter>
                            <Button variant="outline" onClick={() => setIsBookingModalOpen(false)}>Close</Button>
                            <Button onClick={(e) => handleStatusUpdate(e)} disabled={statusForm.processing}>Save Changes</Button>
                        </DialogFooter>
                    </DialogContent>
                </Dialog>
            </div>
        </SocietyAdminLayout>
    );
}

// Calendar View Component
function CalendarView({ facility, society, onBookingClick }: { 
    facility: Facility & { bookings?: FacilityBooking[] }, 
    society: { id: number },
    onBookingClick?: (booking: FacilityBooking) => void 
}) {
    const [currentDate, setCurrentDate] = useState(new Date());

    const daysInMonth = useMemo(() => {
        const year = currentDate.getFullYear();
        const month = currentDate.getMonth();
        const firstDay = new Date(year, month, 1);
        const lastDay = new Date(year, month + 1, 0);
        const daysCount = lastDay.getDate();
        const startingDayOfWeek = firstDay.getDay();

        return { daysCount, startingDayOfWeek, year, month };
    }, [currentDate]);

    const bookingsByDate = useMemo(() => {
        const map: Record<string, FacilityBooking[]> = {};
        facility.bookings?.forEach((booking) => {
            // Normalize date to YYYY-MM-DD format
            const bookingDate = new Date(booking.booking_date);
            const year = bookingDate.getFullYear();
            const month = String(bookingDate.getMonth() + 1).padStart(2, '0');
            const day = String(bookingDate.getDate()).padStart(2, '0');
            const dateString = `${year}-${month}-${day}`;
            
            if (!map[dateString]) map[dateString] = [];
            map[dateString].push(booking);
        });
        return map;
    }, [facility.bookings]);

    const previousMonth = () => {
        setCurrentDate(new Date(currentDate.getFullYear(), currentDate.getMonth() - 1, 1));
    };

    const nextMonth = () => {
        setCurrentDate(new Date(currentDate.getFullYear(), currentDate.getMonth() + 1, 1));
    };

    const getDateString = (day: number) => {
        const year = daysInMonth.year;
        const month = String(daysInMonth.month + 1).padStart(2, '0');
        const dayStr = String(day).padStart(2, '0');
        return `${year}-${month}-${dayStr}`;
    };

    const monthNames = ["January", "February", "March", "April", "May", "June",
        "July", "August", "September", "October", "November", "December"];
    
    const weekDays = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];

    return (
        <Card>
            <CardHeader>
                <div className="flex items-center justify-between">
                    <div>
                        <CardTitle>Booking Calendar</CardTitle>
                        <CardDescription>View bookings by date</CardDescription>
                    </div>
                    <div className="flex items-center gap-2">
                        <Button variant="outline" size="icon" onClick={previousMonth}>
                            <ChevronLeft className="h-4 w-4" />
                        </Button>
                        <div className="text-lg font-semibold min-w-[180px] text-center">
                            {monthNames[daysInMonth.month]} {daysInMonth.year}
                        </div>
                        <Button variant="outline" size="icon" onClick={nextMonth}>
                            <ChevronRight className="h-4 w-4" />
                        </Button>
                    </div>
                </div>
            </CardHeader>
            <CardContent>
                <div className="grid grid-cols-7 gap-3">
                    {/* Week day headers */}
                    {weekDays.map((day) => (
                        <div key={day} className="text-center text-sm font-bold text-muted-foreground py-3">
                            {day}
                        </div>
                    ))}
                    
                    {/* Empty cells for days before month starts */}
                    {Array.from({ length: daysInMonth.startingDayOfWeek }).map((_, index) => (
                        <div key={`empty-${index}`} className="min-h-[120px]" />
                    ))}
                    
                    {/* Days of the month */}
                    {Array.from({ length: daysInMonth.daysCount }).map((_, index) => {
                        const day = index + 1;
                        const dateString = getDateString(day);
                        const dayBookings = bookingsByDate[dateString] || [];
                        const isToday = new Date().toDateString() === new Date(daysInMonth.year, daysInMonth.month, day).toDateString();
                        
                        const confirmedCount = dayBookings.filter(b => b.booking_status === 'confirmed').length;
                        const pendingCount = dayBookings.filter(b => b.booking_status === 'pending').length;
                        const totalBookings = dayBookings.length;

                        return (
                            <div
                                key={day}
                                className={`min-h-[120px] border rounded-lg p-3 hover:bg-muted/50 transition-colors ${
                                    isToday ? 'border-primary border-2 bg-primary/5' : 'border-border'
                                }`}
                            >
                                <div className={`text-lg font-bold mb-2 ${isToday ? 'text-primary' : 'text-foreground'}`}>
                                    {day}
                                </div>
                                {totalBookings > 0 && (
                                    <div className="space-y-1 mt-2">
                                        {dayBookings.map((booking) => (
                                            <div 
                                                key={booking.id}
                                                className={`text-xs px-2 py-1.5 rounded border border-l-2 truncate cursor-pointer transition-transform hover:scale-105 ${
                                                    booking.booking_status === 'confirmed' 
                                                        ? 'bg-green-50 border-green-200 border-l-green-500 text-green-700 dark:bg-green-900/20 dark:border-green-800' 
                                                        : booking.booking_status === 'pending'
                                                        ? 'bg-yellow-50 border-yellow-200 border-l-yellow-500 text-yellow-700 dark:bg-yellow-900/20 dark:border-yellow-800'
                                                        : 'bg-muted border-muted-foreground/20 border-l-muted-foreground'
                                                }`}
                                                title={`${booking.slot?.name} - ${booking.user?.name} (${booking.booking_status})`}
                                                onClick={(e) => {
                                                    e.stopPropagation();
                                                    onBookingClick?.(booking);
                                                }}
                                            >
                                                <div className="font-semibold">{booking.slot?.name}</div>
                                                <div className="opacity-80 text-[10px]">{booking.user?.name}</div>
                                            </div>
                                        ))}
                                    </div>
                                )}
                            </div>
                        );
                    })}
                </div>

                {/* Legend */}
                <div className="flex items-center gap-6 mt-6 pt-4 border-t">
                    <div className="flex items-center gap-2">
                        <div className="w-4 h-4 rounded border-2 border-primary"></div>
                        <span className="text-sm text-muted-foreground">Today</span>
                    </div>
                    <div className="flex items-center gap-2">
                        <div className="w-4 h-4 rounded bg-green-100 dark:bg-green-900/30"></div>
                        <span className="text-sm text-muted-foreground">Confirmed</span>
                    </div>
                    <div className="flex items-center gap-2">
                        <div className="w-4 h-4 rounded bg-yellow-100 dark:bg-yellow-900/30"></div>
                        <span className="text-sm text-muted-foreground">Pending</span>
                    </div>
                </div>
            </CardContent>
        </Card>
    );
}
