import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { type Facility } from '@/types/facility';
import { Head, Link, router } from '@inertiajs/react';
import { Building2, Plus, Search, Eye, Edit, Trash2, CalendarDays, Users, IndianRupee, Clock } from 'lucide-react';
import { useState } from 'react';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Facilities',
        href: '/societies/{society}/facilities',
    },
];

interface IndexProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    facilities: {
        data: Facility[];
        links: any;
        meta: any;
    };
    filters?: {
        search?: string;
    };
}

export default function Index({ society, facilities, filters }: IndexProps) {
    const [search, setSearch] = useState(filters?.search || '');

    const handleSearch = () => {
        router.get(`/societies/${society.id}/facilities`, { search }, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const handleDelete = (id: number) => {
        if (confirm('Are you sure you want to delete this facility? This action cannot be undone if there are no active bookings.')) {
            router.delete(`/societies/${society.id}/facilities/${id}`, {
                preserveScroll: true,
            });
        }
    };

    // Calculate stats from facilities
    const stats = {
        total_facilities: facilities.data.length,
        total_bookings: facilities.data.reduce((sum, f) => sum + (f.bookings_count || 0), 0),
        active_facilities: facilities.data.filter(f => f.is_active).length,
        total_slots: facilities.data.reduce((sum, f) => sum + (f.slots?.length || 0), 0),
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Facilities" />
            <div className="flex flex-1 flex-col gap-8 p-6 md:p-8 overflow-x-hidden bg-muted/10">
                <div className="flex items-center justify-between">
                    <div>
                        <h1 className="text-3xl font-bold tracking-tight">Facilities Management</h1>
                        <p className="text-muted-foreground mt-2">
                            Manage community halls, clubhouses, and bookable spaces
                        </p>
                    </div>
                    
                    <Link href={`/societies/${society.id}/facilities/create`}>
                        <Button className="bg-primary hover:bg-primary/90 shadow-lg shadow-primary/20">
                            <Plus className="mr-2 h-4 w-4" />
                            Add Facility
                        </Button>
                    </Link>
                </div>

                {/* KPI Cards */}
                <div className="grid gap-4 md:grid-cols-4">
                    <Card className="border-t-4 border-t-blue-500 shadow-sm hover:shadow-lg transition-all duration-300">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Total Facilities</CardTitle>
                            <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-blue-400 to-blue-600 flex items-center justify-center text-white shadow-md shadow-blue-500/20">
                                <Building2 className="h-5 w-5" />
                            </div>
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-foreground">{stats.total_facilities}</div>
                            <p className="text-xs text-muted-foreground mt-1">
                                {stats.active_facilities} active
                            </p>
                        </CardContent>
                    </Card>

                    <Card className="border-t-4 border-t-green-500 shadow-sm hover:shadow-lg transition-all duration-300">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Total Bookings</CardTitle>
                            <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-green-400 to-green-600 flex items-center justify-center text-white shadow-md shadow-green-500/20">
                                <CalendarDays className="h-5 w-5" />
                            </div>
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-foreground">{stats.total_bookings}</div>
                            <p className="text-xs text-muted-foreground mt-1">
                                All time bookings
                            </p>
                        </CardContent>
                    </Card>

                    <Card className="border-t-4 border-t-purple-500 shadow-sm hover:shadow-lg transition-all duration-300">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Time Slots</CardTitle>
                            <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-cyan-400 to-teal-600 flex items-center justify-center text-white shadow-md shadow-cyan-500/20">
                                <Clock className="h-5 w-5" />
                            </div>
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-foreground">{stats.total_slots}</div>
                            <p className="text-xs text-muted-foreground mt-1">
                                Available time slots
                            </p>
                        </CardContent>
                    </Card>

                    <Card className="border-t-4 border-t-orange-500 shadow-sm hover:shadow-lg transition-all duration-300">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Quick Actions</CardTitle>
                            <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-orange-400 to-orange-600 flex items-center justify-center text-white shadow-md shadow-orange-500/20">
                                <Users className="h-5 w-5" />
                            </div>
                        </CardHeader>
                        <CardContent>
                            <Link href={`/societies/${society.id}/facility-bookings`}>
                                <Button variant="outline" size="sm" className="w-full text-xs">
                                    View All Bookings
                                </Button>
                            </Link>
                        </CardContent>
                    </Card>
                </div>

                {/* Filters */}
                <Card className="shadow-sm border-none ring-1 ring-border/50">
                    <CardContent className="pt-6">
                        <div className="flex gap-4">
                            <div className="flex-1">
                                <Label htmlFor="search">Search Facilities</Label>
                                <div className="relative mt-1">
                                    <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                    <Input
                                        id="search"
                                        placeholder="Search by facility name..."
                                        value={search}
                                        onChange={(e) => setSearch(e.target.value)}
                                        onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                        className="pl-9"
                                    />
                                </div>
                            </div>
                            <div className="flex items-end">
                                <Button onClick={handleSearch}>Filter</Button>
                            </div>
                        </div>
                    </CardContent>
                </Card>

                {/* Facilities Table */}
                <Card className="shadow-md border-none ring-1 ring-border/50">
                    <CardHeader className="bg-muted/30 border-b border-border/50">
                        <CardTitle className="text-lg font-bold">All Facilities</CardTitle>
                    </CardHeader>
                    <CardContent className="p-0">
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead>
                                    <tr className="border-b bg-muted/20">
                                        <th className="px-6 py-4 text-left text-xs font-semibold text-muted-foreground uppercase tracking-wider">Facility</th>
                                        <th className="px-6 py-4 text-left text-xs font-semibold text-muted-foreground uppercase tracking-wider">Capacity</th>
                                        <th className="px-6 py-4 text-left text-xs font-semibold text-muted-foreground uppercase tracking-wider">Time Slots</th>
                                        <th className="px-6 py-4 text-left text-xs font-semibold text-muted-foreground uppercase tracking-wider">Bookings</th>
                                        <th className="px-6 py-4 text-left text-xs font-semibold text-muted-foreground uppercase tracking-wider">Approval</th>
                                        <th className="px-6 py-4 text-left text-xs font-semibold text-muted-foreground uppercase tracking-wider">Status</th>
                                        <th className="px-6 py-4 text-right text-xs font-semibold text-muted-foreground uppercase tracking-wider">Actions</th>
                                    </tr>
                                </thead>
                                <tbody className="divide-y divide-border/50">
                                    {facilities.data.length > 0 ? (
                                        facilities.data.map((facility) => (
                                            <tr key={facility.id} className="hover:bg-muted/30 transition-colors">
                                                <td className="px-6 py-4">
                                                    <div className="flex items-center gap-3">
                                                        <div className="h-10 w-10 rounded-lg bg-blue-100 flex items-center justify-center text-blue-600">
                                                            <Building2 className="h-5 w-5" />
                                                        </div>
                                                        <div>
                                                            <div className="font-semibold text-foreground">{facility.name}</div>
                                                            {facility.description && (
                                                                <div className="text-xs text-muted-foreground truncate max-w-[200px]">
                                                                    {facility.description}
                                                                </div>
                                                            )}
                                                        </div>
                                                    </div>
                                                </td>
                                                <td className="px-6 py-4 text-sm text-muted-foreground">
                                                    {facility.capacity ? (
                                                        <div className="flex items-center gap-1">
                                                            <Users className="h-4 w-4" />
                                                            <span>{facility.capacity}</span>
                                                        </div>
                                                    ) : '-'}
                                                </td>
                                                <td className="px-6 py-4">
                                                    <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-cyan-100 text-cyan-800 dark:bg-cyan-900/30 dark:text-cyan-400">
                                                        {facility.slots?.length || 0} Slots
                                                    </span>
                                                </td>
                                                <td className="px-6 py-4">
                                                    <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400">
                                                        {facility.bookings_count || 0} Bookings
                                                    </span>
                                                </td>
                                                <td className="px-6 py-4 text-sm">
                                                    {facility.requires_approval ? (
                                                        <span className="text-orange-600 font-medium">Required</span>
                                                    ) : (
                                                        <span className="text-muted-foreground">Not Required</span>
                                                    )}
                                                </td>
                                                <td className="px-6 py-4">
                                                    {facility.is_active ? (
                                                        <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400">
                                                            Active
                                                        </span>
                                                    ) : (
                                                        <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800 dark:bg-gray-900/30 dark:text-gray-400">
                                                            Inactive
                                                        </span>
                                                    )}
                                                </td>
                                                <td className="px-6 py-4">
                                                    <div className="flex items-center justify-end gap-2">
                                                        <Link href={`/societies/${society.id}/facilities/${facility.id}`}>
                                                            <Button 
                                                                variant="ghost" 
                                                                size="icon" 
                                                                className="h-8 w-8 hover:bg-blue-50 hover:text-blue-600"
                                                            >
                                                                <Eye className="h-4 w-4" />
                                                            </Button>
                                                        </Link>
                                                        <Link href={`/societies/${society.id}/facilities/${facility.id}/edit`}>
                                                            <Button 
                                                                variant="ghost" 
                                                                size="icon" 
                                                                className="h-8 w-8 hover:bg-amber-50 hover:text-amber-600"
                                                            >
                                                                <Edit className="h-4 w-4" />
                                                            </Button>
                                                        </Link>
                                                        <Button
                                                            variant="ghost"
                                                            size="icon"
                                                            className="h-8 w-8 hover:bg-red-50 hover:text-red-600"
                                                            onClick={() => handleDelete(facility.id)}
                                                        >
                                                            <Trash2 className="h-4 w-4" />
                                                        </Button>
                                                    </div>
                                                </td>
                                            </tr>
                                        ))
                                    ) : (
                                        <tr>
                                            <td colSpan={7} className="px-6 py-12 text-center text-muted-foreground">
                                                <div className="flex flex-col items-center gap-2">
                                                    <Building2 className="h-10 w-10 text-muted-foreground/30" />
                                                    <p>No facilities found</p>
                                                    <Link href={`/societies/${society.id}/facilities/create`}>
                                                        <Button variant="outline" size="sm" className="mt-2">
                                                            <Plus className="mr-2 h-4 w-4" />
                                                            Add Your First Facility
                                                        </Button>
                                                    </Link>
                                                </div>
                                            </td>
                                        </tr>
                                    )}
                                </tbody>
                            </table>
                        </div>

                        {/* Pagination */}
                        {facilities.links && facilities.links.length > 3 && (
                            <div className="border-t border-border/50 px-6 py-4 flex items-center justify-between bg-muted/10">
                                <div className="text-sm text-muted-foreground">
                                    Showing <span className="font-medium text-foreground">{facilities.meta.from}</span> to <span className="font-medium text-foreground">{facilities.meta.to}</span> of <span className="font-medium text-foreground">{facilities.meta.total}</span> results
                                </div>
                                <div className="flex gap-1">
                                    {facilities.links.map((link: any, index: number) => (
                                        <Link
                                            key={index}
                                            href={link.url || '#'}
                                            className={`px-3 py-1 rounded-md text-sm transition-colors ${
                                                link.active
                                                    ? 'bg-primary text-primary-foreground font-medium shadow-sm'
                                                    : 'bg-background hover:bg-muted text-muted-foreground border border-border/50'
                                            } ${!link.url && 'opacity-50 cursor-not-allowed'}`}
                                            dangerouslySetInnerHTML={{ __html: link.label }}
                                        />
                                    ))}
                                </div>
                            </div>
                        )}
                    </CardContent>
                </Card>
            </div>
        </SocietyAdminLayout>
    );
}
