import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Checkbox } from '@/components/ui/checkbox';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link, useForm } from '@inertiajs/react';
import { ArrowLeft, Building2, Save } from 'lucide-react';

interface CreateProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    errors?: Record<string, string>;
}

export default function Create({ society, errors }: CreateProps) {
    const breadcrumbs: BreadcrumbItem[] = [
        {
            title: 'Facilities',
            href: `/societies/${society.id}/facilities`,
        },
        {
            title: 'Create',
            href: `/societies/${society.id}/facilities/create`,
        },
    ];

    const { data, setData, post, processing } = useForm({
        name: '',
        description: '',
        images: '',
        capacity: '',
        amenities: '',
        rules: '',
        cancellation_policy: '',
        requires_approval: false,
        is_active: true,
    });

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        post(`/societies/${society.id}/facilities`);
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Create Facility" />
            <div className="flex flex-1 flex-col gap-6 p-6 md:p-8 overflow-x-hidden bg-muted/10">
                <div className="flex items-center justify-between">
                    <div>
                        <h1 className="text-3xl font-bold tracking-tight">Create New Facility</h1>
                        <p className="text-muted-foreground mt-2">
                            Add a new bookable facility to your society
                        </p>
                    </div>
                    
                    <Link href={`/societies/${society.id}/facilities`}>
                        <Button variant="outline">
                            <ArrowLeft className="mr-2 h-4 w-4" />
                            Back to List
                        </Button>
                    </Link>
                </div>

                <form onSubmit={handleSubmit} className="space-y-6">
                    <Card>
                        <CardHeader>
                            <CardTitle className="flex items-center gap-2">
                                <Building2 className="h-5 w-5" />
                                Basic Information
                            </CardTitle>
                            <CardDescription>
                                Enter the essential details about the facility
                            </CardDescription>
                        </CardHeader>
                        <CardContent className="space-y-6">
                            <div className="grid gap-6 md:grid-cols-2">
                                <div className="space-y-2">
                                    <Label htmlFor="name">Facility Name *</Label>
                                    <Input
                                        id="name"
                                        value={data.name}
                                        onChange={(e) => setData('name', e.target.value)}
                                        placeholder="e.g. Main Community Hall"
                                        required
                                    />
                                    <InputError message={errors?.name} />
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="capacity">Capacity (People)</Label>
                                    <Input
                                        id="capacity"
                                        type="number"
                                        min="0"
                                        value={data.capacity}
                                        onChange={(e) => setData('capacity', e.target.value)}
                                        placeholder="e.g. 150"
                                    />
                                    <InputError message={errors?.capacity} />
                                </div>
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="description">Description</Label>
                                <Textarea
                                    id="description"
                                    value={data.description}
                                    onChange={(e) => setData('description', e.target.value)}
                                    placeholder="Brief description of the facility and its features..."
                                    rows={3}
                                />
                                <InputError message={errors?.description} />
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="images">Image URLs</Label>
                                <Textarea
                                    id="images"
                                    value={data.images}
                                    onChange={(e) => setData('images', e.target.value)}
                                    placeholder="Enter image URLs separated by commas&#10;https://example.com/hall1.jpg, https://example.com/hall2.jpg"
                                    rows={3}
                                />
                                <p className="text-xs text-muted-foreground">
                                    Enter multiple image URLs separated by commas
                                </p>
                                <InputError message={errors?.images} />
                            </div>
                        </CardContent>
                    </Card>

                    <Card>
                        <CardHeader>
                            <CardTitle>Facilities & Rules</CardTitle>
                            <CardDescription>
                                Specify what's included and the usage guidelines
                            </CardDescription>
                        </CardHeader>
                        <CardContent className="space-y-6">
                            <div className="space-y-2">
                                <Label htmlFor="amenities">Amenities</Label>
                                <Textarea
                                    id="amenities"
                                    value={data.amenities}
                                    onChange={(e) => setData('amenities', e.target.value)}
                                    placeholder="e.g. Air Conditioning, Sound System, Projector, Stage, Kitchen&#10;List all available amenities..."
                                    rows={3}
                                />
                                <p className="text-xs text-muted-foreground">
                                    Describe the amenities and facilities available (AC, Sound System, Kitchen, etc.)
                                </p>
                                <InputError message={errors?.amenities} />
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="rules">Usage Rules</Label>
                                <Textarea
                                    id="rules"
                                    value={data.rules}
                                    onChange={(e) => setData('rules', e.target.value)}
                                    placeholder="e.g.&#10;- No smoking allowed&#10;- No outside catering&#10;- Music must end by 11 PM&#10;- Clean up after use"
                                    rows={4}
                                />
                                <p className="text-xs text-muted-foreground">
                                    Define the rules residents must follow when using this facility
                                </p>
                                <InputError message={errors?.rules} />
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="cancellation_policy">Cancellation Policy</Label>
                                <Textarea
                                    id="cancellation_policy"
                                    value={data.cancellation_policy}
                                    onChange={(e) => setData('cancellation_policy', e.target.value)}
                                    placeholder="e.g. Full refund if cancelled 7 days before booking&#10;50% refund if cancelled 3 days before&#10;No refund if cancelled within 24 hours"
                                    rows={3}
                                />
                                <p className="text-xs text-muted-foreground">
                                    Explain the refund policy for cancellations
                                </p>
                                <InputError message={errors?.cancellation_policy} />
                            </div>
                        </CardContent>
                    </Card>

                    <Card>
                        <CardHeader>
                            <CardTitle>Settings</CardTitle>
                            <CardDescription>
                                Configure booking approval and availability
                            </CardDescription>
                        </CardHeader>
                        <CardContent className="space-y-4">
                            <div className="flex items-center gap-3 p-4 border rounded-lg">
                                <Checkbox
                                    id="requires_approval"
                                    checked={data.requires_approval}
                                    onCheckedChange={(checked) => setData('requires_approval', !!checked)}
                                />
                                <div className="flex-1">
                                    <Label htmlFor="requires_approval" className="cursor-pointer font-medium">
                                        Require Admin Approval
                                    </Label>
                                    <p className="text-sm text-muted-foreground mt-1">
                                        All bookings will require admin approval before confirmation
                                    </p>
                                </div>
                            </div>

                            <div className="flex items-center gap-3 p-4 border rounded-lg">
                                <Checkbox
                                    id="is_active"
                                    checked={data.is_active}
                                    onCheckedChange={(checked) => setData('is_active', !!checked)}
                                />
                                <div className="flex-1">
                                    <Label htmlFor="is_active" className="cursor-pointer font-medium">
                                        Active Status
                                    </Label>
                                    <p className="text-sm text-muted-foreground mt-1">
                                        Make this facility available for bookings immediately
                                    </p>
                                </div>
                            </div>
                        </CardContent>
                    </Card>

                    <div className="flex items-center justify-end gap-4 pb-6">
                        <Link href={`/societies/${society.id}/facilities`}>
                            <Button type="button" variant="outline">
                                Cancel
                            </Button>
                        </Link>
                        <Button type="submit" disabled={processing}>
                            <Save className="mr-2 h-4 w-4" />
                            {processing ? 'Creating...' : 'Create Facility'}
                        </Button>
                    </div>
                </form>
            </div>
        </SocietyAdminLayout>
    );
}
