import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import {
    Dialog,
    DialogContent,
    DialogDescription,
    DialogFooter,
    DialogHeader,
    DialogTitle,
} from '@/components/ui/dialog';
import { Textarea } from '@/components/ui/textarea';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { type Facility, type FacilityBooking } from '@/types/facility';
import { Head, Link, router, useForm } from '@inertiajs/react';
import { Search, Calendar, IndianRupee, User, Filter, Eye, Check, X } from 'lucide-react';
import { useState } from 'react';

interface BookingsProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    bookings: {
        data: FacilityBooking[];
        links: any;
        meta: any;
    };
    facilities: Facility[];
    filters?: {
        facility_id?: string;
        booking_status?: string;
        payment_status?: string;
        from_date?: string;
        to_date?: string;
    };
}

export default function Bookings({ society, bookings, facilities, filters }: BookingsProps) {
    const breadcrumbs: BreadcrumbItem[] = [
        {
            title: 'Facilities',
            href: `/societies/${society.id}/facilities`,
        },
        {
            title: 'Bookings',
            href: `/societies/${society.id}/facility-bookings`,
        },
    ];

    const [facilityFilter, setFacilityFilter] = useState(filters?.facility_id || '');
    const [bookingStatusFilter, setBookingStatusFilter] = useState(filters?.booking_status || '');
    const [paymentStatusFilter, setPaymentStatusFilter] = useState(filters?.payment_status || '');
    const [fromDate, setFromDate] = useState(filters?.from_date || '');
    const [toDate, setToDate] = useState(filters?.to_date || '');
    
    const [viewBooking, setViewBooking] = useState<FacilityBooking | null>(null);
    const [isViewOpen, setIsViewOpen] = useState(false);
    const [isStatusUpdateOpen, setIsStatusUpdateOpen] = useState(false);
    const [updatingBooking, setUpdatingBooking] = useState<FacilityBooking | null>(null);

    const statusForm = useForm({
        booking_status: '',
        admin_notes: '',
    });

    const handleFilter = () => {
        router.get(`/societies/${society.id}/facility-bookings`, {
            facility_id: facilityFilter || undefined,
            booking_status: bookingStatusFilter || undefined,
            payment_status: paymentStatusFilter || undefined,
            from_date: fromDate || undefined,
            to_date: toDate || undefined,
        }, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const openViewDialog = (booking: FacilityBooking) => {
        setViewBooking(booking);
        setIsViewOpen(true);
    };

    const openStatusDialog = (booking: FacilityBooking) => {
        setUpdatingBooking(booking);
        statusForm.setData({
            booking_status: booking.booking_status,
            admin_notes: booking.admin_notes || '',
        });
        setIsStatusUpdateOpen(true);
    };

    const handleStatusUpdate = (e: React.FormEvent) => {
        e.preventDefault();
        if (!updatingBooking) return;

        router.post(`/societies/${society.id}/facility-bookings/${updatingBooking.id}/status`, statusForm.data, {
            onSuccess: () => {
                setIsStatusUpdateOpen(false);
                setUpdatingBooking(null);
                statusForm.reset();
            },
        });
    };

    const formatCurrency = (amount: string | number) => {
        return `₹${parseFloat(amount.toString()).toLocaleString('en-IN')}`;
    };

    const getStatusColor = (status: string) => {
        switch (status) {
            case 'confirmed':
                return 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400';
            case 'pending':
                return 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/30 dark:text-yellow-400';
            case 'cancelled':
            case 'rejected':
                return 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400';
            case 'completed':
                return 'bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400';
            default:
                return 'bg-gray-100 text-gray-800 dark:bg-gray-900/30 dark:text-gray-400';
        }
    };

    const getPaymentStatusColor = (status: string) => {
        switch (status) {
            case 'paid':
                return 'bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-400';
            case 'pending':
                return 'bg-orange-100 text-orange-800 dark:bg-orange-900/30 dark:text-orange-400';
            case 'failed':
                return 'bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-400';
            case 'refunded':
                return 'bg-purple-100 text-purple-800 dark:bg-purple-900/30 dark:text-purple-400';
            default:
                return 'bg-gray-100 text-gray-800 dark:bg-gray-900/30 dark:text-gray-400';
        }
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Facility Bookings" />
            <div className="flex flex-1 flex-col gap-6 p-6 md:p-8 overflow-x-hidden bg-muted/10">
                <div className="flex items-center justify-between">
                    <div>
                        <h1 className="text-3xl font-bold tracking-tight">Facility Bookings</h1>
                        <p className="text-muted-foreground mt-2">
                            View and manage all facility bookings
                        </p>
                    </div>
                </div>

                {/* Filters */}
                <Card className="shadow-sm border-none ring-1 ring-border/50">
                    <CardHeader>
                        <CardTitle className="text-base flex items-center gap-2">
                            <Filter className="h-4 w-4" />
                            Filter Bookings
                        </CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="grid gap-4 md:grid-cols-5">
                            <div className="space-y-2">
                                <Label htmlFor="facility">Facility</Label>
                                <Select value={facilityFilter} onValueChange={setFacilityFilter}>
                                    <SelectTrigger id="facility">
                                        <SelectValue placeholder="All Facilities" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="">All Facilities</SelectItem>
                                        {facilities.map((facility) => (
                                            <SelectItem key={facility.id} value={facility.id.toString()}>
                                                {facility.name}
                                            </SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="booking_status">Booking Status</Label>
                                <Select value={bookingStatusFilter} onValueChange={setBookingStatusFilter}>
                                    <SelectTrigger id="booking_status">
                                        <SelectValue placeholder="All Statuses" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="">All Statuses</SelectItem>
                                        <SelectItem value="pending">Pending</SelectItem>
                                        <SelectItem value="confirmed">Confirmed</SelectItem>
                                        <SelectItem value="cancelled">Cancelled</SelectItem>
                                        <SelectItem value="rejected">Rejected</SelectItem>
                                        <SelectItem value="completed">Completed</SelectItem>
                                    </SelectContent>
                                </Select>
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="payment_status">Payment Status</Label>
                                <Select value={paymentStatusFilter} onValueChange={setPaymentStatusFilter}>
                                    <SelectTrigger id="payment_status">
                                        <SelectValue placeholder="All Payments" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="">All Payments</SelectItem>
                                        <SelectItem value="pending">Pending</SelectItem>
                                        <SelectItem value="paid">Paid</SelectItem>
                                        <SelectItem value="failed">Failed</SelectItem>
                                        <SelectItem value="refunded">Refunded</SelectItem>
                                    </SelectContent>
                                </Select>
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="from_date">From Date</Label>
                                <Input
                                    id="from_date"
                                    type="date"
                                    value={fromDate}
                                    onChange={(e) => setFromDate(e.target.value)}
                                />
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="to_date">To Date</Label>
                                <Input
                                    id="to_date"
                                    type="date"
                                    value={toDate}
                                    onChange={(e) => setToDate(e.target.value)}
                                />
                            </div>
                        </div>
                        <div className="flex justify-end mt-4">
                            <Button onClick={handleFilter}>
                                <Search className="mr-2 h-4 w-4" />
                                Apply Filters
                            </Button>
                        </div>
                    </CardContent>
                </Card>

                {/* Bookings Table */}
                <Card className="shadow-md border-none ring-1 ring-border/50">
                    <CardHeader className="bg-muted/30 border-b border-border/50">
                        <CardTitle className="text-lg font-bold">All Bookings</CardTitle>
                    </CardHeader>
                    <CardContent className="p-0">
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead>
                                    <tr className="border-b bg-muted/20">
                                        <th className="px-6 py-4 text-left text-xs font-semibold text-muted-foreground uppercase tracking-wider">Booking ID</th>
                                        <th className="px-6 py-4 text-left text-xs font-semibold text-muted-foreground uppercase tracking-wider">Resident</th>
                                        <th className="px-6 py-4 text-left text-xs font-semibold text-muted-foreground uppercase tracking-wider">Facility</th>
                                        <th className="px-6 py-4 text-left text-xs font-semibold text-muted-foreground uppercase tracking-wider">Date & Slot</th>
                                        <th className="px-6 py-4 text-left text-xs font-semibold text-muted-foreground uppercase tracking-wider">Amount</th>
                                        <th className="px-6 py-4 text-left text-xs font-semibold text-muted-foreground uppercase tracking-wider">Booked By</th>
                                        <th className="px-6 py-4 text-left text-xs font-semibold text-muted-foreground uppercase tracking-wider">Status</th>
                                        <th className="px-6 py-4 text-left text-xs font-semibold text-muted-foreground uppercase tracking-wider">Payment</th>
                                        <th className="px-6 py-4 text-right text-xs font-semibold text-muted-foreground uppercase tracking-wider">Actions</th>
                                    </tr>
                                </thead>
                                <tbody className="divide-y divide-border/50">
                                    {bookings.data.length > 0 ? (
                                        bookings.data.map((booking) => (
                                            <tr key={booking.id} className="hover:bg-muted/30 transition-colors">
                                                <td className="px-6 py-4 text-sm font-mono">#{booking.id}</td>
                                                <td className="px-6 py-4">
                                                    <div>
                                                        <div className="font-semibold">{booking.user?.name}</div>
                                                        <div className="text-xs text-muted-foreground">
                                                            {booking.user?.unit?.unit_no} - {booking.user?.unit?.building.name}
                                                        </div>
                                                    </div>
                                                </td>
                                                <td className="px-6 py-4 text-sm">{booking.facility?.name}</td>
                                                <td className="px-6 py-4">
                                                    <div className="text-sm">
                                                        <div className="font-medium">{booking.booking_date}</div>
                                                        <div className="text-xs text-muted-foreground">{booking.slot?.name}</div>
                                                    </div>
                                                </td>
                                                <td className="px-6 py-4 text-sm font-medium">{formatCurrency(booking.amount)}</td>
                                                <td className="px-6 py-4 text-sm">
                                                    {booking.bookedBy ? (
                                                        <div>
                                                            <div className="font-medium">{booking.bookedBy.name}</div>
                                                            <div className="text-xs text-muted-foreground capitalize">
                                                                ({booking.bookedBy.role?.name})
                                                            </div>
                                                        </div>
                                                    ) : (
                                                        <span className="text-muted-foreground">Self</span>
                                                    )}
                                                </td>
                                                <td className="px-6 py-4">
                                                    <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${getStatusColor(booking.booking_status)}`}>
                                                        {booking.booking_status}
                                                    </span>
                                                </td>
                                                <td className="px-6 py-4">
                                                    <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${getPaymentStatusColor(booking.payment_status)}`}>
                                                        {booking.payment_status}
                                                    </span>
                                                </td>
                                                <td className="px-6 py-4">
                                                    <div className="flex items-center justify-end gap-2">
                                                        <Button 
                                                            variant="ghost" 
                                                            size="icon" 
                                                            className="h-8 w-8 hover:bg-blue-50 hover:text-blue-600"
                                                            onClick={() => openViewDialog(booking)}
                                                        >
                                                            <Eye className="h-4 w-4" />
                                                        </Button>
                                                        {booking.booking_status === 'pending' && (
                                                            <Button 
                                                                variant="ghost" 
                                                                size="icon" 
                                                                className="h-8 w-8 hover:bg-green-50 hover:text-green-600"
                                                                onClick={() => openStatusDialog(booking)}
                                                            >
                                                                <Check className="h-4 w-4" />
                                                            </Button>
                                                        )}
                                                    </div>
                                                </td>
                                            </tr>
                                        ))
                                    ) : (
                                        <tr>
                                            <td colSpan={9} className="px-6 py-12 text-center text-muted-foreground">
                                                <div className="flex flex-col items-center gap-2">
                                                    <Calendar className="h-10 w-10 text-muted-foreground/30" />
                                                    <p>No bookings found</p>
                                                </div>
                                            </td>
                                        </tr>
                                    )}
                                </tbody>
                            </table>
                        </div>

                        {/* Pagination */}
                        {bookings.links && bookings.links.length > 3 && (
                            <div className="border-t border-border/50 px-6 py-4 flex items-center justify-between bg-muted/10">
                                <div className="text-sm text-muted-foreground">
                                    Showing <span className="font-medium text-foreground">{bookings.meta.from}</span> to <span className="font-medium text-foreground">{bookings.meta.to}</span> of <span className="font-medium text-foreground">{bookings.meta.total}</span> results
                                </div>
                                <div className="flex gap-1">
                                    {bookings.links.map((link: any, index: number) => (
                                        <Link
                                            key={index}
                                            href={link.url || '#'}
                                            className={`px-3 py-1 rounded-md text-sm transition-colors ${
                                                link.active
                                                    ? 'bg-primary text-primary-foreground font-medium shadow-sm'
                                                    : 'bg-background hover:bg-muted text-muted-foreground border border-border/50'
                                            } ${!link.url && 'opacity-50 cursor-not-allowed'}`}
                                            dangerouslySetInnerHTML={{ __html: link.label }}
                                        />
                                    ))}
                                </div>
                            </div>
                        )}
                    </CardContent>
                </Card>

                {/* View Booking Dialog */}
                <Dialog open={isViewOpen} onOpenChange={setIsViewOpen}>
                    <DialogContent className="max-w-2xl">
                        <DialogHeader>
                            <DialogTitle>Booking Details</DialogTitle>
                            <DialogDescription>
                                View complete information about this booking
                            </DialogDescription>
                        </DialogHeader>
                        {viewBooking && (
                            <div className="space-y-6 py-4">
                                <div className="grid gap-4 md:grid-cols-2">
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground mb-1">Booking ID</h4>
                                        <p className="text-base font-mono">#{viewBooking.id}</p>
                                    </div>
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground mb-1">Booking Date</h4>
                                        <p className="text-base">{viewBooking.booking_date}</p>
                                    </div>
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground mb-1">Facility</h4>
                                        <p className="text-base font-semibold">{viewBooking.facility?.name}</p>
                                    </div>
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground mb-1">Time Slot</h4>
                                        <p className="text-base">{viewBooking.slot?.name}</p>
                                    </div>
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground mb-1">Resident</h4>
                                        <p className="text-base">{viewBooking.user?.name}</p>
                                        <p className="text-xs text-muted-foreground">{viewBooking.user?.unit?.unit_no}</p>
                                    </div>
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground mb-1">Booked By</h4>
                                        <p className="text-base">{viewBooking.bookedBy?.name || 'Self'}</p>
                                        {viewBooking.bookedBy && (
                                            <p className="text-xs text-muted-foreground capitalize">({viewBooking.bookedBy.role?.name})</p>
                                        )}
                                    </div>
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground mb-1">Amount</h4>
                                        <p className="text-xl font-bold">{formatCurrency(viewBooking.amount)}</p>
                                    </div>
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground mb-1">Payment Method</h4>
                                        <p className="text-base capitalize">{viewBooking.payment_method || 'Not specified'}</p>
                                    </div>
                                </div>
                                {viewBooking.purpose && (
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground mb-1">Purpose</h4>
                                        <p className="text-base">{viewBooking.purpose}</p>
                                    </div>
                                )}
                                {viewBooking.guest_count && (
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground mb-1">Guest Count</h4>
                                        <p className="text-base">{viewBooking.guest_count} guests</p>
                                    </div>
                                )}
                                {viewBooking.admin_notes && (
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground mb-1">Admin Notes</h4>
                                        <p className="text-sm whitespace-pre-line">{viewBooking.admin_notes}</p>
                                    </div>
                                )}
                            </div>
                        )}
                        <DialogFooter>
                            <Button onClick={() => setIsViewOpen(false)}>Close</Button>
                        </DialogFooter>
                    </DialogContent>
                </Dialog>

                {/* Update Status Dialog */}
                <Dialog open={isStatusUpdateOpen} onOpenChange={setIsStatusUpdateOpen}>
                    <DialogContent>
                        <DialogHeader>
                            <DialogTitle>Update Booking Status</DialogTitle>
                            <DialogDescription>
                                Approve, reject, or update the booking status
                            </DialogDescription>
                        </DialogHeader>
                        <form onSubmit={handleStatusUpdate} className="space-y-4">
                            <div className="space-y-2">
                                <Label htmlFor="status">Booking Status</Label>
                                <Select 
                                    value={statusForm.data.booking_status} 
                                    onValueChange={(value) => statusForm.setData('booking_status', value)}
                                >
                                    <SelectTrigger id="status">
                                        <SelectValue />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="pending">Pending</SelectItem>
                                        <SelectItem value="confirmed">Confirmed</SelectItem>
                                        <SelectItem value="rejected">Rejected</SelectItem>
                                        <SelectItem value="cancelled">Cancelled</SelectItem>
                                        <SelectItem value="completed">Completed</SelectItem>
                                    </SelectContent>
                                </Select>
                            </div>
                            <div className="space-y-2">
                                <Label htmlFor="admin_notes">Admin Notes (Optional)</Label>
                                <Textarea
                                    id="admin_notes"
                                    value={statusForm.data.admin_notes}
                                    onChange={(e) => statusForm.setData('admin_notes', e.target.value)}
                                    placeholder="Add any notes or reasons for status change..."
                                    rows={3}
                                />
                            </div>
                            <DialogFooter>
                                <Button type="button" variant="outline" onClick={() => setIsStatusUpdateOpen(false)}>
                                    Cancel
                                </Button>
                                <Button type="submit" disabled={statusForm.processing}>
                                    {statusForm.processing ? 'Updating...' : 'Update Status'}
                                </Button>
                            </DialogFooter>
                        </form>
                    </DialogContent>
                </Dialog>
            </div>
        </SocietyAdminLayout>
    );
}
