import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Textarea } from '@/components/ui/textarea';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, router, useForm } from '@inertiajs/react';
import { Receipt, Plus, Search, Edit, Trash2, Eye, TrendingUp, DollarSign, Calendar, FileText } from 'lucide-react';
import { useState } from 'react';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Expenses',
        href: '/societies/{society}/expenses',
    },
];

interface Vendor {
    id: number;
    name: string;
    phone: string;
}

interface Expense {
    id: number;
    expense_no: string;
    amount: number;
    category: string | null;
    description: string;
    vendor_name: string | null;
    expense_date: string;
    receipt_path: string | null;
    created_by: {
        id: number;
        name: string;
    };
    vendor: {
        id: number;
        name: string;
        phone: string;
    } | null;
    created_at: string;
}

interface IndexProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    expenses: {
        data: Expense[];
        links: any;
        meta: any;
    };
    categories: string[];
    vendors: Vendor[];
    bankAccounts: {
        id: number;
        account_name: string;
        bank_name: string;
        account_number: string;
    }[];
    filters: {
        search?: string;
        category?: string;
        date_from?: string;
        date_to?: string;
        vendor_id?: string;
    };
    stats: {
        total_expenses: number;
        monthly_expenses: number;
        total_count: number;
        monthly_count: number;
    };
}

export default function Index({ society, expenses, categories, vendors, bankAccounts, filters, stats }: IndexProps) {
    const [search, setSearch] = useState(filters.search || '');
    const [category, setCategory] = useState(filters.category || 'all');
    const [dateFrom, setDateFrom] = useState(filters.date_from || '');
    const [dateTo, setDateTo] = useState(filters.date_to || '');
    const [vendorId, setVendorId] = useState(filters.vendor_id || 'all');
    const [isCreateOpen, setIsCreateOpen] = useState(false);
    const [isEditOpen, setIsEditOpen] = useState(false);
    const [isDetailsOpen, setIsDetailsOpen] = useState(false);
    const [selectedExpense, setSelectedExpense] = useState<Expense | null>(null);
    const [useCustomCategory, setUseCustomCategory] = useState(false);
    const [customCategory, setCustomCategory] = useState('');
    const [vendorMode, setVendorMode] = useState<'select' | 'manual'>('select');

    const createForm = useForm({
        amount: '',
        gst_percentage: '',
        gst_amount: '',
        total_amount: '',
        category: '',
        description: '',
        vendor_id: '',
        vendor_name: '',
        expense_date: new Date().toISOString().split('T')[0],
        bank_account_id: '',
        receipt: null as File | null,
    });

    const editForm = useForm({
        amount: '',
        gst_percentage: '',
        gst_amount: '',
        total_amount: '',
        category: '',
        description: '',
        vendor_id: '',
        vendor_name: '',
        expense_date: '',
        bank_account_id: '',
        receipt: null as File | null,
    });

    const handleSearch = () => {
        const filterParams: any = {};
        if (search) filterParams.search = search;
        if (category && category !== 'all') filterParams.category = category;
        if (dateFrom) filterParams.date_from = dateFrom;
        if (dateTo) filterParams.date_to = dateTo;
        if (vendorId && vendorId !== 'all') filterParams.vendor_id = vendorId;

        router.get(`/societies/${society.id}/expenses`, filterParams, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const handleCreateSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        createForm.post(`/societies/${society.id}/expenses`, {
            onSuccess: () => {
                setIsCreateOpen(false);
                createForm.reset();
            },
        });
    };

    const handleEditSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        if (!selectedExpense) return;
        
        editForm.put(`/societies/${society.id}/expenses/${selectedExpense.id}`, {
            onSuccess: () => {
                setIsEditOpen(false);
                editForm.reset();
                setSelectedExpense(null);
            },
        });
    };

    const openEditModal = (expense: Expense) => {
        setSelectedExpense(expense);
        editForm.setData({
            amount: expense.amount.toString(),
            category: expense.category || '',
            description: expense.description,
            vendor_id: expense.vendor?.id.toString() || '',
            vendor_name: expense.vendor_name || '',
            expense_date: expense.expense_date,
            receipt: null,
        });
        setIsEditOpen(true);
    };

    const handleDelete = (id: number) => {
        if (confirm('Are you sure you want to delete this expense?')) {
            router.delete(`/societies/${society.id}/expenses/${id}`, {
                preserveScroll: true,
            });
        }
    };

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 0,
        }).format(amount);
    };

    const formatDate = (dateString: string) => {
        return new Date(dateString).toLocaleDateString('en-IN', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
        });
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Expenses" />
            <div className="flex h-full flex-1 flex-col gap-6 overflow-x-auto rounded-xl p-6">
                <div className="flex items-center justify-between">
                    <div>
                        <h1 className="text-2xl font-bold">Expenses</h1>
                        <p className="text-muted-foreground">
                            Track and manage all expenses in your society
                        </p>
                    </div>
                    <Dialog open={isCreateOpen} onOpenChange={setIsCreateOpen}>
                        <DialogTrigger asChild>
                            <Button>
                                <Plus className="mr-2 h-4 w-4" />
                                Add Expense
                            </Button>
                        </DialogTrigger>
                        <DialogContent className="max-w-2xl">
                            <DialogHeader>
                                <DialogTitle>Add New Expense</DialogTitle>
                                <DialogDescription>
                                    Record a new expense for your society
                                </DialogDescription>
                            </DialogHeader>
                            <form onSubmit={handleCreateSubmit} className="space-y-4">
                                <div className="grid grid-cols-2 gap-4">
                                    {/* Amount & GST Fields */}
                                    <div className="space-y-2">
                                        <Label htmlFor="amount">Base Amount *</Label>
                                        <Input
                                            id="amount"
                                            type="number"
                                            step="0.01"
                                            placeholder="0.00"
                                            value={createForm.data.amount}
                                            onChange={(e) => {
                                                const amount = parseFloat(e.target.value) || 0;
                                                const gstPercentage = parseFloat(createForm.data.gst_percentage) || 0;
                                                const gstAmount = (amount * gstPercentage) / 100;
                                                const totalAmount = amount + gstAmount;
                                                
                                                createForm.setData({
                                                    ...createForm.data,
                                                    amount: e.target.value,
                                                    gst_amount: gstAmount.toFixed(2),
                                                    total_amount: totalAmount.toFixed(2),
                                                });
                                            }}
                                            required
                                        />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="gst_percentage">GST %</Label>
                                        <Input
                                            id="gst_percentage"
                                            type="number"
                                            step="0.01"
                                            placeholder="0"
                                            value={createForm.data.gst_percentage}
                                            onChange={(e) => {
                                                const amount = parseFloat(createForm.data.amount) || 0;
                                                const gstPercentage = parseFloat(e.target.value) || 0;
                                                const gstAmount = (amount * gstPercentage) / 100;
                                                const totalAmount = amount + gstAmount;
                                                
                                                createForm.setData({
                                                    ...createForm.data,
                                                    gst_percentage: e.target.value,
                                                    gst_amount: gstAmount.toFixed(2),
                                                    total_amount: totalAmount.toFixed(2),
                                                });
                                            }}
                                        />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="gst_amount">GST Amount</Label>
                                        <Input
                                            id="gst_amount"
                                            type="number"
                                            step="0.01"
                                            value={createForm.data.gst_amount}
                                            readOnly
                                            className="bg-muted"
                                        />
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="total_amount">Total Amount</Label>
                                        <Input
                                            id="total_amount"
                                            type="number"
                                            step="0.01"
                                            value={createForm.data.total_amount}
                                            readOnly
                                            className="bg-muted font-medium"
                                        />
                                    </div>

                                    <div className="space-y-2">
                                        <Label htmlFor="expense_date">Date *</Label>
                                        <Input
                                            id="expense_date"
                                            type="date"
                                            value={createForm.data.expense_date}
                                            onChange={(e) => createForm.setData('expense_date', e.target.value)}
                                            required
                                        />
                                    </div>

                                    {/* Category - Custom or Select */}
                                    <div className="space-y-2">
                                        <div className="flex items-center justify-between">
                                            <Label htmlFor="category">Category</Label>
                                            <button
                                                type="button"
                                                onClick={() => setUseCustomCategory(!useCustomCategory)}
                                                className="text-xs text-primary hover:underline"
                                            >
                                                {useCustomCategory ? 'Select from list' : 'Add new category'}
                                            </button>
                                        </div>
                                        {useCustomCategory ? (
                                            <Input
                                                id="custom_category"
                                                placeholder="Enter new category"
                                                value={customCategory}
                                                onChange={(e) => {
                                                    setCustomCategory(e.target.value);
                                                    createForm.setData('category', e.target.value);
                                                }}
                                            />
                                        ) : (
                                            <Select
                                                value={createForm.data.category}
                                                onValueChange={(value) => createForm.setData('category', value)}
                                            >
                                                <SelectTrigger id="category">
                                                    <SelectValue placeholder="Select category" />
                                                </SelectTrigger>
                                                <SelectContent>
                                                    {categories.map((cat) => (
                                                        <SelectItem key={cat} value={cat}>
                                                            {cat}
                                                        </SelectItem>
                                                    ))}
                                                </SelectContent>
                                            </Select>
                                        )}
                                    </div>

                                    {/* Vendor - Select or Manual */}
                                    <div className="space-y-2 col-span-2">
                                        <div className="flex items-center justify-between">
                                            <Label>Vendor</Label>
                                            <div className="flex gap-2">
                                                <button
                                                    type="button"
                                                    onClick={() => setVendorMode('select')}
                                                    className={`text-xs ${vendorMode === 'select' ? 'text-primary font-medium' : 'text-muted-foreground'}`}
                                                >
                                                    Select Vendor
                                                </button>
                                                <span className="text-muted-foreground">|</span>
                                                <button
                                                    type="button"
                                                    onClick={() => setVendorMode('manual')}
                                                    className={`text-xs ${vendorMode === 'manual' ? 'text-primary font-medium' : 'text-muted-foreground'}`}
                                                >
                                                    Enter Manually
                                                </button>
                                            </div>
                                        </div>
                                        {vendorMode === 'select' ? (
                                            <Select
                                                value={createForm.data.vendor_id}
                                                onValueChange={(value) => createForm.setData('vendor_id', value)}
                                            >
                                                <SelectTrigger>
                                                    <SelectValue placeholder="Select vendor" />
                                                </SelectTrigger>
                                                <SelectContent>
                                                    {vendors.map((vendor) => (
                                                        <SelectItem key={vendor.id} value={vendor.id.toString()}>
                                                            {vendor.name} - {vendor.phone}
                                                        </SelectItem>
                                                    ))}
                                                </SelectContent>
                                            </Select>
                                        ) : (
                                            <Input
                                                placeholder="Enter vendor name"
                                                value={createForm.data.vendor_name}
                                                onChange={(e) => createForm.setData('vendor_name', e.target.value)}
                                            />
                                        )}
                                    </div>

                                    {/* Bank Account */}
                                    <div className="space-y-2 col-span-2">
                                        <Label htmlFor="bank_account_id">Paid From Bank Account (Optional)</Label>
                                        <Select
                                            value={createForm.data.bank_account_id}
                                            onValueChange={(value) => createForm.setData('bank_account_id', value === 'none' ? '' : value)}
                                        >
                                            <SelectTrigger id="bank_account_id">
                                                <SelectValue placeholder="Select bank account" />
                                            </SelectTrigger>
                                            <SelectContent>
                                                <SelectItem value="none">None</SelectItem>
                                                {bankAccounts.map((account) => (
                                                    <SelectItem key={account.id} value={account.id.toString()}>
                                                        {account.account_name} - {account.bank_name} ({account.account_number})
                                                    </SelectItem>
                                                ))}
                                            </SelectContent>
                                        </Select>
                                        <p className="text-xs text-muted-foreground">Select if paid from society bank account</p>
                                    </div>

                                    <div className="space-y-2 col-span-2">
                                        <Label htmlFor="description">Description *</Label>
                                        <Textarea
                                            id="description"
                                            placeholder="Enter expense description"
                                            value={createForm.data.description}
                                            onChange={(e) => createForm.setData('description', e.target.value)}
                                            required
                                        />
                                    </div>
                                    <div className="space-y-2 col-span-2">
                                        <Label htmlFor="receipt">Receipt / Invoice</Label>
                                        <Input
                                            id="receipt"
                                            type="file"
                                            accept="image/*,application/pdf"
                                            onChange={(e) => createForm.setData('receipt', e.target.files?.[0] || null)}
                                        />
                                        <p className="text-xs text-muted-foreground">Upload receipt or invoice (JPG, PNG, PDF)</p>
                                    </div>
                                </div>
                                <DialogFooter>
                                    <Button type="button" variant="outline" onClick={() => setIsCreateOpen(false)}>
                                        Cancel
                                    </Button>
                                    <Button type="submit" disabled={createForm.processing}>
                                        {createForm.processing ? 'Adding...' : 'Add Expense'}
                                    </Button>
                                </DialogFooter>
                            </form>
                        </DialogContent>
                    </Dialog>
                </div>

                {/* KPI Cards */}
                <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
                    <Card>
                        <CardContent className="p-6">
                            <div className="flex items-center justify-between">
                                <div className="space-y-1">
                                    <p className="text-sm text-muted-foreground">Total Expenses</p>
                                    <p className="text-2xl font-bold">{formatCurrency(stats?.total_expenses || 0)}</p>
                                </div>
                                <div className="rounded-full bg-primary/10 p-3">
                                    <DollarSign className="h-6 w-6 text-primary" />
                                </div>
                            </div>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardContent className="p-6">
                            <div className="flex items-center justify-between">
                                <div className="space-y-1">
                                    <p className="text-sm text-muted-foreground">This Month</p>
                                    <p className="text-2xl font-bold">{formatCurrency(stats?.monthly_expenses || 0)}</p>
                                </div>
                                <div className="rounded-full bg-green-100 dark:bg-green-950 p-3">
                                    <Calendar className="h-6 w-6 text-green-600 dark:text-green-400" />
                                </div>
                            </div>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardContent className="p-6">
                            <div className="flex items-center justify-between">
                                <div className="space-y-1">
                                    <p className="text-sm text-muted-foreground">Total Count</p>
                                    <p className="text-2xl font-bold">{stats?.total_count || 0}</p>
                                </div>
                                <div className="rounded-full bg-blue-100 dark:bg-blue-950 p-3">
                                    <FileText className="h-6 w-6 text-blue-600 dark:text-blue-400" />
                                </div>
                            </div>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardContent className="p-6">
                            <div className="flex items-center justify-between">
                                <div className="space-y-1">
                                    <p className="text-sm text-muted-foreground">Monthly Count</p>
                                    <p className="text-2xl font-bold">{stats?.monthly_count || 0}</p>
                                </div>
                                <div className="rounded-full bg-yellow-100 dark:bg-yellow-950 p-3">
                                    <TrendingUp className="h-6 w-6 text-yellow-600 dark:text-yellow-400" />
                                </div>
                            </div>
                        </CardContent>
                    </Card>
                </div>

                {/* Filters */}
                <Card>
                    <CardContent className="pt-6">
                        <div className="flex gap-4 flex-wrap">
                            <div className="flex-1 min-w-[200px]">
                                <Label htmlFor="search">Search</Label>
                                <div className="relative mt-1">
                                    <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                    <Input
                                        id="search"
                                        placeholder="Search by expense no, description, vendor..."
                                        value={search}
                                        onChange={(e) => setSearch(e.target.value)}
                                        onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                        className="pl-9"
                                    />
                                </div>
                            </div>
                            <div className="w-48">
                                <Label htmlFor="category">Category</Label>
                                <Select value={category} onValueChange={setCategory}>
                                    <SelectTrigger id="category" className="mt-1">
                                        <SelectValue placeholder="All Categories" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Categories</SelectItem>
                                        {categories.map((cat) => (
                                            <SelectItem key={cat} value={cat}>
                                                {cat}
                                            </SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                            </div>
                            <div className="w-48">
                                <Label htmlFor="date_from">Date From</Label>
                                <Input
                                    id="date_from"
                                    type="date"
                                    value={dateFrom}
                                    onChange={(e) => setDateFrom(e.target.value)}
                                    className="mt-1"
                                />
                            </div>
                            <div className="w-48">
                                <Label htmlFor="date_to">Date To</Label>
                                <Input
                                    id="date_to"
                                    type="date"
                                    value={dateTo}
                                    onChange={(e) => setDateTo(e.target.value)}
                                    className="mt-1"
                                />
                            </div>
                            <div className="w-48">
                                <Label htmlFor="vendor_id">Vendor</Label>
                                <Select value={vendorId} onValueChange={setVendorId}>
                                    <SelectTrigger id="vendor_id" className="mt-1">
                                        <SelectValue placeholder="All Vendors" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Vendors</SelectItem>
                                        {vendors.map((vendor) => (
                                            <SelectItem key={vendor.id} value={vendor.id.toString()}>
                                                {vendor.name}
                                            </SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                            </div>
                            <div className="flex items-end">
                                <Button onClick={handleSearch}>Filter</Button>
                            </div>
                        </div>
                    </CardContent>
                </Card>

                {/* Expenses Table */}
                <Card>
                    <CardHeader>
                        <CardTitle>All Expenses</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead>
                                    <tr className="border-b">
                                        <th className="px-4 py-3 text-left text-sm font-medium">Expense No</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Date</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Description</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Category</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Vendor</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Amount</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Created By</th>
                                        <th className="px-4 py-3 text-right text-sm font-medium">Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {expenses.data.length > 0 ? (
                                        expenses.data.map((expense) => (
                                            <tr key={expense.id} className="border-b hover:bg-muted/50">
                                                <td className="px-4 py-3">
                                                    <div className="flex items-center gap-2">
                                                        <Receipt className="h-4 w-4 text-muted-foreground" />
                                                        <span className="font-medium">{expense.expense_no}</span>
                                                    </div>
                                                </td>
                                                <td className="px-4 py-3 text-sm text-muted-foreground">
                                                    {formatDate(expense.expense_date)}
                                                </td>
                                                <td className="px-4 py-3 text-sm">
                                                    <div className="max-w-xs truncate" title={expense.description}>
                                                        {expense.description}
                                                    </div>
                                                </td>
                                                <td className="px-4 py-3 text-sm">
                                                    {expense.category || '-'}
                                                </td>
                                                <td className="px-4 py-3 text-sm">
                                                    {expense.vendor ? (
                                                        <div>
                                                            <div className="font-medium">{expense.vendor.name}</div>
                                                            <div className="text-xs text-muted-foreground">
                                                                {expense.vendor.phone}
                                                            </div>
                                                        </div>
                                                    ) : expense.vendor_name ? (
                                                        <span>{expense.vendor_name}</span>
                                                    ) : (
                                                        <span className="text-muted-foreground">-</span>
                                                    )}
                                                </td>
                                                <td className="px-4 py-3 text-sm font-medium">
                                                    {formatCurrency(expense.amount)}
                                                </td>
                                                <td className="px-4 py-3 text-sm">
                                                    {expense.created_by.name}
                                                </td>
                                                <td className="px-4 py-3">
                                                    <div className="flex items-center justify-end gap-2">
                                                        <Button
                                                            size="sm"
                                                            variant="ghost"
                                                            onClick={() => {
                                                                setSelectedExpense(expense);
                                                                setIsDetailsOpen(true);
                                                            }}
                                                        >
                                                            <Eye className="h-4 w-4" />
                                                        </Button>
                                                        <Button
                                                            variant="ghost"
                                                            size="sm"
                                                            onClick={() => openEditModal(expense)}
                                                        >
                                                            <Edit className="h-4 w-4" />
                                                        </Button>
                                                        <Button
                                                            variant="ghost"
                                                            size="sm"
                                                            onClick={() => handleDelete(expense.id)}
                                                        >
                                                            <Trash2 className="h-4 w-4 text-destructive" />
                                                        </Button>
                                                    </div>
                                                </td>
                                            </tr>
                                        ))
                                    ) : (
                                        <tr>
                                            <td colSpan={8} className="px-4 py-8 text-center text-muted-foreground">
                                                No expenses found
                                            </td>
                                        </tr>
                                    )}
                                </tbody>
                            </table>
                        </div>

                        {/* Pagination */}
                        {expenses.links && expenses.links.length > 3 && (
                            <div className="mt-4 flex items-center justify-between">
                                <div className="text-sm text-muted-foreground">
                                    Showing {expenses.meta.from} to {expenses.meta.to} of {expenses.meta.total} results
                                </div>
                                <div className="flex gap-2">
                                    {expenses.links.map((link: any, index: number) => (
                                        <button
                                            key={index}
                                            onClick={() => link.url && router.visit(link.url)}
                                            disabled={!link.url}
                                            className={`px-3 py-1 rounded ${
                                                link.active
                                                    ? 'bg-primary text-primary-foreground'
                                                    : 'bg-muted hover:bg-muted/80'
                                            } ${!link.url ? 'opacity-50 cursor-not-allowed' : ''}`}
                                            dangerouslySetInnerHTML={{ __html: link.label }}
                                        />
                                    ))}
                                </div>
                            </div>
                        )}
                    </CardContent>
                </Card>

                {/* Edit Modal */}
                <Dialog open={isEditOpen} onOpenChange={setIsEditOpen}>
                    <DialogContent className="max-w-2xl">
                        <DialogHeader>
                            <DialogTitle>Edit Expense</DialogTitle>
                            <DialogDescription>
                                Update expense information
                            </DialogDescription>
                        </DialogHeader>
                        <form onSubmit={handleEditSubmit} className="space-y-4">
                            <div className="grid grid-cols-2 gap-4">
                                <div className="space-y-2">
                                    <Label htmlFor="edit_amount">Amount *</Label>
                                    <Input
                                        id="edit_amount"
                                        type="number"
                                        step="0.01"
                                        value={editForm.data.amount}
                                        onChange={(e) => editForm.setData('amount', e.target.value)}
                                        required
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_expense_date">Date *</Label>
                                    <Input
                                        id="edit_expense_date"
                                        type="date"
                                        value={editForm.data.expense_date}
                                        onChange={(e) => editForm.setData('expense_date', e.target.value)}
                                        required
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_category">Category</Label>
                                    <Select
                                        value={editForm.data.category}
                                        onValueChange={(value) => editForm.setData('category', value)}
                                    >
                                        <SelectTrigger id="edit_category">
                                            <SelectValue placeholder="Select category" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            {categories.map((cat) => (
                                                <SelectItem key={cat} value={cat}>
                                                    {cat}
                                                </SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_vendor_id">Vendor</Label>
                                    <Select
                                        value={editForm.data.vendor_id}
                                        onValueChange={(value) => editForm.setData('vendor_id', value)}
                                    >
                                        <SelectTrigger id="edit_vendor_id">
                                            <SelectValue placeholder="Select vendor" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            {vendors.map((vendor) => (
                                                <SelectItem key={vendor.id} value={vendor.id.toString()}>
                                                    {vendor.name}
                                                </SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                </div>
                                <div className="space-y-2 col-span-2">
                                    <Label htmlFor="edit_description">Description *</Label>
                                    <Textarea
                                        id="edit_description"
                                        value={editForm.data.description}
                                        onChange={(e) => editForm.setData('description', e.target.value)}
                                        required
                                    />
                                </div>
                                <div className="space-y-2 col-span-2">
                                    <Label htmlFor="edit_receipt">Receipt</Label>
                                    <Input
                                        id="edit_receipt"
                                        type="file"
                                        accept="image/*,application/pdf"
                                        onChange={(e) => editForm.setData('receipt', e.target.files?.[0] || null)}
                                    />
                                </div>
                            </div>
                            <DialogFooter>
                                <Button type="button" variant="outline" onClick={() => setIsEditOpen(false)}>
                                    Cancel
                                </Button>
                                <Button type="submit" disabled={editForm.processing}>
                                    {editForm.processing ? 'Updating...' : 'Update Expense'}
                                </Button>
                            </DialogFooter>
                        </form>
                    </DialogContent>
                </Dialog>

                {/* Details Modal */}
                <Dialog open={isDetailsOpen} onOpenChange={setIsDetailsOpen}>
                    <DialogContent className="max-w-2xl">
                        <DialogHeader>
                            <DialogTitle>Expense Details</DialogTitle>
                            <DialogDescription>
                                View complete expense information
                            </DialogDescription>
                        </DialogHeader>
                        {selectedExpense && (
                            <div className="space-y-4">
                                <div className="grid grid-cols-2 gap-4">
                                    <div>
                                        <p className="text-sm font-medium">Expense No</p>
                                        <p className="text-sm text-muted-foreground">{selectedExpense.expense_no}</p>
                                    </div>
                                    <div>
                                        <p className="text-sm font-medium">Amount</p>
                                        <p className="text-sm text-muted-foreground">{formatCurrency(selectedExpense.amount)}</p>
                                    </div>
                                    <div>
                                        <p className="text-sm font-medium">Date</p>
                                        <p className="text-sm text-muted-foreground">{formatDate(selectedExpense.expense_date)}</p>
                                    </div>
                                    <div>
                                        <p className="text-sm font-medium">Category</p>
                                        <p className="text-sm text-muted-foreground">{selectedExpense.category || '-'}</p>
                                    </div>
                                    <div>
                                        <p className="text-sm font-medium">Vendor</p>
                                        <p className="text-sm text-muted-foreground">
                                            {selectedExpense.vendor?.name || selectedExpense.vendor_name || '-'}
                                        </p>
                                    </div>
                                    <div>
                                        <p className="text-sm font-medium">Created By</p>
                                        <p className="text-sm text-muted-foreground">{selectedExpense.created_by.name}</p>
                                    </div>
                                    <div className="col-span-2">
                                        <p className="text-sm font-medium">Description</p>
                                        <p className="text-sm text-muted-foreground">{selectedExpense.description}</p>
                                    </div>
                                    {selectedExpense.receipt_path && (
                                        <div className="col-span-2">
                                            <p className="text-sm font-medium mb-2">Receipt</p>
                                            <a
                                                href={selectedExpense.receipt_path}
                                                target="_blank"
                                                rel="noopener noreferrer"
                                                className="text-sm text-primary hover:underline"
                                            >
                                                View Receipt
                                            </a>
                                        </div>
                                    )}
                                    <div>
                                        <p className="text-sm font-medium">Created At</p>
                                        <p className="text-sm text-muted-foreground">{formatDate(selectedExpense.created_at)}</p>
                                    </div>
                                </div>
                            </div>
                        )}
                    </DialogContent>
                </Dialog>
            </div>
        </SocietyAdminLayout>
    );
}
