import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { GlassCard } from '@/components/ui/glass-card';
import { PageHeader } from '@/components/ui/page-header';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link, router } from '@inertiajs/react';
import { BookOpen, Search, Phone, Mail, MapPin } from 'lucide-react';
import { useState } from 'react';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Directory',
        href: '/societies/{society}/directory',
    },
];

interface Building {
    id: number;
    name: string;
}

interface Unit {
    id: number;
    unit_no: string;
    building_id: number | null;
}

interface Resident {
    id: number;
    name: string;
    email: string | null;
    phone: string;
    unit: {
        id: number;
        unit_no: string;
        building: {
            id: number;
            name: string;
        } | null;
    } | null;
}

interface IndexProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    residents: {
        data: Resident[];
        links: any;
        meta: any;
    };
    buildings: Building[];
    units: Unit[];
    filters: {
        search?: string;
        building_id?: string;
        unit_id?: string;
    };
}

export default function Index({ society, residents, buildings, units, filters }: IndexProps) {
    const [search, setSearch] = useState(filters.search || '');
    const [buildingId, setBuildingId] = useState(filters.building_id || 'all');
    const [unitId, setUnitId] = useState(filters.unit_id || 'all');

    const handleSearch = () => {
        const filterParams: any = {};
        if (search) filterParams.search = search;
        if (buildingId && buildingId !== 'all') filterParams.building_id = buildingId;
        if (unitId && unitId !== 'all') filterParams.unit_id = unitId;

        router.get(`/societies/${society.id}/directory`, filterParams, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    // Filter units by selected building
    const filteredUnits = buildingId && buildingId !== 'all'
        ? units.filter((unit) => unit.building_id === parseInt(buildingId))
        : units;

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Directory" />
            <div className="flex h-full flex-1 flex-col gap-6 overflow-x-auto rounded-xl p-6">
                <PageHeader
                    title="Resident Directory"
                    description="View residents who have opted into the directory"
                    breadcrumbs={breadcrumbs}
                />

                {/* Filters */}
                <GlassCard>
                    <CardContent className="pt-6">
                        <div className="grid gap-4 md:grid-cols-4">
                            <div className="md:col-span-2">
                                <Label htmlFor="search">Search</Label>
                                <div className="relative mt-1">
                                    <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                    <Input
                                        id="search"
                                        placeholder="Search by name, phone, email, or unit..."
                                        value={search}
                                        onChange={(e) => setSearch(e.target.value)}
                                        onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                        className="pl-9"
                                    />
                                </div>
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="building_id">Building</Label>
                                <Select value={buildingId} onValueChange={setBuildingId}>
                                    <SelectTrigger id="building_id">
                                        <SelectValue />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Buildings</SelectItem>
                                        {buildings.map((building) => (
                                            <SelectItem key={building.id} value={building.id.toString()}>
                                                {building.name}
                                            </SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                            </div>

                            <div className="space-y-2">
                                <Label htmlFor="unit_id">Unit</Label>
                                <Select value={unitId} onValueChange={setUnitId}>
                                    <SelectTrigger id="unit_id">
                                        <SelectValue />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Units</SelectItem>
                                        {filteredUnits.map((unit) => (
                                            <SelectItem key={unit.id} value={unit.id.toString()}>
                                                {unit.unit_no}
                                            </SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                            </div>
                        </div>

                        <div className="mt-4 flex items-end gap-4">
                            <button
                                onClick={handleSearch}
                                className="inline-flex items-center justify-center rounded-md bg-primary px-4 py-2 text-sm font-medium text-primary-foreground hover:bg-primary/90"
                            >
                                Filter
                            </button>
                        </div>
                    </CardContent>
                </GlassCard>

                {/* Directory Grid */}
                <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
                    {residents.data.length > 0 ? (
                        residents.data.map((resident) => (
                            <GlassCard key={resident.id} variant="hover" className="transition-all duration-300 py-6 flex flex-col gap-4">
                                <CardHeader>
                                    <div className="flex items-center gap-3">
                                        <div className="flex h-12 w-12 items-center justify-center rounded-full bg-primary/10">
                                            <BookOpen className="h-6 w-6 text-primary" />
                                        </div>
                                        <div className="flex-1">
                                            <CardTitle className="text-lg">{resident.name}</CardTitle>
                                            {resident.unit && (
                                                <p className="text-sm text-muted-foreground flex items-center gap-1 mt-1">
                                                    <MapPin className="h-3 w-3" />
                                                    {resident.unit.unit_no}
                                                    {resident.unit.building && ` - ${resident.unit.building.name}`}
                                                </p>
                                            )}
                                        </div>
                                    </div>
                                </CardHeader>
                                <CardContent>
                                    <div className="space-y-2">
                                        {resident.phone && (
                                            <div className="flex items-center gap-2 text-sm">
                                                <Phone className="h-4 w-4 text-muted-foreground" />
                                                <a
                                                    href={`tel:${resident.phone}`}
                                                    className="text-primary hover:underline"
                                                >
                                                    {resident.phone}
                                                </a>
                                            </div>
                                        )}
                                        {resident.email && (
                                            <div className="flex items-center gap-2 text-sm">
                                                <Mail className="h-4 w-4 text-muted-foreground" />
                                                <a
                                                    href={`mailto:${resident.email}`}
                                                    className="text-primary hover:underline"
                                                >
                                                    {resident.email}
                                                </a>
                                            </div>
                                        )}
                                    </div>
                                </CardContent>
                            </GlassCard>
                        ))
                    ) : (
                        <div className="col-span-full">
                            <GlassCard>
                                <CardContent className="py-12 text-center">
                                    <BookOpen className="mx-auto h-12 w-12 text-muted-foreground mb-4" />
                                    <p className="text-muted-foreground">
                                        No residents found in the directory
                                    </p>
                                </CardContent>
                            </GlassCard>
                        </div>
                    )}
                </div>

                {/* Pagination */}
                {residents.links && residents.links.length > 3 && (
                    <div className="flex items-center justify-between">
                        <div className="text-sm text-muted-foreground">
                            Showing {residents.meta.from} to {residents.meta.to} of {residents.meta.total} results
                        </div>
                        <div className="flex gap-2">
                            {residents.links.map((link: any, index: number) => (
                                <Link
                                    key={index}
                                    href={link.url || '#'}
                                    className={`px-3 py-1 rounded ${
                                        link.active
                                            ? 'bg-primary text-primary-foreground'
                                            : 'bg-muted hover:bg-muted/80'
                                    }`}
                                    dangerouslySetInnerHTML={{ __html: link.label }}
                                />
                            ))}
                        </div>
                    </div>
                )}
            </div>
        </SocietyAdminLayout>
    );
}

