import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, usePage } from '@inertiajs/react';
import PaymentRequired from '@/components/PaymentRequired';
import { GlassCard } from '@/components/ui/glass-card';
import {
    DollarSign,
    TrendingDown,
    Receipt,
    Building2,
    Home,
    Wrench,
    UserCheck,
    TrendingUp,
    CreditCard,
    Calendar,
    Activity,
    Users,
    ArrowUpRight,
    ArrowDownRight,
    Clock,
    CheckCircle2,

    AlertCircle,
    Megaphone,
} from 'lucide-react';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import {
    BarChart,
    Bar,
    XAxis,
    YAxis,
    CartesianGrid,
    Tooltip,
    ResponsiveContainer,
    PieChart,
    Pie,
    Cell,
    Legend,
} from 'recharts';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Dashboard',
        href: '/societies/{society}/dashboard',
    },
];

interface DashboardProps {
    society: {
        id: number;
        name: string;
        code?: string;
        registration_no?: string;
        address?: string;
        city?: string;
        state?: string;
        pincode?: string;
    };
    kpis: {
        total_collections: number;
        outstanding_dues: number;
        this_month_expenses: number;
        total_units: number;
        occupied_units: number;
        open_tickets: number;
        today_visitors: number;
        collection_rate: number;
        plan_status: string;
        plan_expiry: string | null;

        plan_name: string;
        ad_revenue: number;
    };
    recentActivity: {
        payments: Array<{
            id: number;
            payment_no: string;
            amount: number;
            user_name: string;
            bill_no: string | null;
            created_at: string;
        }>;
        bills: Array<{
            id: number;
            bill_no: string;
            amount: number;
            status: string;
            user_name: string;
            unit_no: string;
            created_at: string;
        }>;
        expenses: Array<{
            id: number;
            expense_no: string;
            amount: number;
            category: string | null;
            created_by_name: string;
            created_at: string;
        }>;
    };
    charts: {
        monthly_collections: Array<{
            month: string;
            total: number;
        }>;
        monthly_expenses: Array<{
            month: string;
            total: number;
        }>;
    };
}

import DashboardSkeleton from './DashboardSkeleton';

export default function Dashboard({
    society,
    kpis,
    recentActivity,
    charts,
}: DashboardProps) {
    if (!society || !kpis) {
        return <DashboardSkeleton />;
    }

    const { auth, subscriptionPaymentRequired } = usePage().props as any;
    const user = auth.user;

    // Check if subscription payment is required - show payment required page
    if (subscriptionPaymentRequired) {
        return (
            <>
                <Head title="Payment Required" />
                <PaymentRequired 
                    status={subscriptionPaymentRequired.status} 
                    message={subscriptionPaymentRequired.message}
                    has_payment_gateway={subscriptionPaymentRequired.has_payment_gateway ?? true}
                />
            </>
        );
    }

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 0,
        }).format(amount);
    };

    // Prepare data for the combined chart
    const combinedChartData = charts.monthly_collections.map((item, index) => ({
        month: item.month,
        collections: item.total,
        expenses: charts.monthly_expenses[index]?.total || 0,
    }));

    // Data for Occupancy Pie Chart
    const occupancyData = [
        { name: 'Occupied', value: kpis.occupied_units },
        { name: 'Vacant', value: kpis.total_units - kpis.occupied_units },
    ];
    // Vibrant colors for charts
    const PIE_COLORS = ['#0891B2', '#e5e7eb']; // Societify Teal, Gray-200

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Dashboard" />
            <div className="flex flex-1 flex-col gap-8 p-6 md:p-8 overflow-x-hidden bg-muted/10">
                
                {/* Welcome & Society Info Gradient Banner */}
                <div className="relative overflow-hidden rounded-2xl bg-gradient-to-br from-sky-700 via-cyan-600 to-teal-500 p-8 text-white shadow-xl shadow-cyan-500/20">
                    <div className="absolute top-0 right-0 -mt-10 -mr-10 h-64 w-64 rounded-full bg-white/10 blur-3xl" />
                    <div className="absolute bottom-0 left-0 -mb-10 -ml-10 h-64 w-64 rounded-full bg-white/10 blur-3xl" />
                    
                    <div className="relative z-10 flex flex-col md:flex-row justify-between items-start md:items-center gap-6">
                        <div>
                            <h1 className="text-3xl font-bold tracking-tight text-white">
                                Dashboard
                            </h1>
                            <p className="mt-2 text-cyan-100 text-lg">
                                Welcome back, <span className="font-semibold text-white">{user.name}</span>.
                            </p>
                        </div>
                        <div className="flex items-center gap-3 bg-white/10 backdrop-blur-md p-3 rounded-xl border border-white/20 shadow-inner">
                            <div className="h-12 w-12 rounded-lg bg-white/20 flex items-center justify-center text-white shadow-sm">
                                <Building2 className="h-6 w-6" />
                            </div>
                            <div className="pr-2">
                                <p className="text-base font-bold text-white">{society.name}</p>
                                <p className="text-xs text-cyan-100 font-medium tracking-wide">REG: {society.registration_no || 'N/A'}</p>
                            </div>
                        </div>
                    </div>
                </div>

                {/* Subscription Alert (if expiring or trial) */}
                {(kpis.plan_status === 'trial' || kpis.plan_status === 'expired') && (
                    <div className={`rounded-xl border-l-4 p-4 flex items-start gap-4 shadow-sm ${
                        kpis.plan_status === 'expired' 
                            ? 'bg-red-50 border-red-500 text-red-900 dark:bg-red-900/20 dark:text-red-200' 
                            : 'bg-blue-50 border-blue-500 text-blue-900 dark:bg-blue-900/20 dark:text-blue-200'
                    }`}>
                        <div className={`p-2 rounded-full ${kpis.plan_status === 'expired' ? 'bg-red-100 text-red-600' : 'bg-blue-100 text-blue-600'}`}>
                            <AlertCircle className="h-5 w-5" />
                        </div>
                        <div>
                            <h3 className="font-bold text-sm uppercase tracking-wide">
                                {kpis.plan_status === 'expired' ? 'Subscription Expired' : 'Trial Period Active'}
                            </h3>
                            <p className="text-sm mt-1 opacity-90">
                                {kpis.plan_status === 'expired' 
                                    ? 'Your subscription has expired. Please renew to continue accessing all features.' 
                                    : `You are currently on the ${kpis.plan_name} trial. It will expire on ${kpis.plan_expiry ? new Date(kpis.plan_expiry).toLocaleDateString() : 'soon'}.`}
                            </p>
                        </div>
                    </div>
                )}

                {/* Financial KPIs */}
                <div>
                    <h3 className="text-lg font-bold mb-4 flex items-center gap-2 text-foreground/80">
                        <div className="p-1.5 rounded-md bg-emerald-100 text-emerald-600 dark:bg-emerald-900/30 dark:text-emerald-400">
                            <DollarSign className="h-5 w-5" />
                        </div>
                        Financial Overview
                    </h3>
                    <div className="grid gap-4 grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-5">
                        <GlassCard variant="hover" className="group py-6 flex flex-col gap-4">
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium text-muted-foreground group-hover:text-emerald-500 transition-colors">Total Collections</CardTitle>
                                <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-emerald-400 to-green-600 flex items-center justify-center text-white shadow-md shadow-emerald-500/20 group-hover:scale-110 transition-transform">
                                    <DollarSign className="h-5 w-5" />
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="text-2xl font-bold text-foreground">{formatCurrency(kpis.total_collections)}</div>
                                <p className="text-xs text-emerald-600 font-medium mt-1 flex items-center gap-1">
                                    <ArrowUpRight className="h-3 w-3" />
                                    All time revenue
                                </p>
                            </CardContent>
                        </GlassCard>

                        <GlassCard variant="hover" className="group py-6 flex flex-col gap-4">
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium text-muted-foreground group-hover:text-rose-500 transition-colors">Outstanding Dues</CardTitle>
                                <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-rose-400 to-red-600 flex items-center justify-center text-white shadow-md shadow-rose-500/20 group-hover:scale-110 transition-transform">
                                    <AlertCircle className="h-5 w-5" />
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="text-2xl font-bold text-rose-600 dark:text-rose-400">
                                    {formatCurrency(kpis.outstanding_dues)}
                                </div>
                                <p className="text-xs text-muted-foreground mt-1">
                                    Pending from residents
                                </p>
                            </CardContent>
                        </GlassCard>

                        <GlassCard variant="hover" className="group py-6 flex flex-col gap-4">
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium text-muted-foreground group-hover:text-orange-500 transition-colors">Monthly Expenses</CardTitle>
                                <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-orange-400 to-amber-600 flex items-center justify-center text-white shadow-md shadow-orange-500/20 group-hover:scale-110 transition-transform">
                                    <Receipt className="h-5 w-5" />
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="text-2xl font-bold text-foreground">{formatCurrency(kpis.this_month_expenses)}</div>
                                <p className="text-xs text-muted-foreground mt-1">
                                    For {new Date().toLocaleString('default', { month: 'long' })}
                                </p>
                            </CardContent>
                        </GlassCard>

                        <GlassCard variant="hover" className="group py-6 flex flex-col gap-4">
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium text-muted-foreground group-hover:text-violet-500 transition-colors">Collection Rate</CardTitle>
                                <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-cyan-400 to-teal-600 flex items-center justify-center text-white shadow-md shadow-cyan-500/20 group-hover:scale-110 transition-transform">
                                    <Activity className="h-5 w-5" />
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="text-2xl font-bold text-foreground">{kpis.collection_rate}%</div>
                                <div className="w-full bg-secondary h-2 rounded-full mt-2 overflow-hidden">
                                    <div 
                                        className="bg-gradient-to-r from-cyan-500 to-teal-600 h-full rounded-full transition-all duration-500 shadow-[0_0_10px_rgba(8,145,178,0.5)]" 
                                        style={{ width: `${kpis.collection_rate}%` }}
                                    />
                                </div>
                            </CardContent>
                        </GlassCard>


                        <GlassCard variant="hover" className="group py-6 flex flex-col gap-4">
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium text-muted-foreground group-hover:text-fuchsia-500 transition-colors">Ad Revenue</CardTitle>
                                <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-fuchsia-400 to-purple-600 flex items-center justify-center text-white shadow-md shadow-fuchsia-500/20 group-hover:scale-110 transition-transform">
                                    <Megaphone className="h-5 w-5" />
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="text-2xl font-bold text-foreground">{formatCurrency(kpis.ad_revenue)}</div>
                                <p className="text-xs text-cyan-600 font-medium mt-1">
                                    Total from Advertisements
                                </p>
                            </CardContent>
                        </GlassCard>
                    </div>
                </div>

                {/* Charts Section */}
                <div className="grid gap-6 md:grid-cols-3">
                    {/* Revenue vs Expenses Chart */}
                    <GlassCard variant="hover" className="md:col-span-2 py-6 flex flex-col gap-4">
                        <CardHeader>
                            <CardTitle className="text-lg font-bold text-foreground/80">Financial Performance</CardTitle>
                            <CardDescription>Collections vs Expenses over the last 6 months</CardDescription>
                        </CardHeader>
                        <CardContent className="pl-0">
                            <div className="h-[350px] w-full">
                                <ResponsiveContainer width="100%" height="100%">
                                    <BarChart data={combinedChartData} margin={{ top: 20, right: 30, left: 0, bottom: 0 }}>
                                        <CartesianGrid strokeDasharray="3 3" vertical={false} className="stroke-muted/50" />
                                        <XAxis 
                                            dataKey="month" 
                                            axisLine={false} 
                                            tickLine={false} 
                                            tick={{ fill: 'hsl(var(--muted-foreground))', fontSize: 12, fontWeight: 500 }} 
                                            dy={10}
                                        />
                                        <YAxis 
                                            axisLine={false} 
                                            tickLine={false} 
                                            tick={{ fill: 'hsl(var(--muted-foreground))', fontSize: 12, fontWeight: 500 }}
                                            tickFormatter={(value) => `₹${value / 1000}k`}
                                        />
                                        <Tooltip 
                                            cursor={{ fill: 'hsl(var(--muted))', opacity: 0.2 }}
                                            contentStyle={{ 
                                                backgroundColor: 'hsl(var(--popover))', 
                                                borderColor: 'hsl(var(--border))',
                                                borderRadius: '12px',
                                                boxShadow: '0 10px 30px -10px rgba(0,0,0,0.2)',
                                                padding: '12px'
                                            }}
                                            formatter={(value: number) => [formatCurrency(value), '']}
                                        />
                                        <Legend wrapperStyle={{ paddingTop: '20px' }} iconType="circle" />
                                        <Bar 
                                            dataKey="collections" 
                                            name="Collections" 
                                            fill="url(#colorCollections)" 
                                            radius={[6, 6, 0, 0]} 
                                            barSize={20}
                                        >
                                            <defs>
                                                <linearGradient id="colorCollections" x1="0" y1="0" x2="0" y2="1">
                                                    <stop offset="0%" stopColor="#10b981" stopOpacity={1}/>
                                                    <stop offset="100%" stopColor="#059669" stopOpacity={0.8}/>
                                                </linearGradient>
                                            </defs>
                                        </Bar>
                                        <Bar 
                                            dataKey="expenses" 
                                            name="Expenses" 
                                            fill="url(#colorExpenses)" 
                                            radius={[6, 6, 0, 0]} 
                                            barSize={20}
                                        >
                                            <defs>
                                                <linearGradient id="colorExpenses" x1="0" y1="0" x2="0" y2="1">
                                                    <stop offset="0%" stopColor="#f43f5e" stopOpacity={1}/>
                                                    <stop offset="100%" stopColor="#e11d48" stopOpacity={0.8}/>
                                                </linearGradient>
                                            </defs>
                                        </Bar>
                                    </BarChart>
                                </ResponsiveContainer>
                            </div>
                        </CardContent>
                    </GlassCard>

                    {/* Occupancy Pie Chart */}
                    <GlassCard variant="hover" className="py-6 flex flex-col gap-4">
                        <CardHeader>
                            <CardTitle className="text-lg font-bold text-foreground/80">Occupancy Status</CardTitle>
                            <CardDescription>Total Units: {kpis.total_units}</CardDescription>
                        </CardHeader>
                        <CardContent>
                            <div className="h-[220px] w-full relative">
                                <ResponsiveContainer width="100%" height="100%">
                                    <PieChart>
                                        <Pie
                                            data={occupancyData}
                                            cx="50%"
                                            cy="50%"
                                            innerRadius={65}
                                            outerRadius={85}
                                            paddingAngle={5}
                                            dataKey="value"
                                            stroke="none"
                                        >
                                            {occupancyData.map((entry, index) => (
                                                <Cell key={`cell-${index}`} fill={PIE_COLORS[index % PIE_COLORS.length]} />
                                            ))}
                                        </Pie>
                                        <Tooltip 
                                            contentStyle={{ 
                                                backgroundColor: 'hsl(var(--popover))', 
                                                borderColor: 'hsl(var(--border))',
                                                borderRadius: '12px',
                                                boxShadow: '0 10px 30px -10px rgba(0,0,0,0.2)'
                                            }}
                                        />
                                    </PieChart>
                                </ResponsiveContainer>
                                <div className="absolute inset-0 flex flex-col items-center justify-center pointer-events-none">
                                    <span className="text-4xl font-black text-cyan-600 dark:text-cyan-400">{kpis.occupied_units}</span>
                                    <span className="text-xs font-semibold text-muted-foreground uppercase tracking-wider">Occupied</span>
                                </div>
                            </div>
                            <div className="mt-6 space-y-3">
                                <div className="flex items-center justify-between text-sm p-3 rounded-lg bg-cyan-50 dark:bg-cyan-900/10 border border-cyan-100 dark:border-cyan-800/30">
                                    <div className="flex items-center gap-2">
                                        <div className="w-3 h-3 rounded-full bg-cyan-500" />
                                        <span className="font-medium text-cyan-900 dark:text-cyan-100">Occupied</span>
                                    </div>
                                    <span className="font-bold text-cyan-700 dark:text-cyan-300">{kpis.occupied_units}</span>
                                </div>
                                <div className="flex items-center justify-between text-sm p-3 rounded-lg bg-gray-50 dark:bg-gray-800/50 border border-gray-100 dark:border-gray-700">
                                    <div className="flex items-center gap-2">
                                        <div className="w-3 h-3 rounded-full bg-gray-300" />
                                        <span className="font-medium text-gray-700 dark:text-gray-300">Vacant</span>
                                    </div>
                                    <span className="font-bold text-gray-600 dark:text-gray-400">{kpis.total_units - kpis.occupied_units}</span>
                                </div>
                            </div>
                        </CardContent>
                    </GlassCard>
                </div>

                {/* Operational KPIs */}
                <div>
                    <h3 className="text-lg font-bold mb-4 flex items-center gap-2 text-foreground/80">
                        <div className="p-1.5 rounded-md bg-blue-100 text-blue-600 dark:bg-blue-900/30 dark:text-blue-400">
                            <Building2 className="h-5 w-5" />
                        </div>
                        Operational Overview
                    </h3>
                    <div className="grid gap-4 md:grid-cols-3">
                        <GlassCard variant="hover" className="group border-l-4 border-l-amber-500 py-6 flex flex-col gap-4">
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium text-muted-foreground">Open Tickets</CardTitle>
                                <div className="h-8 w-8 rounded-lg bg-amber-100 text-amber-600 flex items-center justify-center group-hover:bg-amber-500 group-hover:text-white transition-colors">
                                    <Wrench className="h-4 w-4" />
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="text-2xl font-bold text-foreground">{kpis.open_tickets}</div>
                                <p className="text-xs text-muted-foreground mt-1">
                                    Active maintenance requests
                                </p>
                            </CardContent>
                        </GlassCard>

                        <GlassCard variant="hover" className="group border-l-4 border-l-cyan-500 py-6 flex flex-col gap-4">
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium text-muted-foreground">Today's Visitors</CardTitle>
                                <div className="h-8 w-8 rounded-lg bg-cyan-100 text-cyan-600 flex items-center justify-center group-hover:bg-cyan-500 group-hover:text-white transition-colors">
                                    <UserCheck className="h-4 w-4" />
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="text-2xl font-bold text-foreground">{kpis.today_visitors}</div>
                                <p className="text-xs text-muted-foreground mt-1">
                                    Checked in today
                                </p>
                            </CardContent>
                        </GlassCard>

                        <GlassCard variant="hover" className="group border-l-4 border-l-purple-500 py-6 flex flex-col gap-4">
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium text-muted-foreground">Subscription Plan</CardTitle>
                                <div className="h-8 w-8 rounded-lg bg-purple-100 text-purple-600 flex items-center justify-center group-hover:bg-purple-500 group-hover:text-white transition-colors">
                                    <CreditCard className="h-4 w-4" />
                                </div>
                            </CardHeader>
                            <CardContent className="space-y-3">
                                <div className="text-lg font-bold truncate text-foreground">{kpis.plan_name || 'No Plan'}</div>
                                <div className="flex flex-wrap items-center gap-2">
                                    <span className={`text-xs px-2 py-0.5 rounded-md font-bold ${
                                        kpis.plan_status === 'active' ? 'bg-green-100 text-green-700' : 
                                        kpis.plan_status === 'trial' ? 'bg-blue-100 text-blue-700' : 
                                        kpis.plan_status === 'expired' ? 'bg-red-100 text-red-700' :
                                        kpis.plan_status === 'pending_payment' ? 'bg-yellow-100 text-yellow-700' :
                                        'bg-gray-100 text-gray-700'
                                    }`}>
                                        {(kpis.plan_status || 'None').toUpperCase().replace('_', ' ')}
                                    </span>
                                </div>
                                {kpis.plan_expiry && (
                                    <div className="flex items-center gap-1.5 text-xs text-muted-foreground">
                                        <Calendar className="h-3 w-3" />
                                        <span>
                                            {kpis.plan_status === 'active' ? 'Expires:' : 'Expired:'} {new Date(kpis.plan_expiry).toLocaleDateString('en-IN', { 
                                                day: 'numeric', 
                                                month: 'short', 
                                                year: 'numeric' 
                                            })}
                                        </span>
                                    </div>
                                )}
                                {/* Renew Button - Only show for non-active plans or plans expiring soon */}
                                {(kpis.plan_status !== 'active' || (kpis.plan_expiry && new Date(kpis.plan_expiry) <= new Date(Date.now() + 30 * 24 * 60 * 60 * 1000))) && (
                                    <button
                                        onClick={() => window.location.href = '/subscription/renew'}
                                        className="mt-2 w-full bg-gradient-to-r from-cyan-500 to-teal-600 hover:from-cyan-600 hover:to-teal-700 text-white text-xs font-semibold py-2 px-3 rounded-lg transition-all duration-200 flex items-center justify-center gap-1.5"
                                    >
                                        <CreditCard className="h-3 w-3" />
                                        {kpis.plan_status === 'active' ? 'Renew Early' : 'Renew Now'}
                                    </button>
                                )}
                            </CardContent>
                        </GlassCard>
                    </div>
                </div>

                {/* Recent Activity Tabs */}
                <GlassCard variant="hover">
                    <CardHeader className="bg-muted/30 border-b border-white/10">
                        <div className="flex items-center justify-between">
                            <div>
                                <CardTitle className="text-lg font-bold">Recent Activity</CardTitle>
                                <CardDescription>Latest transactions and updates</CardDescription>
                            </div>
                            <div className="p-2 bg-white/10 rounded-full shadow-sm">
                                <Clock className="h-5 w-5 text-muted-foreground" />
                            </div>
                        </div>
                    </CardHeader>
                    <CardContent className="p-6">
                        <Tabs defaultValue="payments" className="w-full">
                            <TabsList className="grid w-full grid-cols-3 lg:w-[400px] mb-6 p-1 bg-muted/50 rounded-xl">
                                <TabsTrigger value="payments" className="rounded-lg data-[state=active]:bg-white data-[state=active]:text-green-600 data-[state=active]:shadow-sm font-medium">Payments</TabsTrigger>
                                <TabsTrigger value="bills" className="rounded-lg data-[state=active]:bg-white data-[state=active]:text-blue-600 data-[state=active]:shadow-sm font-medium">Bills</TabsTrigger>
                                <TabsTrigger value="expenses" className="rounded-lg data-[state=active]:bg-white data-[state=active]:text-orange-600 data-[state=active]:shadow-sm font-medium">Expenses</TabsTrigger>
                            </TabsList>
                            
                            <TabsContent value="payments" className="mt-0">
                                <div className="space-y-4">
                                    {recentActivity.payments.length > 0 ? (
                                        recentActivity.payments.map((payment) => (
                                            <div key={payment.id} className="group flex items-center justify-between p-3 rounded-xl hover:bg-muted/50 transition-colors border border-transparent hover:border-border/50">
                                                <div className="flex items-center gap-4">
                                                    <div className="h-10 w-10 rounded-full bg-green-100 flex items-center justify-center text-green-600 group-hover:scale-110 transition-transform">
                                                        <ArrowDownRight className="h-5 w-5" />
                                                    </div>
                                                    <div>
                                                        <p className="text-sm font-bold text-foreground">{payment.user_name}</p>
                                                        <p className="text-xs text-muted-foreground font-medium">{payment.payment_no} • {new Date(payment.created_at).toLocaleDateString()}</p>
                                                    </div>
                                                </div>
                                                <div className="font-bold text-green-600 bg-green-50 px-3 py-1 rounded-lg border border-green-100">
                                                    +{formatCurrency(payment.amount)}
                                                </div>
                                            </div>
                                        ))
                                    ) : (
                                        <div className="text-center py-12 text-muted-foreground bg-muted/20 rounded-xl border border-dashed">No recent payments</div>
                                    )}
                                </div>
                            </TabsContent>

                            <TabsContent value="bills" className="mt-0">
                                <div className="space-y-4">
                                    {recentActivity.bills.length > 0 ? (
                                        recentActivity.bills.map((bill) => (
                                            <div key={bill.id} className="group flex items-center justify-between p-3 rounded-xl hover:bg-muted/50 transition-colors border border-transparent hover:border-border/50">
                                                <div className="flex items-center gap-4">
                                                    <div className="h-10 w-10 rounded-full bg-blue-100 flex items-center justify-center text-blue-600 group-hover:scale-110 transition-transform">
                                                        <Receipt className="h-5 w-5" />
                                                    </div>
                                                    <div>
                                                        <p className="text-sm font-bold text-foreground">{bill.user_name} <span className="text-muted-foreground font-normal">({bill.unit_no})</span></p>
                                                        <p className="text-xs text-muted-foreground font-medium">{bill.bill_no} • {new Date(bill.created_at).toLocaleDateString()}</p>
                                                    </div>
                                                </div>
                                                <div className="text-right flex flex-col items-end gap-1">
                                                    <div className="font-bold text-foreground">{formatCurrency(bill.amount)}</div>
                                                    <span className={`text-[10px] px-2 py-0.5 rounded-full font-bold uppercase tracking-wide ${
                                                        bill.status === 'paid' ? 'bg-green-100 text-green-700' :
                                                        bill.status === 'overdue' ? 'bg-red-100 text-red-700' :
                                                        'bg-yellow-100 text-yellow-700'
                                                    }`}>
                                                        {bill.status}
                                                    </span>
                                                </div>
                                            </div>
                                        ))
                                    ) : (
                                        <div className="text-center py-12 text-muted-foreground bg-muted/20 rounded-xl border border-dashed">No recent bills</div>
                                    )}
                                </div>
                            </TabsContent>

                            <TabsContent value="expenses" className="mt-0">
                                <div className="space-y-4">
                                    {recentActivity.expenses.length > 0 ? (
                                        recentActivity.expenses.map((expense) => (
                                            <div key={expense.id} className="group flex items-center justify-between p-3 rounded-xl hover:bg-muted/50 transition-colors border border-transparent hover:border-border/50">
                                                <div className="flex items-center gap-4">
                                                    <div className="h-10 w-10 rounded-full bg-orange-100 flex items-center justify-center text-orange-600 group-hover:scale-110 transition-transform">
                                                        <ArrowUpRight className="h-5 w-5" />
                                                    </div>
                                                    <div>
                                                        <p className="text-sm font-bold text-foreground">{expense.category || 'Uncategorized'}</p>
                                                        <p className="text-xs text-muted-foreground font-medium">By {expense.created_by_name} • {new Date(expense.created_at).toLocaleDateString()}</p>
                                                    </div>
                                                </div>
                                                <div className="font-bold text-red-600 bg-red-50 px-3 py-1 rounded-lg border border-red-100">
                                                    -{formatCurrency(expense.amount)}
                                                </div>
                                            </div>
                                        ))
                                    ) : (
                                        <div className="text-center py-12 text-muted-foreground bg-muted/20 rounded-xl border border-dashed">No recent expenses</div>
                                    )}
                                </div>
                            </TabsContent>
                        </Tabs>
                    </CardContent>
                </GlassCard>
            </div>
        </SocietyAdminLayout>
    );
}
