import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Badge } from '@/components/ui/badge';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { Head, Link, router } from '@inertiajs/react';
import { Trash2, Eye, Search, MessageSquare, ThumbsUp, Image as ImageIcon, Video } from 'lucide-react';
import { useState } from 'react';
import Swal from 'sweetalert2';
import { format } from 'date-fns';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogFooter } from '@/components/ui/dialog';

export default function Index({ society, posts }: any) {
    const [search, setSearch] = useState('');
    const [viewingPost, setViewingPost] = useState<any>(null);

    const handleDelete = (id: number) => {
        Swal.fire({
            title: 'Delete Post?',
            text: "This cannot be undone.",
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#EF4444',
            confirmButtonText: 'Yes, DELETE'
        }).then((result) => {
            if (result.isConfirmed) {
                router.delete(`/societies/${society.id}/community/${id}`);
            }
        });
    };

    const handleSearch = () => {
        router.get(`/societies/${society.id}/community`, { search }, { preserveState: true });
    };

    const stripHtml = (html: string) => {
        const tmp = document.createElement('DIV');
        tmp.innerHTML = html;
        return tmp.textContent || tmp.innerText || '';
    };

    return (
        <SocietyAdminLayout society={society} breadcrumbs={[{ title: 'Community', href: '#' }]}>
            <Head title="Community Posts" />
            <div className="flex h-full flex-1 flex-col gap-6 overflow-x-auto rounded-xl p-6">
                <div className="flex items-center justify-between">
                    <div>
                        <h1 className="text-2xl font-bold">Community Feed</h1>
                        <p className="text-muted-foreground">Manage posts and discussions</p>
                    </div>
                </div>

                {/* Filters */}
                {/* Filters */}
                <Card>
                    <CardContent className="pt-6">
                        <div className="flex gap-4">
                            <div className="flex-1">
                                <Label htmlFor="search">Search</Label>
                                <div className="relative mt-1">
                                    <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                    <Input
                                        id="search"
                                        placeholder="Search by author or content..."
                                        value={search}
                                        onChange={(e) => setSearch(e.target.value)}
                                        onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                        className="pl-9"
                                    />
                                </div>
                            </div>
                            <div className="flex items-end">
                                <Button onClick={handleSearch}>Filter</Button>
                            </div>
                        </div>
                    </CardContent>
                </Card>

                {/* Posts Table */}
                <Card>
                    <CardHeader>
                        <CardTitle>Recent Posts</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead>
                                    <tr className="border-b">
                                        <th className="px-4 py-3 text-left text-sm font-medium">Date</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Author</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Content</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Engagement</th>
                                        <th className="px-4 py-3 text-right text-sm font-medium">Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {posts.data.length === 0 ? (
                                        <tr>
                                            <td colSpan={5} className="px-4 py-8 text-center text-muted-foreground">
                                                No posts found.
                                            </td>
                                        </tr>
                                    ) : (
                                        posts.data.map((post: any) => (
                                            <tr key={post.id} className="border-b hover:bg-muted/50">
                                                <td className="px-4 py-3 text-sm text-muted-foreground whitespace-nowrap">
                                                    {format(new Date(post.created_at), 'dd MMM yyyy')}
                                                    <div className="text-xs">{format(new Date(post.created_at), 'hh:mm a')}</div>
                                                </td>
                                                <td className="px-4 py-3">
                                                    <div className="font-medium text-sm">{post.user?.name || 'Unknown'}</div>
                                                    <div className="text-xs text-muted-foreground">
                                                        {post.user?.unit ? `${post.user.unit.building?.name || ''} - ${post.user.unit.unit_no}` : 'No Unit'}
                                                    </div>
                                                </td>
                                                <td className="px-4 py-3 max-w-md">
                                                    <div className="font-medium text-sm truncate">{post.title !== 'Post' ? post.title : ''}</div>
                                                    <div className="text-sm text-muted-foreground line-clamp-2">
                                                        {stripHtml(post.content)}
                                                    </div>
                                                    {(post.image || (post.media_urls && post.media_urls.length > 0)) && (
                                                        <div className="flex items-center gap-1 text-xs text-blue-600 mt-1">
                                                            {post.media_type === 'video' ? <Video className="h-3 w-3" /> : <ImageIcon className="h-3 w-3" />}
                                                            <span>Has Media</span>
                                                        </div>
                                                    )}
                                                </td>
                                                <td className="px-4 py-3">
                                                    <div className="flex items-center gap-3 text-sm text-muted-foreground">
                                                        <span className="flex items-center gap-1"><ThumbsUp className="h-3 w-3" /> {post.likes_count}</span>
                                                        <span className="flex items-center gap-1"><MessageSquare className="h-3 w-3" /> {post.comments_count}</span>
                                                    </div>
                                                </td>
                                                <td className="px-4 py-3 text-right">
                                                    <div className="flex justify-end gap-2">
                                                        <Button variant="ghost" size="icon" onClick={() => setViewingPost(post)}>
                                                            <Eye className="h-4 w-4 text-blue-500" />
                                                        </Button>
                                                        <Button variant="ghost" size="icon" onClick={() => handleDelete(post.id)}>
                                                            <Trash2 className="h-4 w-4 text-red-500" />
                                                        </Button>
                                                    </div>
                                                </td>
                                            </tr>
                                        ))
                                    )}
                                </tbody>
                            </table>
                        </div>

                        {/* Pagination */}
                        {posts.links && posts.links.length > 3 && (
                            <div className="mt-4 flex items-center justify-between">
                                <div className="text-sm text-muted-foreground">
                                    Page {posts.current_page} of {posts.last_page}
                                </div>
                                <div className="flex gap-2">
                                    {posts.links.map((link: any, index: number) => (
                                        <Link
                                            key={index}
                                            href={link.url || '#'}
                                            className={`px-3 py-1 rounded text-sm ${
                                                link.active
                                                    ? 'bg-primary text-primary-foreground'
                                                    : 'bg-muted hover:bg-muted/80'
                                            } ${!link.url ? 'opacity-50 pointer-events-none' : ''}`}
                                            dangerouslySetInnerHTML={{ __html: link.label }}
                                        />
                                    ))}
                                </div>
                            </div>
                        )}
                    </CardContent>
                </Card>

                {/* View Dialog */}
                <Dialog open={!!viewingPost} onOpenChange={(open) => !open && setViewingPost(null)}>
                    <DialogContent className="sm:max-w-[600px] max-h-[80vh] overflow-y-auto">
                        <DialogHeader>
                            <DialogTitle>{viewingPost?.title !== 'Post' ? viewingPost?.title : 'Community Post'}</DialogTitle>
                            <DialogDescription>
                                Posted by {viewingPost?.user?.name} on {viewingPost && format(new Date(viewingPost.created_at), 'PPP p')}
                            </DialogDescription>
                        </DialogHeader>
                        <div className="space-y-4">
                            <div className="whitespace-pre-wrap text-sm">{viewingPost?.content}</div>
                            {viewingPost?.image && (
                                <img src={viewingPost.image} alt="Post" className="rounded-lg w-full object-cover max-h-[400px]" />
                            )}
                            {viewingPost?.media_urls && viewingPost.media_urls.map((url: string, idx: number) => (
                                viewingPost.media_type === 'video' ? (
                                    <video key={idx} src={url} controls className="w-full rounded-lg max-h-[400px]" />
                                ) : (
                                    <img key={idx} src={url} alt={`Media ${idx}`} className="rounded-lg w-full object-cover max-h-[400px]" />
                                )
                            ))}
                        </div>
                        <DialogFooter>
                            <Button variant="outline" onClick={() => setViewingPost(null)}>Close</Button>
                        </DialogFooter>
                    </DialogContent>
                </Dialog>
            </div>
        </SocietyAdminLayout>
    );
}
