import { useState } from 'react';
import { Head, router } from '@inertiajs/react';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Wallet, IndianRupee, ArrowDownRight, Users, Eye, ArrowUpRight } from 'lucide-react';
import {
    Dialog,
    DialogContent,
    DialogDescription,
    DialogHeader,
    DialogTitle,
} from "@/components/ui/dialog";
import { Textarea } from '@/components/ui/textarea';

interface CollectorData {
    id: number;
    name: string;
    email: string | null;
    phone: string | null;
    wallet: {
        id: number | null;
        balance: number;
        total_collected: number;
        total_handover: number;
    };
}

interface WalletsProps {
    society: { id: number; name: string; code: string };
    collectors: CollectorData[];
    summary: {
        total_cash_in_hand: number;
        total_collected: number;
        total_handover: number;
        collectors_count: number;
    };
}

export default function Wallets({ society, collectors, summary }: WalletsProps) {
    const [handoverModalOpen, setHandoverModalOpen] = useState(false);
    const [selectedCollector, setSelectedCollector] = useState<CollectorData | null>(null);
    const [amount, setAmount] = useState('');
    const [notes, setNotes] = useState('');
    const [processing, setProcessing] = useState(false);

    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Dashboard', href: `/societies/${society.id}/dashboard` },
        { title: 'Collector Wallets', href: `/societies/${society.id}/collectors/wallets` },
    ];

    const formatCurrency = (val: number) => {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 0,
        }).format(val);
    };

    const openHandoverModal = (collector: CollectorData) => {
        setSelectedCollector(collector);
        setAmount('');
        setNotes('');
        setHandoverModalOpen(true);
    };

    const handleRecordHandover = (e: React.FormEvent) => {
        e.preventDefault();
        if (!selectedCollector || !amount) return;

        setProcessing(true);
        router.post(
            `/societies/${society.id}/collectors/${selectedCollector.id}/handover`,
            { amount: parseFloat(amount), notes },
            {
                onSuccess: () => {
                    setHandoverModalOpen(false);
                    setAmount('');
                    setNotes('');
                },
                onFinish: () => setProcessing(false),
            }
        );
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Collector Wallets" />

            <div className="flex flex-col gap-6 p-6">
                {/* Header */}
                <div className="flex items-center justify-between">
                    <div>
                        <h1 className="text-2xl font-bold text-foreground">Collector Wallets</h1>
                        <p className="text-muted-foreground">Track cash collected by collectors and record handovers</p>
                    </div>
                </div>

                {/* Summary Cards */}
                <div className="grid gap-4 md:grid-cols-4">
                    <Card className="border-l-4 border-l-amber-500">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Cash in Hand</CardTitle>
                            <Wallet className="h-4 w-4 text-amber-500" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-amber-600">{formatCurrency(summary.total_cash_in_hand)}</div>
                            <p className="text-xs text-muted-foreground">Pending handover</p>
                        </CardContent>
                    </Card>

                    <Card className="border-l-4 border-l-green-500">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Total Collected</CardTitle>
                            <ArrowDownRight className="h-4 w-4 text-green-500" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-green-600">{formatCurrency(summary.total_collected)}</div>
                            <p className="text-xs text-muted-foreground">All time collections</p>
                        </CardContent>
                    </Card>

                    <Card className="border-l-4 border-l-blue-500">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Total Handover</CardTitle>
                            <ArrowUpRight className="h-4 w-4 text-blue-500" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-blue-600">{formatCurrency(summary.total_handover)}</div>
                            <p className="text-xs text-muted-foreground">Cash handed over</p>
                        </CardContent>
                    </Card>

                    <Card className="border-l-4 border-l-purple-500">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Collectors</CardTitle>
                            <Users className="h-4 w-4 text-purple-500" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-purple-600">{summary.collectors_count}</div>
                            <p className="text-xs text-muted-foreground">Active collectors</p>
                        </CardContent>
                    </Card>
                </div>

                {/* Collectors Table */}
                <Card>
                    <CardHeader>
                        <CardTitle>Collectors</CardTitle>
                        <CardDescription>View collector balances and record cash handovers</CardDescription>
                    </CardHeader>
                    <CardContent>
                        {collectors.length === 0 ? (
                            <div className="text-center py-8 text-muted-foreground">
                                No collectors found in this society.
                            </div>
                        ) : (
                            <div className="overflow-x-auto">
                                <table className="w-full">
                                    <thead>
                                        <tr className="border-b">
                                            <th className="text-left py-3 px-2 font-medium text-muted-foreground">Collector</th>
                                            <th className="text-left py-3 px-2 font-medium text-muted-foreground">Phone</th>
                                            <th className="text-right py-3 px-2 font-medium text-muted-foreground">Cash in Hand</th>
                                            <th className="text-right py-3 px-2 font-medium text-muted-foreground">Total Collected</th>
                                            <th className="text-right py-3 px-2 font-medium text-muted-foreground">Total Handover</th>
                                            <th className="text-right py-3 px-2 font-medium text-muted-foreground">Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        {collectors.map((collector) => (
                                            <tr key={collector.id} className="border-b hover:bg-muted/50">
                                                <td className="py-3 px-2">
                                                    <div className="font-medium">{collector.name}</div>
                                                    {collector.email && (
                                                        <div className="text-xs text-muted-foreground">{collector.email}</div>
                                                    )}
                                                </td>
                                                <td className="py-3 px-2 text-muted-foreground">{collector.phone || '-'}</td>
                                                <td className="py-3 px-2 text-right">
                                                    <span className={`font-bold ${collector.wallet.balance > 0 ? 'text-amber-600' : 'text-muted-foreground'}`}>
                                                        {formatCurrency(collector.wallet.balance)}
                                                    </span>
                                                </td>
                                                <td className="py-3 px-2 text-right text-green-600">
                                                    {formatCurrency(collector.wallet.total_collected)}
                                                </td>
                                                <td className="py-3 px-2 text-right text-blue-600">
                                                    {formatCurrency(collector.wallet.total_handover)}
                                                </td>
                                                <td className="py-3 px-2 text-right">
                                                    <div className="flex gap-2 justify-end">
                                                        <Button
                                                            size="sm"
                                                            variant="outline"
                                                            onClick={() => router.visit(`/societies/${society.id}/collectors/${collector.id}/wallet`)}
                                                        >
                                                            <Eye className="h-3 w-3 mr-1" />
                                                            Details
                                                        </Button>
                                                        {collector.wallet.balance > 0 && (
                                                            <Button
                                                                size="sm"
                                                                onClick={() => openHandoverModal(collector)}
                                                            >
                                                                <IndianRupee className="h-3 w-3 mr-1" />
                                                                Record Handover
                                                            </Button>
                                                        )}
                                                    </div>
                                                </td>
                                            </tr>
                                        ))}
                                    </tbody>
                                </table>
                            </div>
                        )}
                    </CardContent>
                </Card>
            </div>

            {/* Handover Modal */}
            <Dialog open={handoverModalOpen} onOpenChange={setHandoverModalOpen}>
                <DialogContent>
                    <DialogHeader>
                        <DialogTitle>Record Cash Handover</DialogTitle>
                        <DialogDescription>
                            Record cash received from {selectedCollector?.name}
                        </DialogDescription>
                    </DialogHeader>
                    <form onSubmit={handleRecordHandover} className="space-y-4">
                        <div className="bg-amber-50 dark:bg-amber-900/20 p-3 rounded-lg">
                            <div className="text-sm text-muted-foreground">Available Balance</div>
                            <div className="text-xl font-bold text-amber-600">
                                {formatCurrency(selectedCollector?.wallet.balance || 0)}
                            </div>
                        </div>

                        <div className="space-y-2">
                            <Label htmlFor="amount">Amount to Receive</Label>
                            <div className="relative">
                                <IndianRupee className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                                <Input
                                    id="amount"
                                    type="number"
                                    step="0.01"
                                    min="0.01"
                                    max={selectedCollector?.wallet.balance || 0}
                                    value={amount}
                                    onChange={(e) => setAmount(e.target.value)}
                                    className="pl-9"
                                    placeholder="0.00"
                                    required
                                />
                            </div>
                        </div>

                        <div className="space-y-2">
                            <Label htmlFor="notes">Notes (Optional)</Label>
                            <Textarea
                                id="notes"
                                value={notes}
                                onChange={(e) => setNotes(e.target.value)}
                                placeholder="Any additional notes..."
                                rows={2}
                            />
                        </div>

                        <div className="flex gap-3 justify-end">
                            <Button type="button" variant="outline" onClick={() => setHandoverModalOpen(false)}>
                                Cancel
                            </Button>
                            <Button type="submit" disabled={processing || !amount}>
                                {processing ? 'Recording...' : 'Record Handover'}
                            </Button>
                        </div>
                    </form>
                </DialogContent>
            </Dialog>
        </SocietyAdminLayout>
    );
}
