import { useState } from 'react';
import { Head, router, Link } from '@inertiajs/react';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Wallet, IndianRupee, ArrowDownRight, ArrowUpRight, ArrowLeft } from 'lucide-react';
import {
    Dialog,
    DialogContent,
    DialogDescription,
    DialogHeader,
    DialogTitle,
} from "@/components/ui/dialog";
import { Textarea } from '@/components/ui/textarea';

interface Transaction {
    id: number;
    type: 'credit' | 'debit';
    amount: number;
    balance_after: number;
    reference_type: string | null;
    notes: string | null;
    received_by: string | null;
    created_at: string;
}

interface WalletDetailsProps {
    society: { id: number; name: string; code: string };
    collector: {
        id: number;
        name: string;
        email: string | null;
        phone: string | null;
    };
    wallet: {
        id: number;
        balance: number;
        total_collected: number;
        total_handover: number;
    };
    transactions: {
        data: Transaction[];
        links: any;
        meta?: any;
    };
}

export default function WalletDetails({ society, collector, wallet, transactions }: WalletDetailsProps) {
    const [handoverModalOpen, setHandoverModalOpen] = useState(false);
    const [amount, setAmount] = useState('');
    const [notes, setNotes] = useState('');
    const [processing, setProcessing] = useState(false);

    const breadcrumbs: BreadcrumbItem[] = [
        { title: 'Dashboard', href: `/societies/${society.id}/dashboard` },
        { title: 'Collector Wallets', href: `/societies/${society.id}/collectors/wallets` },
        { title: collector.name, href: `/societies/${society.id}/collectors/${collector.id}/wallet` },
    ];

    const formatCurrency = (val: number) => {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 0,
        }).format(val);
    };

    const handleRecordHandover = (e: React.FormEvent) => {
        e.preventDefault();
        if (!amount) return;

        setProcessing(true);
        router.post(
            `/societies/${society.id}/collectors/${collector.id}/handover`,
            { amount: parseFloat(amount), notes },
            {
                onSuccess: () => {
                    setHandoverModalOpen(false);
                    setAmount('');
                    setNotes('');
                },
                onFinish: () => setProcessing(false),
            }
        );
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title={`${collector.name} - Wallet`} />

            <div className="flex flex-col gap-6 p-6">
                {/* Header */}
                <div className="flex items-center justify-between">
                    <div className="flex items-center gap-4">
                        <Button variant="outline" size="sm" asChild>
                            <Link href={`/societies/${society.id}/collectors/wallets`}>
                                <ArrowLeft className="h-4 w-4 mr-1" />
                                Back
                            </Link>
                        </Button>
                        <div>
                            <h1 className="text-2xl font-bold text-foreground">{collector.name}</h1>
                            <p className="text-muted-foreground">
                                {collector.phone} {collector.email && `• ${collector.email}`}
                            </p>
                        </div>
                    </div>
                    {wallet.balance > 0 && (
                        <Button onClick={() => setHandoverModalOpen(true)}>
                            <IndianRupee className="h-4 w-4 mr-1" />
                            Record Handover
                        </Button>
                    )}
                </div>

                {/* Wallet Summary Cards */}
                <div className="grid gap-4 md:grid-cols-3">
                    <Card className="border-l-4 border-l-amber-500">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Cash in Hand</CardTitle>
                            <Wallet className="h-4 w-4 text-amber-500" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-amber-600">{formatCurrency(wallet.balance)}</div>
                        </CardContent>
                    </Card>

                    <Card className="border-l-4 border-l-green-500">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Total Collected</CardTitle>
                            <ArrowDownRight className="h-4 w-4 text-green-500" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-green-600">{formatCurrency(wallet.total_collected)}</div>
                        </CardContent>
                    </Card>

                    <Card className="border-l-4 border-l-blue-500">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Total Handover</CardTitle>
                            <ArrowUpRight className="h-4 w-4 text-blue-500" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-blue-600">{formatCurrency(wallet.total_handover)}</div>
                        </CardContent>
                    </Card>
                </div>

                {/* Transactions Table */}
                <Card>
                    <CardHeader>
                        <CardTitle>Transaction History</CardTitle>
                        <CardDescription>All wallet transactions for this collector</CardDescription>
                    </CardHeader>
                    <CardContent>
                        {transactions.data.length === 0 ? (
                            <div className="text-center py-8 text-muted-foreground">
                                No transactions found.
                            </div>
                        ) : (
                            <div className="overflow-x-auto">
                                <table className="w-full">
                                    <thead>
                                        <tr className="border-b">
                                            <th className="text-left py-3 px-2 font-medium text-muted-foreground">Date</th>
                                            <th className="text-left py-3 px-2 font-medium text-muted-foreground">Type</th>
                                            <th className="text-right py-3 px-2 font-medium text-muted-foreground">Amount</th>
                                            <th className="text-right py-3 px-2 font-medium text-muted-foreground">Balance After</th>
                                            <th className="text-left py-3 px-2 font-medium text-muted-foreground">Notes</th>
                                            <th className="text-left py-3 px-2 font-medium text-muted-foreground">Received By</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        {transactions.data.map((txn) => (
                                            <tr key={txn.id} className="border-b hover:bg-muted/50">
                                                <td className="py-3 px-2 text-sm text-muted-foreground">
                                                    {new Date(txn.created_at).toLocaleString()}
                                                </td>
                                                <td className="py-3 px-2">
                                                    <span className={`inline-flex items-center rounded-full px-2 py-1 text-xs font-medium ${
                                                        txn.type === 'credit' 
                                                            ? 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200' 
                                                            : 'bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200'
                                                    }`}>
                                                        {txn.type === 'credit' ? (
                                                            <><ArrowDownRight className="h-3 w-3 mr-1" /> Collection</>
                                                        ) : (
                                                            <><ArrowUpRight className="h-3 w-3 mr-1" /> Handover</>
                                                        )}
                                                    </span>
                                                </td>
                                                <td className={`py-3 px-2 text-right font-medium ${
                                                    txn.type === 'credit' ? 'text-green-600' : 'text-blue-600'
                                                }`}>
                                                    {txn.type === 'credit' ? '+' : '-'}{formatCurrency(txn.amount)}
                                                </td>
                                                <td className="py-3 px-2 text-right text-muted-foreground">
                                                    {formatCurrency(txn.balance_after)}
                                                </td>
                                                <td className="py-3 px-2 text-sm text-muted-foreground max-w-xs truncate">
                                                    {txn.notes || '-'}
                                                </td>
                                                <td className="py-3 px-2 text-sm text-muted-foreground">
                                                    {txn.received_by || '-'}
                                                </td>
                                            </tr>
                                        ))}
                                    </tbody>
                                </table>
                            </div>
                        )}

                        {/* Pagination */}
                        {transactions.links && transactions.links.length > 3 && (
                            <div className="mt-4 flex items-center justify-center gap-2">
                                {transactions.links.map((link: any, index: number) => (
                                    <Link
                                        key={index}
                                        href={link.url || '#'}
                                        className={`px-3 py-1 rounded ${
                                            link.active
                                                ? 'bg-primary text-primary-foreground'
                                                : 'bg-muted hover:bg-muted/80'
                                        }`}
                                        dangerouslySetInnerHTML={{ __html: link.label }}
                                    />
                                ))}
                            </div>
                        )}
                    </CardContent>
                </Card>
            </div>

            {/* Handover Modal */}
            <Dialog open={handoverModalOpen} onOpenChange={setHandoverModalOpen}>
                <DialogContent>
                    <DialogHeader>
                        <DialogTitle>Record Cash Handover</DialogTitle>
                        <DialogDescription>
                            Record cash received from {collector.name}
                        </DialogDescription>
                    </DialogHeader>
                    <form onSubmit={handleRecordHandover} className="space-y-4">
                        <div className="bg-amber-50 dark:bg-amber-900/20 p-3 rounded-lg">
                            <div className="text-sm text-muted-foreground">Available Balance</div>
                            <div className="text-xl font-bold text-amber-600">
                                {formatCurrency(wallet.balance)}
                            </div>
                        </div>

                        <div className="space-y-2">
                            <Label htmlFor="amount">Amount to Receive</Label>
                            <div className="relative">
                                <IndianRupee className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                                <Input
                                    id="amount"
                                    type="number"
                                    step="0.01"
                                    min="0.01"
                                    max={wallet.balance}
                                    value={amount}
                                    onChange={(e) => setAmount(e.target.value)}
                                    className="pl-9"
                                    placeholder="0.00"
                                    required
                                />
                            </div>
                        </div>

                        <div className="space-y-2">
                            <Label htmlFor="notes">Notes (Optional)</Label>
                            <Textarea
                                id="notes"
                                value={notes}
                                onChange={(e) => setNotes(e.target.value)}
                                placeholder="Any additional notes..."
                                rows={2}
                            />
                        </div>

                        <div className="flex gap-3 justify-end">
                            <Button type="button" variant="outline" onClick={() => setHandoverModalOpen(false)}>
                                Cancel
                            </Button>
                            <Button type="submit" disabled={processing || !amount}>
                                {processing ? 'Recording...' : 'Record Handover'}
                            </Button>
                        </div>
                    </form>
                </DialogContent>
            </Dialog>
        </SocietyAdminLayout>
    );
}
