import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { GlassCard } from '@/components/ui/glass-card';
import { PageHeader } from '@/components/ui/page-header';
import {
    ModernTable,
    ModernTableHeader,
    ModernTableBody,
    ModernTableHead,
    ModernTableRow,
    ModernTableCell,
} from '@/components/ui/modern-table';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import {
    Dialog,
    DialogContent,
    DialogDescription,
    DialogFooter,
    DialogHeader,
    DialogTitle,
    DialogTrigger,
} from "@/components/ui/dialog";
import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from "@/components/ui/select";
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link, router, useForm } from '@inertiajs/react';
import { Building2, Plus, Search, Edit, Trash2, Eye, Home, Layers } from 'lucide-react';
import { useState } from 'react';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Buildings',
        href: '/societies/{society}/buildings',
    },
];

interface Building {
    id: number;
    name: string;
    building_no: string | null;
    floors: number;
    total_units: number | null;
    address: string | null;
    units_count: number;
    type: string;
    created_at: string;
}

interface IndexProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    buildings: {
        data: Building[];
        links: any;
        meta: any;
    };
    filters: {
        search?: string;
    };
    stats: {
        total_buildings: number;
        total_floors: number;
        total_units: number;
    };
}

export default function Index({ society, buildings, filters, stats }: IndexProps) {
    const [search, setSearch] = useState(filters.search || '');
    
    // Modal States
    const [isCreateOpen, setIsCreateOpen] = useState(false);
    const [isEditOpen, setIsEditOpen] = useState(false);
    const [isViewOpen, setIsViewOpen] = useState(false);
    const [editingBuilding, setEditingBuilding] = useState<Building | null>(null);
    const [viewingBuilding, setViewingBuilding] = useState<Building | null>(null);

    // Create Form
    const createForm = useForm({
        name: '',
        building_no: '',
        floors: '',
        total_units: '',
        address: '',
        type: 'building',
    });

    // Edit Form
    const editForm = useForm({
        name: '',
        building_no: '',
        floors: '',
        total_units: '',
        address: '',
        type: 'building',
    });

    const handleSearch = () => {
        router.get(`/societies/${society.id}/buildings`, { search }, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const handleDelete = (id: number) => {
        if (confirm('Are you sure you want to delete this building?')) {
            router.delete(`/societies/${society.id}/buildings/${id}`, {
                preserveScroll: true,
            });
        }
    };

    const handleCreateSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        createForm.post(`/societies/${society.id}/buildings`, {
            onSuccess: () => {
                setIsCreateOpen(false);
                createForm.reset();
            },
        });
    };

    const openEditModal = (building: Building) => {
        setEditingBuilding(building);
        editForm.setData({
            name: building.name,
            building_no: building.building_no || '',
            floors: building.floors.toString(),
            total_units: building.total_units ? building.total_units.toString() : '',
            address: building.address || '',
            type: building.type || 'building',
        });
        setIsEditOpen(true);
    };

    const openViewModal = (building: Building) => {
        setViewingBuilding(building);
        setIsViewOpen(true);
    };

    const handleEditSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        if (!editingBuilding) return;
        
        editForm.put(`/societies/${society.id}/buildings/${editingBuilding.id}`, {
            onSuccess: () => {
                setIsEditOpen(false);
                setEditingBuilding(null);
                editForm.reset();
            },
        });
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Buildings" />
            <div className="flex flex-1 flex-col gap-8 p-6 md:p-8 overflow-x-hidden bg-muted/10">
                <PageHeader
                    title="Buildings"
                    description="Manage all buildings in your society"
                    breadcrumbs={breadcrumbs}
                    actions={
                        <Button onClick={() => setIsCreateOpen(true)} className="bg-primary hover:bg-primary/90 shadow-lg shadow-primary/20">
                            <Plus className="mr-2 h-4 w-4" />
                            Add Building
                        </Button>
                    }
                />
                    


                    {/* Edit Dialog */}
                    <Dialog open={isEditOpen} onOpenChange={setIsEditOpen}>
                        <DialogContent className="sm:max-w-[500px]">
                            <DialogHeader>
                                <DialogTitle>Edit Building</DialogTitle>
                                <DialogDescription>
                                    Update building details.
                                </DialogDescription>
                            </DialogHeader>
                            <form onSubmit={handleEditSubmit} className="space-y-4 py-4">
                                <div className="grid grid-cols-2 gap-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="edit_name">Building Name</Label>
                                        <Input
                                            id="edit_name"
                                            value={editForm.data.name}
                                            onChange={(e) => editForm.setData('name', e.target.value)}
                                            placeholder="e.g. Block A"
                                            required
                                        />
                                        {editForm.errors.name && <p className="text-sm text-red-500">{editForm.errors.name}</p>}
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="edit_building_no">Building No</Label>
                                        <Input
                                            id="edit_building_no"
                                            value={editForm.data.building_no}
                                            onChange={(e) => editForm.setData('building_no', e.target.value)}
                                            placeholder="e.g. 12"
                                        />
                                        {editForm.errors.building_no && <p className="text-sm text-red-500">{editForm.errors.building_no}</p>}
                                    </div>
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="edit_type">Type</Label>
                                    <Select value={editForm.data.type} onValueChange={(val) => editForm.setData('type', val)}>
                                        <SelectTrigger>
                                            <SelectValue placeholder="Select Type" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="building">Building</SelectItem>
                                            <SelectItem value="extension">Extension</SelectItem>
                                            <SelectItem value="bungalow">Bungalow</SelectItem>
                                        </SelectContent>
                                    </Select>
                                    {editForm.errors.type && <p className="text-sm text-red-500">{editForm.errors.type}</p>}
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="edit_floors">Number of Floors</Label>
                                    <Input
                                        id="edit_floors"
                                        type="number"
                                        min="1"
                                        value={editForm.data.floors}
                                        onChange={(e) => editForm.setData('floors', e.target.value)}
                                        placeholder="e.g. 5"
                                        required
                                    />
                                    {editForm.errors.floors && <p className="text-sm text-red-500">{editForm.errors.floors}</p>}
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="edit_total_units">Total Units (Capacity)</Label>
                                    <Input
                                        id="edit_total_units"
                                        type="number"
                                        min="0"
                                        value={editForm.data.total_units}
                                        onChange={(e) => editForm.setData('total_units', e.target.value)}
                                        placeholder="e.g. 20"
                                    />
                                    {editForm.errors.total_units && <p className="text-sm text-red-500">{editForm.errors.total_units}</p>}
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="edit_address">Address / Location</Label>
                                    <Input
                                        id="edit_address"
                                        value={editForm.data.address}
                                        onChange={(e) => editForm.setData('address', e.target.value)}
                                        placeholder="e.g. Near Main Gate"
                                    />
                                    {editForm.errors.address && <p className="text-sm text-red-500">{editForm.errors.address}</p>}
                                </div>

                                <DialogFooter>
                                    <Button type="button" variant="outline" onClick={() => setIsEditOpen(false)}>Cancel</Button>
                                    <Button type="submit" disabled={editForm.processing}>
                                        {editForm.processing ? 'Saving...' : 'Save Changes'}
                                    </Button>
                                </DialogFooter>
                            </form>
                        </DialogContent>
                    </Dialog>

                    {/* View Dialog */}
                    <Dialog open={isViewOpen} onOpenChange={setIsViewOpen}>
                        <DialogContent className="sm:max-w-[500px]">
                            <DialogHeader>
                                <DialogTitle>Building Details</DialogTitle>
                                <DialogDescription>
                                    View detailed information about this building.
                                </DialogDescription>
                            </DialogHeader>
                            {viewingBuilding && (
                                <div className="space-y-6 py-4">
                                    <div className="grid grid-cols-2 gap-4">
                                        <div>
                                            <h4 className="text-sm font-medium text-muted-foreground">Building Name</h4>
                                            <p className="text-lg font-semibold">{viewingBuilding.name}</p>
                                        </div>
                                        <div>
                                            <h4 className="text-sm font-medium text-muted-foreground">Building No</h4>
                                            <p className="text-lg font-semibold">{viewingBuilding.building_no || '-'}</p>
                                        </div>
                                        <div>
                                            <h4 className="text-sm font-medium text-muted-foreground">Type</h4>
                                            <p className="text-base capitalize">{viewingBuilding.type}</p>
                                        </div>
                                        <div>
                                            <h4 className="text-sm font-medium text-muted-foreground">Floors</h4>
                                            <p className="text-base">{viewingBuilding.floors}</p>
                                        </div>
                                        <div>
                                            <h4 className="text-sm font-medium text-muted-foreground">Total Units</h4>
                                            <p className="text-base">{viewingBuilding.total_units || '-'}</p>
                                        </div>
                                        <div>
                                            <h4 className="text-sm font-medium text-muted-foreground">Registered Units</h4>
                                            <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400">
                                                {viewingBuilding.units_count} Units
                                            </span>
                                        </div>
                                    </div>
                                    <div>
                                        <h4 className="text-sm font-medium text-muted-foreground">Address</h4>
                                        <p className="text-base mt-1">{viewingBuilding.address || '-'}</p>
                                    </div>
                                </div>
                            )}
                            <DialogFooter>
                                <Button onClick={() => setIsViewOpen(false)}>Close</Button>
                            </DialogFooter>
                        </DialogContent>
                    </Dialog>

                    {/* Create Dialog */}
                    <Dialog open={isCreateOpen} onOpenChange={setIsCreateOpen}>
                        <DialogContent className="sm:max-w-[500px]">
                            <DialogHeader>
                                <DialogTitle>Add New Building</DialogTitle>
                                <DialogDescription>
                                    Create a new building structure in your society.
                                </DialogDescription>
                            </DialogHeader>
                            <form onSubmit={handleCreateSubmit} className="space-y-4 py-4">
                                <div className="grid grid-cols-2 gap-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="name">Building Name</Label>
                                        <Input
                                            id="name"
                                            value={createForm.data.name}
                                            onChange={(e) => createForm.setData('name', e.target.value)}
                                            placeholder="e.g. Block A"
                                            required
                                        />
                                        {createForm.errors.name && <p className="text-sm text-red-500">{createForm.errors.name}</p>}
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="building_no">Building No</Label>
                                        <Input
                                            id="building_no"
                                            value={createForm.data.building_no}
                                            onChange={(e) => createForm.setData('building_no', e.target.value)}
                                            placeholder="e.g. 12"
                                        />
                                        {createForm.errors.building_no && <p className="text-sm text-red-500">{createForm.errors.building_no}</p>}
                                    </div>
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="type">Type</Label>
                                    <Select value={createForm.data.type} onValueChange={(val) => createForm.setData('type', val)}>
                                        <SelectTrigger>
                                            <SelectValue placeholder="Select Type" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="building">Building</SelectItem>
                                            <SelectItem value="extension">Extension</SelectItem>
                                            <SelectItem value="bungalow">Bungalow</SelectItem>
                                        </SelectContent>
                                    </Select>
                                    {createForm.errors.type && <p className="text-sm text-red-500">{createForm.errors.type}</p>}
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="floors">Number of Floors</Label>
                                    <Input
                                        id="floors"
                                        type="number"
                                        min="1"
                                        value={createForm.data.floors}
                                        onChange={(e) => createForm.setData('floors', e.target.value)}
                                        placeholder="e.g. 5"
                                        required
                                    />
                                    {createForm.errors.floors && <p className="text-sm text-red-500">{createForm.errors.floors}</p>}
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="total_units">Total Units (Capacity)</Label>
                                    <Input
                                        id="total_units"
                                        type="number"
                                        min="0"
                                        value={createForm.data.total_units}
                                        onChange={(e) => createForm.setData('total_units', e.target.value)}
                                        placeholder="e.g. 20"
                                    />
                                    {createForm.errors.total_units && <p className="text-sm text-red-500">{createForm.errors.total_units}</p>}
                                </div>

                                <div className="space-y-2">
                                    <Label htmlFor="address">Address / Location</Label>
                                    <Input
                                        id="address"
                                        value={createForm.data.address}
                                        onChange={(e) => createForm.setData('address', e.target.value)}
                                        placeholder="e.g. Near Main Gate"
                                    />
                                    {createForm.errors.address && <p className="text-sm text-red-500">{createForm.errors.address}</p>}
                                </div>

                                <DialogFooter>
                                    <Button type="button" variant="outline" onClick={() => setIsCreateOpen(false)}>Cancel</Button>
                                    <Button type="submit" disabled={createForm.processing}>
                                        {createForm.processing ? 'Creating...' : 'Create Building'}
                                    </Button>
                                </DialogFooter>
                            </form>
                        </DialogContent>
                    </Dialog>

                {/* KPI Cards */}
                <div className="grid gap-4 md:grid-cols-3">
                <GlassCard variant="hover" className="border-t-4 border-t-purple-500 py-6 flex flex-col gap-4">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Total Buildings</CardTitle>
                            <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-cyan-400 to-cyan-600 flex items-center justify-center text-white shadow-md shadow-cyan-500/20">
                                <Building2 className="h-5 w-5" />
                            </div>
                        </CardHeader>
                        <CardContent className="p-6">
                            <div className="text-2xl font-bold text-foreground">{stats.total_buildings}</div>
                            <p className="text-xs text-muted-foreground mt-1">
                                Registered structures
                            </p>
                        </CardContent>
                    </GlassCard>

                <GlassCard variant="hover" className="border-t-4 border-t-indigo-500 py-6 flex flex-col gap-4">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Total Floors</CardTitle>
                            <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-indigo-400 to-blue-600 flex items-center justify-center text-white shadow-md shadow-indigo-500/20">
                                <Layers className="h-5 w-5" />
                            </div>
                        </CardHeader>
                        <CardContent className="p-6">
                            <div className="text-2xl font-bold text-foreground">{stats.total_floors}</div>
                            <p className="text-xs text-muted-foreground mt-1">
                                Across all buildings
                            </p>
                        </CardContent>
                    </GlassCard>

                <GlassCard variant="hover" className="border-t-4 border-t-cyan-500 py-6 flex flex-col gap-4">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Total Units</CardTitle>
                            <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-cyan-400 to-teal-600 flex items-center justify-center text-white shadow-md shadow-cyan-500/20">
                                <Home className="h-5 w-5" />
                            </div>
                        </CardHeader>
                        <CardContent className="p-6">
                            <div className="text-2xl font-bold text-foreground">{stats.total_units}</div>
                            <p className="text-xs text-muted-foreground mt-1">
                                {stats.total_buildings > 0 ? Math.round(stats.total_units / stats.total_buildings) : 0} avg units per building
                            </p>
                        </CardContent>
                    </GlassCard>
                </div>

                <Tabs defaultValue="buildings" className="w-full">
                    <TabsList className="grid w-full grid-cols-2 lg:w-[400px] mb-6 p-1 bg-muted/50 rounded-xl">
                        <Link href={`/societies/${society.id}/units`} className="flex-1">
                            <TabsTrigger value="units" className="w-full rounded-lg bg-transparent">Units</TabsTrigger>
                        </Link>
                        <TabsTrigger value="buildings" className="rounded-lg data-[state=active]:bg-white data-[state=active]:text-primary data-[state=active]:shadow-sm font-medium">Buildings</TabsTrigger>
                    </TabsList>
                </Tabs>

                {/* Filters */}
                <GlassCard>
                    <CardContent className="p-6">
                        <div className="flex gap-4">
                            <div className="flex-1">
                                <Label htmlFor="search">Search</Label>
                                <div className="relative mt-1">
                                    <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                    <Input
                                        id="search"
                                        placeholder="Search by name or building number..."
                                        value={search}
                                        onChange={(e) => setSearch(e.target.value)}
                                        onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                        className="pl-9"
                                    />
                                </div>
                            </div>
                            <div className="flex items-end">
                                <Button onClick={handleSearch}>Filter</Button>
                            </div>
                        </div>
                    </CardContent>
                </GlassCard>

                {/* Buildings Table */}
                <div className="flex flex-col gap-4">
                    <div className="rounded-xl border border-border/50 bg-card shadow-sm overflow-hidden">
                        <ModernTable>
                            <ModernTableHeader>
                                <ModernTableRow>
                                    <ModernTableHead>Name</ModernTableHead>
                                    <ModernTableHead>Type</ModernTableHead>
                                    <ModernTableHead>Building No</ModernTableHead>
                                    <ModernTableHead>Floors</ModernTableHead>
                                    <ModernTableHead>Total Units</ModernTableHead>
                                    <ModernTableHead>Units Count</ModernTableHead>
                                    <ModernTableHead>Address</ModernTableHead>
                                    <ModernTableHead className="text-right">Actions</ModernTableHead>
                                </ModernTableRow>
                            </ModernTableHeader>
                            <ModernTableBody>
                                {buildings.data.length > 0 ? (
                                    buildings.data.map((building) => (
                                        <ModernTableRow key={building.id}>
                                            <ModernTableCell>
                                                <div className="flex items-center gap-3">
                                                    <div className="h-8 w-8 rounded-lg bg-cyan-100 dark:bg-cyan-900/30 flex items-center justify-center text-cyan-600 dark:text-cyan-400">
                                                        <Building2 className="h-4 w-4" />
                                                    </div>
                                                    <span className="font-semibold text-foreground">{building.name}</span>
                                                </div>
                                            </ModernTableCell>
                                            <ModernTableCell>
                                                <span className="capitalize px-2 py-1 rounded-md bg-muted text-muted-foreground text-xs font-medium">
                                                    {building.type}
                                                </span>
                                            </ModernTableCell>
                                            <ModernTableCell className="text-muted-foreground">{building.building_no || '-'}</ModernTableCell>
                                            <ModernTableCell className="font-medium">{building.floors}</ModernTableCell>
                                            <ModernTableCell className="text-muted-foreground">{building.total_units || '-'}</ModernTableCell>
                                            <ModernTableCell>
                                                <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400">
                                                    {building.units_count} Units
                                                </span>
                                            </ModernTableCell>
                                            <ModernTableCell className="text-muted-foreground max-w-[200px] truncate" title={building.address || ''}>
                                                {building.address || '-'}
                                            </ModernTableCell>
                                            <ModernTableCell className="text-right">
                                                <div className="flex items-center justify-end gap-2">
                                                    <Button 
                                                        variant="ghost" 
                                                        size="icon" 
                                                        className="h-8 w-8 hover:bg-blue-50 hover:text-blue-600"
                                                        onClick={() => openViewModal(building)}
                                                    >
                                                        <Eye className="h-4 w-4" />
                                                    </Button>
                                                    <Button 
                                                        variant="ghost" 
                                                        size="icon" 
                                                        className="h-8 w-8 hover:bg-amber-50 hover:text-amber-600"
                                                        onClick={() => openEditModal(building)}
                                                    >
                                                        <Edit className="h-4 w-4" />
                                                    </Button>
                                                    <Button
                                                        variant="ghost"
                                                        size="icon"
                                                        className="h-8 w-8 hover:bg-red-50 hover:text-red-600"
                                                        onClick={() => handleDelete(building.id)}
                                                    >
                                                        <Trash2 className="h-4 w-4" />
                                                    </Button>
                                                </div>
                                            </ModernTableCell>
                                        </ModernTableRow>
                                    ))
                                ) : (
                                    <ModernTableRow>
                                        <ModernTableCell colSpan={8} className="h-24 text-center">
                                            <div className="flex flex-col items-center gap-2 text-muted-foreground">
                                                <Building2 className="h-10 w-10 text-muted-foreground/30" />
                                                <p>No buildings found matching your criteria</p>
                                            </div>
                                        </ModernTableCell>
                                    </ModernTableRow>
                                )}
                            </ModernTableBody>
                        </ModernTable>

                        {/* Pagination */}
                        {buildings.links && buildings.links.length > 3 && (
                            <div className="border-t border-border/50 px-6 py-4 flex items-center justify-between bg-muted/10">
                                <div className="text-sm text-muted-foreground">
                                    Showing <span className="font-medium text-foreground">{buildings.meta.from}</span> to <span className="font-medium text-foreground">{buildings.meta.to}</span> of <span className="font-medium text-foreground">{buildings.meta.total}</span> results
                                </div>
                                <div className="flex gap-1">
                                    {buildings.links.map((link: any, index: number) => (
                                        <Link
                                            key={index}
                                            href={link.url || '#'}
                                            className={`px-3 py-1 rounded-md text-sm transition-colors ${
                                                link.active
                                                    ? 'bg-primary text-primary-foreground font-medium shadow-sm'
                                                    : 'bg-background hover:bg-muted text-muted-foreground border border-border/50'
                                            } ${!link.url && 'opacity-50 cursor-not-allowed'}`}
                                            dangerouslySetInnerHTML={{ __html: link.label }}
                                        />
                                    ))}
                                </div>
                            </div>
                        )}
                    </div>
                </div>
            </div>
        </SocietyAdminLayout>
    );
}

