import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link, router, useForm } from '@inertiajs/react';
import { Receipt, Plus, Search, Eye, CheckCircle, AlertCircle, Clock, IndianRupee, Pencil } from 'lucide-react';
import { useState } from 'react';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Checkbox } from '@/components/ui/checkbox';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Bills',
        href: '/societies/{society}/bills',
    },
];

interface BillCycle {
    id: number;
    name: string;
}

interface Resident {
    id: number;
    name: string;
    phone: string;
    unit_id: number;
    unit?: {
        id: number;
        unit_no: string;
    };
}

interface Bill {
    id: number;
    bill_no: string;
    amount: number;
    status: string;
    due_date: string;
    user: {
        id: number;
        name: string;
        phone: string;
        email?: string;
        unit?: {
            id: number;
            unit_no: string;
        };
    };
    bill_cycle: {
        id: number;
        name: string;
    } | null;
    generated_by: {
        id: number;
        name: string;
    } | null;
    created_at: string;
    maintenance: number;
    water: number;
    electricity: number;
    parking: number;
    other_charges: number;
    gst: number;
    discount: number;
}

interface IndexProps {
    society: {
        id: number;
        name: string;
        code?: string;
        address?: string;
        city?: string;
        state?: string;
        pincode?: string;
        phone?: string;
        email?: string;
        gst_no?: string;
        registration_no?: string;
        logo?: string;
    };
    bills: {
        data: Bill[];
        links: any;
        meta: any;
    };
    billCycles: BillCycle[];
    residents: Resident[];
    filters: {
        search?: string;
        status?: string;
        bill_cycle_id?: string;
        user_id?: string;
    };
    stats: {
        total_bills: number;
        pending_amount: number;
        collected_amount: number;
        under_review: number;
    };
    maintenanceStructures: {
        id: number;
        name: string;
        maintenance: number;
        parking: number;
        other_charges: number;
    }[];
}

export default function Index({ society, bills, billCycles, residents, filters, stats, maintenanceStructures }: IndexProps) {
    const [search, setSearch] = useState(filters.search || '');
    const [status, setStatus] = useState(filters.status || 'all');
    const [billCycleId, setBillCycleId] = useState(filters.bill_cycle_id || 'all');
    const [userId, setUserId] = useState(filters.user_id || 'all');
    const [selectedBills, setSelectedBills] = useState<number[]>([]);
    const [isCreateOpen, setIsCreateOpen] = useState(false);
    const [viewBill, setViewBill] = useState<Bill | null>(null);

    const handleSearch = () => {
        const filterParams: any = {};
        if (search) filterParams.search = search;
        if (status && status !== 'all') filterParams.status = status;
        if (billCycleId && billCycleId !== 'all') filterParams.bill_cycle_id = billCycleId;
        if (userId && userId !== 'all') filterParams.user_id = userId;

        router.get(`/societies/${society.id}/bills`, filterParams, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 0,
        }).format(amount);
    };

    const handleSelectAll = (checked: boolean) => {
        if (checked) {
            setSelectedBills(bills.data.filter(b => b.status === 'under_review').map(b => b.id));
        } else {
            setSelectedBills([]);
        }
    };

    const handleSelectBill = (billId: number, checked: boolean) => {
        if (checked) {
            setSelectedBills([...selectedBills, billId]);
        } else {
            setSelectedBills(selectedBills.filter(id => id !== billId));
        }
    };

    const handleApprove = () => {
        if (selectedBills.length === 0) return;

        if (confirm(`Are you sure you want to approve ${selectedBills.length} bills?`)) {
            router.post(`/societies/${society.id}/bills/approve`, {
                bill_ids: selectedBills,
            }, {
                onSuccess: () => setSelectedBills([]),
            });
        }
    };

    // Create Form
    const createForm = useForm({
        user_id: '',
        bill_cycle_id: 'none',
        amount: '', // Auto-calculated in backend or frontend? Let's use individual fields
        maintenance: '',
        water: '',
        electricity: '',
        parking: '',
        other_charges: '',
        gst: '',
        discount: '',
        due_date: new Date(new Date().setDate(new Date().getDate() + 15)).toISOString().split('T')[0],
        status: 'due', // Default for manual creation
    });

    const handleCreateSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        createForm.post(`/societies/${society.id}/bills`, {
            onSuccess: () => {
                setIsCreateOpen(false);
                createForm.reset();
            },
        });
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Bills" />
            <div className="flex h-full flex-1 flex-col gap-6 overflow-x-auto rounded-xl p-6">
                <div className="flex items-center justify-between">
                    <div>
                        <h1 className="text-2xl font-bold">Bills</h1>
                        <p className="text-muted-foreground">
                            Manage all bills in your society
                        </p>
                    </div>
                    <div className="flex gap-2">
                        {selectedBills.length > 0 && (
                            <Button onClick={handleApprove} variant="default" className="bg-green-600 hover:bg-green-700">
                                <CheckCircle className="mr-2 h-4 w-4" />
                                Approve ({selectedBills.length})
                            </Button>
                        )}
                    </div>
                </div>

                {/* KPI Cards */}
                <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Total Bills</CardTitle>
                            <Receipt className="h-4 w-4 text-muted-foreground" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.total_bills}</div>
                            <p className="text-xs text-muted-foreground">All time</p>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Pending Amount</CardTitle>
                            <AlertCircle className="h-4 w-4 text-red-500" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{formatCurrency(stats.pending_amount)}</div>
                            <p className="text-xs text-muted-foreground">Due & Overdue</p>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Collected</CardTitle>
                            <IndianRupee className="h-4 w-4 text-green-500" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{formatCurrency(stats.collected_amount)}</div>
                            <p className="text-xs text-muted-foreground">Paid bills</p>
                        </CardContent>
                    </Card>
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Under Review</CardTitle>
                            <Clock className="h-4 w-4 text-yellow-500" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.under_review}</div>
                            <p className="text-xs text-muted-foreground">Waiting for approval</p>
                        </CardContent>
                    </Card>
                </div>

                {/* Filters */}
                <Card>
                    <CardContent className="pt-6">
                        <div className="flex gap-4">
                            <div className="flex-1">
                                <Label htmlFor="search">Search</Label>
                                <div className="relative mt-1">
                                    <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                    <Input
                                        id="search"
                                        placeholder="Search by bill number, resident name or phone..."
                                        value={search}
                                        onChange={(e) => setSearch(e.target.value)}
                                        onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                        className="pl-9"
                                    />
                                </div>
                            </div>
                            <div className="w-48">
                                <Label htmlFor="status">Status</Label>
                                <Select value={status} onValueChange={setStatus}>
                                    <SelectTrigger id="status" className="mt-1">
                                        <SelectValue placeholder="All Status" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Status</SelectItem>
                                        <SelectItem value="draft">Draft</SelectItem>
                                        <SelectItem value="under_review">Under Review</SelectItem>
                                        <SelectItem value="due">Due</SelectItem>
                                        <SelectItem value="paid">Paid</SelectItem>
                                        <SelectItem value="overdue">Overdue</SelectItem>
                                        <SelectItem value="cancelled">Cancelled</SelectItem>
                                    </SelectContent>
                                </Select>
                            </div>
                            <div className="w-48">
                                <Label htmlFor="bill_cycle_id">Bill Cycle</Label>
                                <Select value={billCycleId} onValueChange={setBillCycleId}>
                                    <SelectTrigger id="bill_cycle_id" className="mt-1">
                                        <SelectValue placeholder="All Cycles" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Cycles</SelectItem>
                                        {billCycles.map((cycle) => (
                                            <SelectItem key={cycle.id} value={cycle.id.toString()}>
                                                {cycle.name}
                                            </SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                            </div>
                            <div className="w-48">
                                <Label htmlFor="user_id">Resident</Label>
                                <Select value={userId} onValueChange={setUserId}>
                                    <SelectTrigger id="user_id" className="mt-1">
                                        <SelectValue placeholder="All Residents" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Residents</SelectItem>
                                        {residents.map((resident) => (
                                            <SelectItem key={resident.id} value={resident.id.toString()}>
                                                {resident.name}
                                            </SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                            </div>
                            <div className="flex items-end">
                                <Button onClick={handleSearch}>Filter</Button>
                            </div>
                        </div>
                    </CardContent>
                </Card>

                {/* Bills Table */}
                <Card>
                    <CardHeader>
                        <CardTitle>All Bills</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="overflow-x-auto">
                            <table className="w-full">
                                <thead>
                                    <tr className="border-b">
                                        <th className="px-4 py-3 text-left">
                                            <Checkbox
                                                checked={selectedBills.length > 0 && selectedBills.length === bills.data.filter(b => b.status === 'under_review').length}
                                                onCheckedChange={(checked) => handleSelectAll(checked as boolean)}
                                                disabled={bills.data.filter(b => b.status === 'under_review').length === 0}
                                            />
                                        </th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Bill No</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Resident</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Bill Cycle</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Amount</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Due Date</th>
                                        <th className="px-4 py-3 text-left text-sm font-medium">Status</th>
                                        <th className="px-4 py-3 text-right text-sm font-medium">Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {bills.data.length > 0 ? (
                                        bills.data.map((bill) => (
                                            <tr key={bill.id} className="border-b hover:bg-muted/50">
                                                <td className="px-4 py-3">
                                                    {bill.status === 'under_review' && (
                                                        <Checkbox
                                                            checked={selectedBills.includes(bill.id)}
                                                            onCheckedChange={(checked) => handleSelectBill(bill.id, checked as boolean)}
                                                        />
                                                    )}
                                                </td>
                                                <td className="px-4 py-3">
                                                    <div className="flex items-center gap-2">
                                                        <Receipt className="h-4 w-4 text-muted-foreground" />
                                                        <span className="font-medium">{bill.bill_no}</span>
                                                    </div>
                                                </td>
                                                <td className="px-4 py-3">
                                                    <div>
                                                        <div className="font-medium">{bill.user.name}</div>
                                                        <div className="text-xs text-muted-foreground">
                                                            {bill.user.phone}
                                                        </div>
                                                    </div>
                                                </td>
                                                <td className="px-4 py-3 text-sm text-muted-foreground">
                                                    {bill.bill_cycle?.name || '-'}
                                                </td>
                                                <td className="px-4 py-3 text-sm font-medium">
                                                    {formatCurrency(bill.amount)}
                                                </td>
                                                <td className="px-4 py-3 text-sm text-muted-foreground">
                                                    {new Date(bill.due_date).toLocaleDateString()}
                                                </td>
                                                <td className="px-4 py-3">
                                                    <span
                                                        className={`inline-flex rounded-full px-2 py-1 text-xs font-medium ${
                                                            bill.status === 'paid'
                                                                ? 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200'
                                                                : bill.status === 'overdue'
                                                                  ? 'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200'
                                                                  : bill.status === 'due'
                                                                    ? 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200'
                                                                    : bill.status === 'under_review'
                                                                      ? 'bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-200'
                                                                      : bill.status === 'cancelled'
                                                                        ? 'bg-gray-100 text-gray-800 dark:bg-gray-900 dark:text-gray-200'
                                                                        : 'bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200'
                                                        }`}
                                                    >
                                                        {bill.status === 'under_review' ? 'Under Review' : bill.status}
                                                    </span>
                                                </td>
                                                <td className="px-4 py-3">
                                                    <div className="flex items-center justify-end gap-2">
                                                        <Dialog>
                                                            <DialogTrigger asChild>
                                                                <Button variant="ghost" size="sm" onClick={() => setViewBill(bill)}>
                                                                    <Eye className="h-4 w-4" />
                                                                </Button>
                                                            </DialogTrigger>
                                                            <DialogContent className="max-w-3xl">
                                                                <div className="p-6 bg-white text-black" id="invoice-content">
                                                                    {/* Header */}
                                                                    <div className="flex justify-between items-start border-b pb-6 mb-6">
                                                                        <div className="flex gap-4">
                                                                            {society.logo && (
                                                                                <img src={`/storage/${society.logo}`} alt="Logo" className="h-16 w-16 object-contain" />
                                                                            )}
                                                                            <div>
                                                                                <h2 className="text-2xl font-bold text-gray-900">{society.name}</h2>
                                                                                <div className="text-sm text-gray-600 mt-1 space-y-0.5">
                                                                                    {society.address && <p>{society.address}</p>}
                                                                                    <p>
                                                                                        {[society.city, society.state, society.pincode].filter(Boolean).join(', ')}
                                                                                    </p>
                                                                                    {society.phone && <p>Phone: {society.phone}</p>}
                                                                                    {society.email && <p>Email: {society.email}</p>}
                                                                                </div>
                                                                            </div>
                                                                        </div>
                                                                        <div className="text-right">
                                                                            <h1 className="text-3xl font-bold text-gray-200 uppercase tracking-widest">Invoice</h1>
                                                                            <div className="mt-2 text-sm text-gray-600">
                                                                                <p><span className="font-medium">Invoice #:</span> {bill.bill_no}</p>
                                                                                <p><span className="font-medium">Date:</span> {new Date(bill.created_at).toLocaleDateString()}</p>
                                                                                <p><span className="font-medium">Due Date:</span> {new Date(bill.due_date).toLocaleDateString()}</p>
                                                                                <p className="mt-1">
                                                                                    <span className={`inline-flex rounded-full px-2 py-0.5 text-xs font-medium capitalize ${
                                                                                        bill.status === 'paid' ? 'bg-green-100 text-green-800' :
                                                                                        bill.status === 'overdue' ? 'bg-red-100 text-red-800' :
                                                                                        'bg-blue-100 text-blue-800'
                                                                                    }`}>
                                                                                        {bill.status.replace('_', ' ')}
                                                                                    </span>
                                                                                </p>
                                                                            </div>
                                                                        </div>
                                                                    </div>

                                                                    {/* Bill To */}
                                                                    <div className="mb-8">
                                                                        <h3 className="text-sm font-semibold text-gray-500 uppercase tracking-wider mb-2">Bill To</h3>
                                                                        <div className="text-gray-900">
                                                                            <p className="font-bold text-lg">{bill.user.name}</p>
                                                                            <p className="text-sm">Unit: {bill.user.unit?.unit_no || '-'}</p>
                                                                            <p className="text-sm">{bill.user.phone}</p>
                                                                            <p className="text-sm">{bill.user.email}</p>
                                                                        </div>
                                                                    </div>

                                                                    {/* Line Items */}
                                                                    <div className="mb-8">
                                                                        <table className="w-full text-sm">
                                                                            <thead>
                                                                                <tr className="border-b-2 border-gray-200">
                                                                                    <th className="text-left py-2 font-semibold text-gray-600">Description</th>
                                                                                    <th className="text-right py-2 font-semibold text-gray-600">Amount</th>
                                                                                </tr>
                                                                            </thead>
                                                                            <tbody className="divide-y divide-gray-100">
                                                                                <tr>
                                                                                    <td className="py-3">Maintenance Charges</td>
                                                                                    <td className="py-3 text-right">{formatCurrency(bill.maintenance)}</td>
                                                                                </tr>
                                                                                {bill.parking > 0 && (
                                                                                    <tr>
                                                                                        <td className="py-3">Parking Charges</td>
                                                                                        <td className="py-3 text-right">{formatCurrency(bill.parking)}</td>
                                                                                    </tr>
                                                                                )}
                                                                                {bill.water > 0 && (
                                                                                    <tr>
                                                                                        <td className="py-3">Water Charges</td>
                                                                                        <td className="py-3 text-right">{formatCurrency(bill.water)}</td>
                                                                                    </tr>
                                                                                )}
                                                                                {bill.electricity > 0 && (
                                                                                    <tr>
                                                                                        <td className="py-3">Electricity Charges</td>
                                                                                        <td className="py-3 text-right">{formatCurrency(bill.electricity)}</td>
                                                                                    </tr>
                                                                                )}
                                                                                {bill.other_charges > 0 && (
                                                                                    <tr>
                                                                                        <td className="py-3">Other Charges</td>
                                                                                        <td className="py-3 text-right">{formatCurrency(bill.other_charges)}</td>
                                                                                    </tr>
                                                                                )}
                                                                                {bill.gst > 0 && (
                                                                                    <tr>
                                                                                        <td className="py-3 text-gray-600">GST</td>
                                                                                        <td className="py-3 text-right text-gray-600">{formatCurrency(bill.gst)}</td>
                                                                                    </tr>
                                                                                )}
                                                                                {bill.discount > 0 && (
                                                                                    <tr>
                                                                                        <td className="py-3 text-green-600">Discount</td>
                                                                                        <td className="py-3 text-right text-green-600">-{formatCurrency(bill.discount)}</td>
                                                                                    </tr>
                                                                                )}
                                                                            </tbody>
                                                                            <tfoot>
                                                                                <tr className="border-t-2 border-gray-200">
                                                                                    <td className="pt-4 font-bold text-gray-900 text-lg">Total Amount</td>
                                                                                    <td className="pt-4 font-bold text-gray-900 text-lg text-right">{formatCurrency(bill.amount)}</td>
                                                                                </tr>
                                                                            </tfoot>
                                                                        </table>
                                                                    </div>

                                                                    {/* Footer */}
                                                                    <div className="border-t pt-6 text-sm text-gray-500">
                                                                        <div className="flex justify-between items-end">
                                                                            <div>
                                                                                <p className="font-medium text-gray-900 mb-1">Payment Details:</p>
                                                                                <p>Please pay by the due date to avoid late fees.</p>
                                                                                {society.gst_no && <p>GST No: {society.gst_no}</p>}
                                                                                {society.registration_no && <p>Reg No: {society.registration_no}</p>}
                                                                            </div>
                                                                            <div className="text-right">
                                                                                <p className="font-medium text-gray-900">Authorized Signatory</p>
                                                                                <p className="mt-8 border-t border-gray-300 inline-block px-8 pt-1">{society.name}</p>
                                                                            </div>
                                                                        </div>
                                                                    </div>
                                                                    
                                                                    {/* Linked Payments - Show only if bill is paid/partially paid */}
                                                                    {(bill.status === 'paid' || (bill as any).payment) && (
                                                                         <div className="mt-6 border-t pt-6">
                                                                             <h4 className="text-sm font-semibold text-gray-900 uppercase tracking-wider mb-2">Payment History</h4>
                                                                             {/* Check if payment is an object (single) or array (multiple in future) - current API returns single 'payment' relation or null? 
                                                                                 Controller says: 'payment'. Let's check relation. Usually hasOne or hasMany? 
                                                                                 If hasOne, it's an object. If hasMany, array.
                                                                                 Assuming 'payment' is the latest or single payment for now based on BillController load.
                                                                             */}
                                                                             {(bill as any).payment ? (
                                                                                 <div className="bg-green-50 p-3 rounded-md border border-green-100 flex justify-between items-center">
                                                                                     <div>
                                                                                         <p className="text-sm font-medium text-green-800">Payment Received</p>
                                                                                         <p className="text-xs text-green-600">
                                                                                             Ref: {(bill as any).payment.payment_no} • {new Date((bill as any).payment.created_at).toLocaleDateString()}
                                                                                         </p>
                                                                                     </div>
                                                                                     <div className="text-right">
                                                                                         <p className="text-sm font-bold text-green-800">{formatCurrency((bill as any).payment.amount)}</p>
                                                                                         <p className="text-xs text-green-600 capitalize">{(bill as any).payment.payment_method}</p>
                                                                                     </div>
                                                                                 </div>
                                                                             ) : (
                                                                                 <p className="text-sm text-gray-500">No payment details linked.</p>
                                                                             )}
                                                                         </div>
                                                                    )}
                                                                </div>
                                                                <DialogFooter className="sm:justify-between px-6 pb-6">
                                                                    <Button variant="outline" onClick={() => window.print()}>
                                                                        Print Invoice
                                                                    </Button>
                                                                    <Button onClick={() => setViewBill(null)}>Close</Button>
                                                                </DialogFooter>
                                                            </DialogContent>
                                                        </Dialog>

                                                        {bill.status === 'under_review' && (
                                                            <Dialog>
                                                                <DialogTrigger asChild>
                                                                    <Button variant="ghost" size="sm">
                                                                        <Pencil className="h-4 w-4" />
                                                                    </Button>
                                                                </DialogTrigger>
                                                                <DialogContent className="max-w-5xl w-[90vw]">
                                                                    <DialogHeader>
                                                                        <DialogTitle>Edit Bill #{bill.bill_no}</DialogTitle>
                                                                        <DialogDescription>
                                                                            Update bill details before approval.
                                                                        </DialogDescription>
                                                                    </DialogHeader>
                                                                    <EditBillForm bill={bill} society={society} />
                                                                </DialogContent>
                                                            </Dialog>
                                                        )}
                                                    </div>
                                                </td>
                                            </tr>
                                        ))
                                    ) : (
                                        <tr>
                                            <td colSpan={8} className="px-4 py-8 text-center text-muted-foreground">
                                                No bills found
                                            </td>
                                        </tr>
                                    )}
                                </tbody>
                            </table>
                        </div>

                        {/* Pagination */}
                        {bills.links && bills.links.length > 3 && (
                            <div className="mt-4 flex items-center justify-between">
                                <div className="text-sm text-muted-foreground">
                                    Showing {bills.meta.from} to {bills.meta.to} of {bills.meta.total} results
                                </div>
                                <div className="flex gap-2">
                                    {bills.links.map((link: any, index: number) => (
                                        <Link
                                            key={index}
                                            href={link.url || '#'}
                                            className={`px-3 py-1 rounded ${
                                                link.active
                                                    ? 'bg-primary text-primary-foreground'
                                                    : 'bg-muted hover:bg-muted/80'
                                            }`}
                                            dangerouslySetInnerHTML={{ __html: link.label }}
                                        />
                                    ))}
                                </div>
                            </div>
                        )}
                    </CardContent>
                </Card>
            </div>
        </SocietyAdminLayout>
    );
}

function EditBillForm({ bill, society }: { bill: Bill; society: any }) {
    const form = useForm({
        user_id: bill.user.id,
        status: bill.status,
        maintenance: bill.maintenance,
        water: bill.water,
        electricity: bill.electricity,
        parking: bill.parking,
        other_charges: bill.other_charges,
        gst: bill.gst,
        discount: bill.discount,
        due_date: bill.due_date,
    });

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        form.put(`/societies/${society.id}/bills/${bill.id}`, {
            onSuccess: () => {
                // Close dialog logic handled by parent re-render or separate state if needed
                // For simplicity, we rely on Inertia reload
                document.dispatchEvent(new KeyboardEvent('keydown', {'key': 'Escape'}));
            },
        });
    };

    return (
        <form onSubmit={handleSubmit} className="grid gap-4 py-4">
            <div className="grid grid-cols-3 gap-4">
                <div className="space-y-2">
                    <Label htmlFor="edit-maintenance">Maintenance</Label>
                    <Input
                        id="edit-maintenance"
                        type="number"
                        value={form.data.maintenance}
                        onChange={(e) => form.setData('maintenance', parseFloat(e.target.value) || 0)}
                    />
                </div>
                <div className="space-y-2">
                    <Label htmlFor="edit-water">Water</Label>
                    <Input
                        id="edit-water"
                        type="number"
                        value={form.data.water}
                        onChange={(e) => form.setData('water', parseFloat(e.target.value) || 0)}
                    />
                </div>
                <div className="space-y-2">
                    <Label htmlFor="edit-electricity">Electricity</Label>
                    <Input
                        id="edit-electricity"
                        type="number"
                        value={form.data.electricity}
                        onChange={(e) => form.setData('electricity', parseFloat(e.target.value) || 0)}
                    />
                </div>
            </div>

            <div className="grid grid-cols-3 gap-4">
                <div className="space-y-2">
                    <Label htmlFor="edit-parking">Parking</Label>
                    <Input
                        id="edit-parking"
                        type="number"
                        value={form.data.parking}
                        onChange={(e) => form.setData('parking', parseFloat(e.target.value) || 0)}
                    />
                </div>
                <div className="space-y-2">
                    <Label htmlFor="edit-other">Other Charges</Label>
                    <Input
                        id="edit-other"
                        type="number"
                        value={form.data.other_charges}
                        onChange={(e) => form.setData('other_charges', parseFloat(e.target.value) || 0)}
                    />
                </div>
                <div className="space-y-2">
                    <Label htmlFor="edit-discount">Discount</Label>
                    <Input
                        id="edit-discount"
                        type="number"
                        value={form.data.discount}
                        onChange={(e) => form.setData('discount', parseFloat(e.target.value) || 0)}
                    />
                </div>
            </div>

            <div className="grid grid-cols-3 gap-4">
                <div className="space-y-2">
                    <Label htmlFor="edit-gst">GST</Label>
                    <Input
                        id="edit-gst"
                        type="number"
                        value={form.data.gst}
                        onChange={(e) => form.setData('gst', parseFloat(e.target.value) || 0)}
                    />
                </div>
                <div className="space-y-2">
                    <Label htmlFor="edit-status">Status</Label>
                    <Select
                        value={form.data.status}
                        onValueChange={(value) => form.setData('status', value)}
                    >
                        <SelectTrigger id="edit-status">
                            <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                            <SelectItem value="under_review">Under Review</SelectItem>
                            <SelectItem value="pending">Pending</SelectItem>
                            <SelectItem value="approved">Approved</SelectItem>
                            <SelectItem value="due">Due</SelectItem>
                            <SelectItem value="overdue">Overdue</SelectItem>
                        </SelectContent>
                    </Select>
                    {form.errors.status && <p className="text-sm text-red-500">{form.errors.status}</p>}
                </div>
                <div className="space-y-2">
                    <Label htmlFor="edit-due-date">Due Date</Label>
                    <Input
                        id="edit-due-date"
                        type="date"
                        value={form.data.due_date}
                        onChange={(e) => form.setData('due_date', e.target.value)}
                        required
                    />
                </div>
            </div>

            <DialogFooter>
                <Button type="button" variant="outline" onClick={() => document.dispatchEvent(new KeyboardEvent('keydown', {'key': 'Escape'}))}>
                    Cancel
                </Button>
                <Button type="submit" disabled={form.processing}>
                    {form.processing ? 'Updating...' : 'Update Bill'}
                </Button>
            </DialogFooter>
        </form>
    );
}

