import { Button } from '@/components/ui/button';
import { CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { GlassCard } from '@/components/ui/glass-card';
import { PageHeader } from '@/components/ui/page-header';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link, router, useForm } from '@inertiajs/react';
import { Calendar, Plus, Play, Eye, Send, CheckCircle, Clock, Archive, FileText, AlertCircle, Loader2 } from 'lucide-react';
import { useState } from 'react';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Input } from '@/components/ui/input';
import {
    ModernTable,
    ModernTableHeader,
    ModernTableBody,
    ModernTableRow,
    ModernTableHead,
    ModernTableCell,
} from '@/components/ui/modern-table';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Bill Cycles',
        href: '/societies/{society}/bill-cycles',
    },
];

interface BillCycle {
    id: number;
    name: string;
    year: number;
    generate_date: string;
    due_date: string;
    status: 'draft' | 'under_review' | 'generated' | 'posted' | 'closed';
    bills_count: number;
    total_amount: number;
    created_at: string;
}

interface IndexProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    billCycles: {
        data: BillCycle[];
        links: any;
        meta: any;
    };
    filters: {
        status?: string;
        year?: string;
    };
    stats: {
        total: number;
        draft: number;
        under_review: number;
        posted: number;
        closed: number;
    };
    maintenanceStructures: {
        id: number;
        name: string;
        maintenance: number;
        parking: number;
        other_charges: number;
    }[];
}

export default function Index({ society, billCycles, filters, stats, maintenanceStructures }: IndexProps) {
    const currentYear = new Date().getFullYear();
    const [status, setStatus] = useState(filters.status || 'all');
    const [year, setYear] = useState(filters.year || currentYear.toString());
    const [isCreateOpen, setIsCreateOpen] = useState(false);

    // Years from 2024 to 2045
    const startYear = 2024;
    const years = Array.from({ length: 22 }, (_, i) => startYear + i);

    const handleFilter = () => {
        const filterParams: any = {};
        if (status && status !== 'all') filterParams.status = status;
        if (year && year !== 'all') filterParams.year = year;

        router.get(`/societies/${society.id}/bill-cycles`, filterParams, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    // Generate Modal State
    const [generateModalOpen, setGenerateModalOpen] = useState(false);
    const [selectedCycleId, setSelectedCycleId] = useState<number | null>(null);
    const [selectedStructureId, setSelectedStructureId] = useState<string>('none');
    const [actionType, setActionType] = useState<'generate' | 'regenerate'>('generate');
    const [isProcessing, setIsProcessing] = useState(false);

    const openGenerateModal = (cycleId: number, type: 'generate' | 'regenerate') => {
        setSelectedCycleId(cycleId);
        setActionType(type);
        setGenerateModalOpen(true);
    };

    const handleGenerateSubmit = () => {
        if (!selectedCycleId) return;
        setIsProcessing(true);

        router.post(`/societies/${society.id}/bill-cycles/${selectedCycleId}/generate`, {
            action: actionType,
            maintenance_structure_id: selectedStructureId === 'none' ? null : selectedStructureId,
        }, {
            preserveScroll: true,
            onSuccess: () => {
                setGenerateModalOpen(false);
                setSelectedCycleId(null);
                setSelectedStructureId('none');
                setIsProcessing(false);
            },
            onError: () => {
                setIsProcessing(false);
            },
        });
    };

    const handlePostBills = (cycleId: number) => {
        if (confirm('Are you sure you want to post all bills to residents? This action cannot be undone.')) {
            router.post(`/societies/${society.id}/bill-cycles/${cycleId}/post`, {}, {
                preserveScroll: true,
            });
        }
    };

    // Create Form
    const createForm = useForm({
        name: '',
        year: currentYear.toString(),
        generate_date: new Date().toISOString().split('T')[0],
        due_date: new Date(new Date().setDate(new Date().getDate() + 15)).toISOString().split('T')[0],
        status: 'draft',
    });

    const handleCreateSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        createForm.post(`/societies/${society.id}/bill-cycles`, {
            onSuccess: () => {
                setIsCreateOpen(false);
                createForm.reset();
            },
        });
    };

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 0,
        }).format(amount);
    };

    const getStatusBadge = (status: string) => {
        const statusConfig: Record<string, { bg: string; text: string; label: string }> = {
            draft: { bg: 'bg-yellow-100 dark:bg-yellow-900/30', text: 'text-yellow-700 dark:text-yellow-300', label: 'Draft' },
            under_review: { bg: 'bg-blue-100 dark:bg-blue-900/30', text: 'text-blue-700 dark:text-blue-300', label: 'Under Review' },
            posted: { bg: 'bg-green-100 dark:bg-green-900/30', text: 'text-green-700 dark:text-green-300', label: 'Posted' },
            closed: { bg: 'bg-gray-100 dark:bg-gray-800', text: 'text-gray-700 dark:text-gray-300', label: 'Closed' },
        };
        const config = statusConfig[status] || statusConfig.draft;
        return (
            <span className={`inline-flex items-center rounded-full px-2.5 py-0.5 text-xs font-medium ${config.bg} ${config.text}`}>
                {config.label}
            </span>
        );
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Bill Cycles" />
            <div className="flex flex-1 flex-col gap-8 p-6 md:p-8 overflow-x-hidden bg-muted/10">
                <PageHeader
                    title="Bill Cycles"
                    description="Manage billing cycles and generate bills for residents"
                    breadcrumbs={breadcrumbs}
                    actions={
                        <Dialog open={isCreateOpen} onOpenChange={setIsCreateOpen}>
                            <DialogTrigger asChild>
                                <Button className="bg-primary hover:bg-primary/90 shadow-lg shadow-primary/20">
                                    <Plus className="mr-2 h-4 w-4" />
                                    Create Cycle
                                </Button>
                            </DialogTrigger>
                            <DialogContent>
                                <DialogHeader>
                                    <DialogTitle>Create Bill Cycle</DialogTitle>
                                    <DialogDescription>
                                        Define a new billing cycle for your society.
                                    </DialogDescription>
                                </DialogHeader>
                                <form onSubmit={handleCreateSubmit} className="space-y-4">
                                    <div className="space-y-2">
                                        <Label htmlFor="name">Cycle Name</Label>
                                        <Input
                                            id="name"
                                            placeholder="e.g. January 2025"
                                            value={createForm.data.name}
                                            onChange={(e) => createForm.setData('name', e.target.value)}
                                            required
                                        />
                                        {createForm.errors.name && <p className="text-sm text-red-500">{createForm.errors.name}</p>}
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="year">Year</Label>
                                        <Select
                                            value={createForm.data.year}
                                            onValueChange={(value) => createForm.setData('year', value)}
                                        >
                                            <SelectTrigger id="year">
                                                <SelectValue placeholder="Select Year" />
                                            </SelectTrigger>
                                            <SelectContent>
                                                {years.map((y) => (
                                                    <SelectItem key={y} value={y.toString()}>
                                                        {y}
                                                    </SelectItem>
                                                ))}
                                            </SelectContent>
                                        </Select>
                                        {createForm.errors.year && <p className="text-sm text-red-500">{createForm.errors.year}</p>}
                                    </div>
                                    <div className="grid grid-cols-2 gap-4">
                                        <div className="space-y-2">
                                            <Label htmlFor="generate_date">Generate Date</Label>
                                            <Input
                                                id="generate_date"
                                                type="date"
                                                value={createForm.data.generate_date}
                                                onChange={(e) => createForm.setData('generate_date', e.target.value)}
                                                required
                                            />
                                            {createForm.errors.generate_date && <p className="text-sm text-red-500">{createForm.errors.generate_date}</p>}
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="due_date">Due Date</Label>
                                            <Input
                                                id="due_date"
                                                type="date"
                                                value={createForm.data.due_date}
                                                onChange={(e) => createForm.setData('due_date', e.target.value)}
                                                required
                                            />
                                            {createForm.errors.due_date && <p className="text-sm text-red-500">{createForm.errors.due_date}</p>}
                                        </div>
                                    </div>
                                    <DialogFooter>
                                        <Button type="button" variant="outline" onClick={() => setIsCreateOpen(false)}>
                                            Cancel
                                        </Button>
                                        <Button type="submit" disabled={createForm.processing}>
                                            {createForm.processing ? 'Creating...' : 'Create Cycle'}
                                        </Button>
                                    </DialogFooter>
                                </form>
                            </DialogContent>
                        </Dialog>
                    }
                />

                {/* KPI Cards */}
                <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-5">
                    <GlassCard variant="hover" className="border-t-4 border-t-blue-500 py-6 flex flex-col gap-4">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Total Cycles</CardTitle>
                            <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-blue-400 to-indigo-600 flex items-center justify-center text-white shadow-md shadow-blue-500/20">
                                <Calendar className="h-5 w-5" />
                            </div>
                        </CardHeader>
                        <CardContent>
                            <div className="text-3xl font-bold">{stats.total}</div>
                            <p className="text-xs text-muted-foreground mt-1">All time</p>
                        </CardContent>
                    </GlassCard>

                    <GlassCard variant="hover" className="border-t-4 border-t-yellow-500 py-6 flex flex-col gap-4">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Draft</CardTitle>
                            <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-yellow-400 to-orange-500 flex items-center justify-center text-white shadow-md shadow-yellow-500/20">
                                <Clock className="h-5 w-5" />
                            </div>
                        </CardHeader>
                        <CardContent>
                            <div className="text-3xl font-bold">{stats.draft}</div>
                            <p className="text-xs text-muted-foreground mt-1">Ready to generate</p>
                        </CardContent>
                    </GlassCard>

                    <GlassCard variant="hover" className="border-t-4 border-t-indigo-500 py-6 flex flex-col gap-4">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Under Review</CardTitle>
                            <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-indigo-400 to-purple-600 flex items-center justify-center text-white shadow-md shadow-indigo-500/20">
                                <AlertCircle className="h-5 w-5" />
                            </div>
                        </CardHeader>
                        <CardContent>
                            <div className="text-3xl font-bold">{stats.under_review}</div>
                            <p className="text-xs text-muted-foreground mt-1">Pending approval</p>
                        </CardContent>
                    </GlassCard>

                    <GlassCard variant="hover" className="border-t-4 border-t-green-500 py-6 flex flex-col gap-4">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Posted</CardTitle>
                            <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-green-400 to-emerald-600 flex items-center justify-center text-white shadow-md shadow-green-500/20">
                                <CheckCircle className="h-5 w-5" />
                            </div>
                        </CardHeader>
                        <CardContent>
                            <div className="text-3xl font-bold">{stats.posted}</div>
                            <p className="text-xs text-muted-foreground mt-1">Sent to residents</p>
                        </CardContent>
                    </GlassCard>

                    <GlassCard variant="hover" className="border-t-4 border-t-gray-500 py-6 flex flex-col gap-4">
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium text-muted-foreground">Closed</CardTitle>
                            <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-gray-400 to-gray-600 flex items-center justify-center text-white shadow-md shadow-gray-500/20">
                                <Archive className="h-5 w-5" />
                            </div>
                        </CardHeader>
                        <CardContent>
                            <div className="text-3xl font-bold">{stats.closed}</div>
                            <p className="text-xs text-muted-foreground mt-1">Past cycles</p>
                        </CardContent>
                    </GlassCard>
                </div>

                {/* Filters */}
                <GlassCard>
                    <CardContent className="py-6">
                        <div className="grid gap-4 md:grid-cols-4 items-end">
                            <div className="space-y-2">
                                <Label htmlFor="filter-status">Status</Label>
                                <Select value={status} onValueChange={setStatus}>
                                    <SelectTrigger id="filter-status" className="bg-background/50">
                                        <SelectValue placeholder="All Status" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Status</SelectItem>
                                        <SelectItem value="draft">Draft</SelectItem>
                                        <SelectItem value="under_review">Under Review</SelectItem>
                                        <SelectItem value="posted">Posted</SelectItem>
                                        <SelectItem value="closed">Closed</SelectItem>
                                    </SelectContent>
                                </Select>
                            </div>
                            <div className="space-y-2">
                                <Label htmlFor="filter-year">Year</Label>
                                <Select value={year} onValueChange={setYear}>
                                    <SelectTrigger id="filter-year" className="bg-background/50">
                                        <SelectValue placeholder="Select Year" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="all">All Years</SelectItem>
                                        {years.map((y) => (
                                            <SelectItem key={y} value={y.toString()}>
                                                {y}
                                            </SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                            </div>
                            <div>
                                <Button onClick={handleFilter} className="bg-primary hover:bg-primary/90">
                                    Filter
                                </Button>
                            </div>
                        </div>
                    </CardContent>
                </GlassCard>

                {/* Bill Cycles Table */}
                <GlassCard>
                    <CardHeader>
                        <CardTitle className="flex items-center gap-2">
                            <FileText className="h-5 w-5" />
                            Bill Cycles ({year !== 'all' ? year : 'All Years'})
                        </CardTitle>
                        <CardDescription>
                            Manage billing cycles: Generate → Review → Post to Residents
                        </CardDescription>
                    </CardHeader>
                    <CardContent>
                        <ModernTable>
                            <ModernTableHeader>
                                <ModernTableRow>
                                    <ModernTableHead>Cycle Name</ModernTableHead>
                                    <ModernTableHead>Year</ModernTableHead>
                                    <ModernTableHead>Generate Date</ModernTableHead>
                                    <ModernTableHead>Due Date</ModernTableHead>
                                    <ModernTableHead className="text-center">Bills</ModernTableHead>
                                    <ModernTableHead>Status</ModernTableHead>
                                    <ModernTableHead className="text-right">Actions</ModernTableHead>
                                </ModernTableRow>
                            </ModernTableHeader>
                            <ModernTableBody>
                                {billCycles.data.length > 0 ? (
                                    billCycles.data.map((cycle) => (
                                        <ModernTableRow key={cycle.id}>
                                            <ModernTableCell>
                                                <div className="flex items-center gap-2">
                                                    <div className="h-8 w-8 rounded-lg bg-primary/10 flex items-center justify-center text-primary">
                                                        <Calendar className="h-4 w-4" />
                                                    </div>
                                                    <span className="font-medium">{cycle.name}</span>
                                                </div>
                                            </ModernTableCell>
                                            <ModernTableCell>{cycle.year}</ModernTableCell>
                                            <ModernTableCell className="text-muted-foreground">
                                                {new Date(cycle.generate_date).toLocaleDateString('en-IN', {
                                                    day: '2-digit',
                                                    month: 'short',
                                                    year: 'numeric',
                                                })}
                                            </ModernTableCell>
                                            <ModernTableCell className="text-muted-foreground">
                                                {new Date(cycle.due_date).toLocaleDateString('en-IN', {
                                                    day: '2-digit',
                                                    month: 'short',
                                                    year: 'numeric',
                                                })}
                                            </ModernTableCell>
                                            <ModernTableCell className="text-center">
                                                <span className="font-semibold">{cycle.bills_count}</span>
                                            </ModernTableCell>
                                            <ModernTableCell>
                                                {getStatusBadge(cycle.status)}
                                            </ModernTableCell>
                                            <ModernTableCell>
                                                <div className="flex items-center justify-end gap-2">
                                                    {cycle.status === 'draft' && (
                                                        <Button
                                                            size="sm"
                                                            onClick={() => openGenerateModal(cycle.id, 'generate')}
                                                            className="bg-blue-600 hover:bg-blue-700"
                                                        >
                                                            <Play className="mr-1 h-3 w-3" />
                                                            Generate
                                                        </Button>
                                                    )}
                                                    {(cycle.status === 'under_review' || cycle.status === 'generated') && (
                                                        <>
                                                            <Button
                                                                size="sm"
                                                                variant="outline"
                                                                onClick={() => openGenerateModal(cycle.id, 'regenerate')}
                                                            >
                                                                <Play className="mr-1 h-3 w-3" />
                                                                Regenerate
                                                            </Button>
                                                            <Button
                                                                size="sm"
                                                                onClick={() => handlePostBills(cycle.id)}
                                                                className="bg-green-600 hover:bg-green-700"
                                                            >
                                                                <Send className="mr-1 h-3 w-3" />
                                                                Post
                                                            </Button>
                                                        </>
                                                    )}
                                                    <Link href={`/societies/${society.id}/bills?bill_cycle_id=${cycle.id}`}>
                                                        <Button variant="ghost" size="sm">
                                                            <Eye className="h-4 w-4" />
                                                        </Button>
                                                    </Link>
                                                </div>
                                            </ModernTableCell>
                                        </ModernTableRow>
                                    ))
                                ) : (
                                    <ModernTableRow>
                                        <ModernTableCell colSpan={7} className="text-center py-12">
                                            <div className="flex flex-col items-center gap-2">
                                                <Calendar className="h-12 w-12 text-muted-foreground/50" />
                                                <p className="text-muted-foreground">No bill cycles found for {year}</p>
                                                <Button variant="outline" size="sm" onClick={() => setIsCreateOpen(true)}>
                                                    <Plus className="mr-2 h-4 w-4" />
                                                    Create First Cycle
                                                </Button>
                                            </div>
                                        </ModernTableCell>
                                    </ModernTableRow>
                                )}
                            </ModernTableBody>
                        </ModernTable>

                        {/* Pagination */}
                        {billCycles.links && billCycles.links.length > 3 && (
                            <div className="mt-4 flex items-center justify-between">
                                <div className="text-sm text-muted-foreground">
                                    Showing {billCycles.meta.from} to {billCycles.meta.to} of {billCycles.meta.total} results
                                </div>
                                <div className="flex gap-2">
                                    {billCycles.links.map((link: any, index: number) => (
                                        <Link
                                            key={index}
                                            href={link.url || '#'}
                                            className={`px-3 py-1 rounded ${
                                                link.active
                                                    ? 'bg-primary text-primary-foreground'
                                                    : 'bg-muted hover:bg-muted/80'
                                            }`}
                                            dangerouslySetInnerHTML={{ __html: link.label }}
                                        />
                                    ))}
                                </div>
                            </div>
                        )}
                    </CardContent>
                </GlassCard>
            </div>

            {/* Generate/Regenerate Modal */}
            <Dialog open={generateModalOpen} onOpenChange={setGenerateModalOpen}>
                <DialogContent>
                    <DialogHeader>
                        <DialogTitle>
                            {actionType === 'generate' ? 'Generate Bills' : 'Regenerate Bills'}
                        </DialogTitle>
                        <DialogDescription>
                            {actionType === 'generate'
                                ? 'Generate bills for all residents. Bills will be created in "Under Review" status for admin approval.'
                                : 'Regenerate bills for this cycle. Only "Under Review" bills will be updated.'}
                        </DialogDescription>
                    </DialogHeader>
                    <div className="space-y-4 py-4">
                        <div className="space-y-2">
                            <Label htmlFor="structure">Maintenance Structure (Optional)</Label>
                            <Select
                                value={selectedStructureId}
                                onValueChange={setSelectedStructureId}
                            >
                                <SelectTrigger id="structure">
                                    <SelectValue placeholder="Select Structure" />
                                </SelectTrigger>
                                <SelectContent>
                                    <SelectItem value="none">None (Use unit-assigned defaults)</SelectItem>
                                    {maintenanceStructures.map((structure) => (
                                        <SelectItem key={structure.id} value={structure.id.toString()}>
                                            {structure.name} (₹{Number(structure.maintenance) + Number(structure.parking) + Number(structure.other_charges)})
                                        </SelectItem>
                                    ))}
                                </SelectContent>
                            </Select>
                            <p className="text-xs text-muted-foreground">
                                If not selected, each unit's assigned maintenance structure will be used.
                            </p>
                        </div>

                        <div className="rounded-lg bg-blue-50 dark:bg-blue-950/30 p-4 text-sm">
                            <div className="flex items-start gap-2">
                                <AlertCircle className="h-4 w-4 text-blue-600 mt-0.5" />
                                <div>
                                    <p className="font-medium text-blue-800 dark:text-blue-200">Bill Generation Flow</p>
                                    <ol className="list-decimal list-inside mt-2 text-blue-700 dark:text-blue-300 space-y-1">
                                        <li>Bills are generated with "Under Review" status</li>
                                        <li>Admin can review and modify individual bills</li>
                                        <li>Click "Post" to make bills visible to residents</li>
                                    </ol>
                                </div>
                            </div>
                        </div>
                    </div>
                    <DialogFooter>
                        <Button variant="outline" onClick={() => setGenerateModalOpen(false)}>
                            Cancel
                        </Button>
                        <Button onClick={handleGenerateSubmit} disabled={isProcessing}>
                            {isProcessing ? (
                                <>
                                    <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                                    Processing...
                                </>
                            ) : (
                                actionType === 'generate' ? 'Generate Bills' : 'Regenerate Bills'
                            )}
                        </Button>
                    </DialogFooter>
                </DialogContent>
            </Dialog>
        </SocietyAdminLayout>
    );
}
