import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, Link, router } from '@inertiajs/react';
import { ArrowLeft, Building2, FileText, TrendingDown, TrendingUp } from 'lucide-react';

const breadcrumbs: BreadcrumbItem[] = [
    { title: 'Bank Accounts', href: '/societies/{society}/bank-accounts' },
    { title: 'Statement', href: '#' },
];

interface Transaction {
    id: number;
    type: 'credit' | 'debit';
    date: string | null;
    description: string;
    reference: string;
    payment_method: string;
    debit: number;
    credit: number;
    balance: number;
    status: string;
    category?: string;
    created_by_name?: string;
}

interface TransactionsProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    bankAccount: {
        id: number;
        account_name: string;
        account_number: string;
        ifsc_code: string;
        bank_name: string;
        branch_name: string | null;
        balance: number;
        account_type: string;
    };
    transactions: {
        data: Transaction[];
        current_page: number;
        per_page: number;
        total: number;
        last_page: number;
        from: number;
        to: number;
    };
    filters?: {
        start_date?: string;
        end_date?: string;
    };
}

export default function Transactions({ society, bankAccount, transactions, filters }: TransactionsProps) {
    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 2,
        }).format(amount);
    };

    const formatDate = (dateString: string | null) => {
        // Handle null dates
        if (!dateString) {
            return 'N/A';
        }
        
        // Handle both date formats: "2025-12-05" and "2025-12-05 10:30:00"
        const date = new Date(dateString.replace(' ', 'T')); // Convert to ISO format if needed
        
        // Check if date is valid
        if (isNaN(date.getTime())) {
            console.error('Invalid date:', dateString);
            return 'Invalid Date';
        }
        
        return date.toLocaleDateString('en-IN', {
            year: 'numeric',
            month: 'short',
            day: '2-digit',
        });
    };

    const handlePageChange = (page: number) => {
        router.get(`/societies/${society.id}/bank-accounts/${bankAccount.id}/transactions?page=${page}`, {}, {
            preserveState: true,
            preserveScroll: false,
        });
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title={`${bankAccount.account_name} - Bank Statement`} />
            <div className="flex h-full flex-1 flex-col gap-6 overflow-x-auto rounded-xl p-6 md:p-8">
                <div className="flex flex-col md:flex-row md:items-center justify-between gap-4">
                    <div className="flex items-center gap-4">
                        <Link href={`/societies/${society.id}/bank-accounts`}>
                            <Button variant="ghost" size="sm">
                                <ArrowLeft className="mr-2 h-4 w-4" />
                                Back
                            </Button>
                        </Link>
                        <div>
                            <h1 className="text-2xl font-bold">Bank Statement</h1>
                            <p className="text-muted-foreground">
                                {bankAccount.account_name}
                            </p>
                        </div>
                    </div>
                    
                    <div className="flex items-center gap-2">
                        <div className="flex items-center gap-2 bg-white dark:bg-black/20 p-1 rounded-lg border border-border/50">
                            <input
                                type="date"
                                className="h-8 rounded-md border-0 bg-transparent px-2 text-sm focus:ring-0"
                                value={filters?.start_date || ''}
                                onChange={(e) => router.get(
                                    window.location.pathname,
                                    { ...filters, start_date: e.target.value },
                                    { preserveState: true }
                                )}
                                placeholder="From Date"
                            />
                            <span className="text-muted-foreground">-</span>
                            <input
                                type="date"
                                className="h-8 rounded-md border-0 bg-transparent px-2 text-sm focus:ring-0"
                                value={filters?.end_date || ''}
                                onChange={(e) => router.get(
                                    window.location.pathname,
                                    { ...filters, end_date: e.target.value },
                                    { preserveState: true }
                                )}
                                placeholder="To Date"
                            />
                        </div>

                        <a 
                            href={`/societies/${society.id}/bank-accounts/${bankAccount.id}/export?start_date=${filters?.start_date || ''}&end_date=${filters?.end_date || ''}`} 
                            target="_blank"
                            rel="noopener noreferrer"
                        >
                            <Button variant="outline" size="sm" className="gap-2">
                                <FileText className="h-4 w-4" />
                                Export
                            </Button>
                        </a>
                    </div>
                </div>

                {/* Account Details Card */}
                <Card>
                    <CardHeader>
                        <CardTitle className="flex items-center gap-2">
                            <Building2 className="h-5 w-5" />
                            Account Information
                        </CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="grid grid-cols-2 md:grid-cols-5 gap-4">
                            <div>
                                <p className="text-sm text-muted-foreground">Account Name</p>
                                <p className="font-medium">{bankAccount.account_name}</p>
                            </div>
                            <div>
                                <p className="text-sm text-muted-foreground">Bank</p>
                                <p className="font-medium">{bankAccount.bank_name}</p>
                                {bankAccount.branch_name && (
                                    <p className="text-xs text-muted-foreground">{bankAccount.branch_name}</p>
                                )}
                            </div>
                            <div>
                                <p className="text-sm text-muted-foreground">Account Number</p>
                                <p className="font-medium">{bankAccount.account_number}</p>
                            </div>
                            <div>
                                <p className="text-sm text-muted-foreground">IFSC Code</p>
                                <p className="font-medium">{bankAccount.ifsc_code}</p>
                            </div>
                            <div>
                                <p className="text-sm text-muted-foreground">Current Balance</p>
                                <p className="font-medium text-lg text-primary">{formatCurrency(bankAccount.balance)}</p>
                            </div>
                        </div>
                    </CardContent>
                </Card>

                {/* Bank Statement */}
                <Card>
                    <CardHeader>
                        <CardTitle className="flex items-center gap-2">
                            <FileText className="h-5 w-5" />
                            Statement of Account
                        </CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="overflow-x-auto">
                            <table className="w-full border-collapse">
                                <thead>
                                    <tr className="border-b-2 border-primary">
                                        <th className="px-4 py-3 text-left text-sm font-semibold">Date</th>
                                        <th className="px-4 py-3 text-left text-sm font-semibold">Description</th>
                                        <th className="px-4 py-3 text-left text-sm font-semibold">Category</th>
                                        <th className="px-4 py-3 text-left text-sm font-semibold">Reference</th>
                                        <th className="px-4 py-3 text-left text-sm font-semibold">Issued By</th>
                                        <th className="px-4 py-3 text-right text-sm font-semibold text-red-600 dark:text-red-400">Debit (₹)</th>
                                        <th className="px-4 py-3 text-right text-sm font-semibold text-green-600 dark:text-green-400">Credit (₹)</th>
                                        <th className="px-4 py-3 text-right text-sm font-semibold">Balance (₹)</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {transactions.data.length > 0 ? (
                                        transactions.data.map((transaction, index) => (
                                            <tr 
                                                key={`${transaction.type}-${transaction.id}`} 
                                                className={`border-b hover:bg-muted/30 ${
                                                    transaction.type === 'debit' ? 'bg-red-50/30 dark:bg-red-950/10' : 'bg-green-50/30 dark:bg-green-950/10'
                                                }`}
                                            >
                                                <td className="px-4 py-3 text-sm whitespace-nowrap">
                                                    {formatDate(transaction.date)}
                                                </td>
                                                <td className="px-4 py-3 text-sm">
                                                    <div className="flex items-start gap-2">
                                                        {transaction.type === 'credit' ? (
                                                            <TrendingUp className="h-4 w-4 text-green-600 dark:text-green-400 mt-0.5 flex-shrink-0" />
                                                        ) : (
                                                            <TrendingDown className="h-4 w-4 text-red-600 dark:text-red-400 mt-0.5 flex-shrink-0" />
                                                        )}
                                                        <div>
                                                            <div className="font-medium">{transaction.description}</div>
                                                            <div className="text-xs text-muted-foreground capitalize">{transaction.payment_method.replace('_', ' ')}</div>
                                                        </div>
                                                    </div>
                                                </td>
                                                <td className="px-4 py-3 text-sm text-muted-foreground capitalize">
                                                    {transaction.category}
                                                </td>
                                                <td className="px-4 py-3 text-sm text-muted-foreground">
                                                    {transaction.reference}
                                                </td>
                                                <td className="px-4 py-3 text-sm text-muted-foreground">
                                                    {transaction.created_by_name}
                                                </td>
                                                <td className="px-4 py-3 text-sm text-right font-medium">
                                                    {transaction.debit > 0 ? (
                                                        <span className="text-red-600 dark:text-red-400">
                                                            {formatCurrency(transaction.debit)}
                                                        </span>
                                                    ) : (
                                                        <span className="text-muted-foreground">-</span>
                                                    )}
                                                </td>
                                                <td className="px-4 py-3 text-sm text-right font-medium">
                                                    {transaction.credit > 0 ? (
                                                        <span className="text-green-600 dark:text-green-400">
                                                            {formatCurrency(transaction.credit)}
                                                        </span>
                                                    ) : (
                                                        <span className="text-muted-foreground">-</span>
                                                    )}
                                                </td>
                                                <td className="px-4 py-3 text-sm text-right font-semibold">
                                                    {formatCurrency(transaction.balance)}
                                                </td>
                                            </tr>
                                        ))
                                    ) : (
                                        <tr>
                                            <td colSpan={6} className="px-4 py-8 text-center text-muted-foreground">
                                                No transactions found for this account
                                            </td>
                                        </tr>
                                    )}
                                </tbody>
                            </table>
                        </div>

                        {/* Pagination */}
                        {transactions.total > transactions.per_page && (
                            <div className="mt-4 flex items-center justify-between border-t pt-4">
                                <div className="text-sm text-muted-foreground">
                                    Showing {transactions.from} to {transactions.to} of {transactions.total} transactions
                                </div>
                                <div className="flex gap-2">
                                    <Button
                                        variant="outline"
                                        size="sm"
                                        onClick={() => handlePageChange(transactions.current_page - 1)}
                                        disabled={transactions.current_page === 1}
                                    >
                                        Previous
                                    </Button>
                                    {Array.from({ length: transactions.last_page }, (_, i) => i + 1)
                                        .filter(page => {
                                            return page === 1 || 
                                                   page === transactions.last_page || 
                                                   Math.abs(page - transactions.current_page) <= 2;
                                        })
                                        .map((page, index, array) => (
                                            <div key={page} className="flex gap-2">
                                                {index > 0 && array[index - 1] !== page - 1 && (
                                                    <span className="px-3 py-1">...</span>
                                                )}
                                                <Button
                                                    variant={page === transactions.current_page ? 'default' : 'outline'}
                                                    size="sm"
                                                    onClick={() => handlePageChange(page)}
                                                >
                                                    {page}
                                                </Button>
                                            </div>
                                        ))}
                                    <Button
                                        variant="outline"
                                        size="sm"
                                        onClick={() => handlePageChange(transactions.current_page + 1)}
                                        disabled={transactions.current_page === transactions.last_page}
                                    >
                                        Next
                                    </Button>
                                </div>
                            </div>
                        )}
                    </CardContent>
                </Card>
            </div>
        </SocietyAdminLayout>
    );
}
