import { Head } from '@inertiajs/react';
import { Building2, FileText, Phone, Mail, MapPin } from 'lucide-react';
import { useEffect } from 'react';

interface Transaction {
    id: number;
    type: 'credit' | 'debit';
    date: string;
    description: string;
    reference: string;
    payment_method: string;
    debit: number;
    credit: number;
    balance: number;
}

interface StatementProps {
    society: {
        id: number;
        name: string;
        code?: string;
        address?: string;
        phone?: string;
        email?: string;
    };
    bankAccount: {
        account_name: string;
        account_number: string;
        bank_name: string;
        ifsc_code: string;
        branch_name: string | null;
    };
    transactions: Transaction[];
    filters: {
        start_date?: string;
        end_date?: string;
    };
    openingBalance: number;
    closingBalance: number;
}

export default function Statement({
    society,
    bankAccount,
    transactions,
    filters,
    openingBalance,
    closingBalance,
}: StatementProps) {

    useEffect(() => {
        // Auto-print when opened
        setTimeout(() => {
            window.print();
        }, 1000);
    }, []);

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 2,
        }).format(amount);
    };

    return (
        <div className="bg-white min-h-screen text-black p-8 md:p-12 print:p-0">
            <Head title={`Statement - ${bankAccount.account_name}`} />

            {/* Print Container */}
            <div className="max-w-[210mm] mx-auto print:max-w-none">
                
                {/* Header / Letterhead */}
                <div className="border-b-2 border-black pb-6 mb-8">
                    <div className="flex justify-between items-start">
                        <div className="flex gap-4 items-center">
                            {/* Logo Placeholder */}
                            <div className="h-16 w-16 bg-black text-white flex items-center justify-center rounded-lg print:border print:border-black print:text-black print:bg-transparent">
                                <Building2 className="h-8 w-8" />
                            </div>
                            <div>
                                <h1 className="text-2xl font-bold uppercase tracking-wide">{society.name}</h1>
                                <p className="text-sm text-gray-600">{society.code}</p>
                                {society.address && (
                                    <div className="flex items-center gap-1 text-sm mt-1 text-gray-600">
                                        <MapPin className="h-3 w-3" />
                                        <span>{society.address}</span>
                                    </div>
                                )}
                            </div>
                        </div>
                        <div className="text-right text-sm space-y-1">
                            {society.phone && (
                                <div className="flex items-center justify-end gap-2">
                                    <span>{society.phone}</span>
                                    <Phone className="h-3 w-3" />
                                </div>
                            )}
                            {society.email && (
                                <div className="flex items-center justify-end gap-2">
                                    <span>{society.email}</span>
                                    <Mail className="h-3 w-3" />
                                </div>
                            )}
                        </div>
                    </div>
                </div>

                {/* Statement Info */}
                <div className="flex justify-between items-end mb-8 bg-gray-50 p-4 rounded-lg print:bg-transparent print:p-0 print:mb-6">
                    <div>
                        <h2 className="text-xl font-bold mb-2">Statement of Account</h2>
                        <div className="space-y-1 text-sm">
                            <p><span className="font-semibold w-24 inline-block">Account Name:</span> {bankAccount.account_name}</p>
                            <p><span className="font-semibold w-24 inline-block">Bank:</span> {bankAccount.bank_name}</p>
                            <p><span className="font-semibold w-24 inline-block">Account No:</span> {bankAccount.account_number}</p>
                            <p><span className="font-semibold w-24 inline-block">IFSC Code:</span> {bankAccount.ifsc_code}</p>
                        </div>
                    </div>
                    <div className="text-right">
                        <div className="inline-block bg-black text-white px-4 py-2 rounded-md print:bg-transparent print:text-black print:border print:border-black">
                            <p className="text-xs uppercase tracking-wider opacity-80 print:opacity-100">Period</p>
                            <p className="font-bold">
                                {filters.start_date || 'Inception'} <span className="mx-1">to</span> {filters.end_date || 'Present'}
                            </p>
                        </div>
                    </div>
                </div>

                {/* Opening Balance */}
                <div className="flex justify-between items-center mb-4 text-sm font-medium border p-3 rounded bg-gray-50 print:bg-transparent print:border-gray-300">
                    <span>Opening Balance</span>
                    <span>{formatCurrency(openingBalance)}</span>
                </div>

                {/* Transactions Table */}
                <table className="w-full text-sm border-collapse mb-8">
                    <thead>
                        <tr className="border-b-2 border-black">
                            <th className="py-2 text-left w-24">Date</th>
                            <th className="py-2 text-left">Description</th>
                            <th className="py-2 text-left w-32">Reference</th>
                            <th className="py-2 text-right w-24">Debit</th>
                            <th className="py-2 text-right w-24">Credit</th>
                            <th className="py-2 text-right w-28">Balance</th>
                        </tr>
                    </thead>
                    <tbody className="divide-y divide-gray-200 print:divide-gray-300">
                        {transactions.map((tx) => (
                            <tr key={tx.id} className="break-inside-avoid">
                                <td className="py-3 align-top text-gray-600">{tx.date}</td>
                                <td className="py-3 align-top font-medium">
                                    {tx.description}
                                    <div className="text-xs text-gray-500 capitalize mt-0.5">
                                        {tx.payment_method.replace('_', ' ')}
                                    </div>
                                </td>
                                <td className="py-3 align-top text-gray-600 text-xs">{tx.reference}</td>
                                <td className="py-3 align-top text-right text-red-600 print:text-black">
                                    {tx.debit > 0 ? formatCurrency(tx.debit) : '-'}
                                </td>
                                <td className="py-3 align-top text-right text-green-600 print:text-black">
                                    {tx.credit > 0 ? formatCurrency(tx.credit) : '-'}
                                </td>
                                <td className="py-3 align-top text-right font-semibold">
                                    {formatCurrency(tx.balance)}
                                </td>
                            </tr>
                        ))}
                    </tbody>
                </table>

                {/* Closing Balance */}
                <div className="flex justify-end mb-12">
                     <div className="w-64 border-t-2 border-black pt-2">
                        <div className="flex justify-between items-center text-sm font-medium mb-1">
                            <span>Closing Balance</span>
                            <span>{formatCurrency(closingBalance)}</span>
                        </div>
                         <div className="flex justify-between items-center text-xs text-gray-500">
                            <span>Total Debits</span>
                            <span>{formatCurrency(transactions.reduce((sum, t) => sum + t.debit, 0))}</span>
                        </div>
                         <div className="flex justify-between items-center text-xs text-gray-500">
                            <span>Total Credits</span>
                            <span>{formatCurrency(transactions.reduce((sum, t) => sum + t.credit, 0))}</span>
                        </div>
                     </div>
                </div>

                 {/* Footer */}
                 <div className="fixed bottom-0 left-0 w-full text-center text-xs text-gray-400 p-8 print:relative print:p-0 print:text-black print:mt-12 break-inside-avoid">
                    <p>Generated by Societify on {new Date().toLocaleString('en-IN', { dateStyle: 'full', timeStyle: 'short' })}</p>
                    <p>This is a computer-generated document and does not require a signature.</p>
                 </div>
            </div>

            {/* Screen-only Print Button */}
            <button 
                onClick={() => window.print()}
                className="fixed bottom-8 right-8 bg-blue-600 text-white p-4 rounded-full shadow-lg hover:bg-blue-700 transition-colors print:hidden flex items-center gap-2 z-50"
            >
                <FileText className="h-5 w-5" />
                <span>Print Statement</span>
            </button>
        </div>
    );
}
