import { Button } from '@/components/ui/button';
import { GlassCard } from '@/components/ui/glass-card';
import { PageHeader } from '@/components/ui/page-header';
import { ModernTable, ModernTableHeader, ModernTableBody, ModernTableRow, ModernTableHead, ModernTableCell } from '@/components/ui/modern-table';
import { Badge } from '@/components/ui/badge';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, router, useForm } from '@inertiajs/react';
import { Building2, Plus, Search, Edit, TrendingUp, DollarSign, CreditCard, FileText, Eye } from 'lucide-react';
import { useState } from 'react';
import { cn } from '@/lib/utils';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Bank Accounts',
        href: '/societies/{society}/bank-accounts',
    },
];

interface BankAccount {
    id: number;
    account_name: string;
    account_number: string;
    ifsc_code: string;
    bank_name: string;
    branch_name: string | null;
    balance: number;
    account_type: string;
    last_reconciled_at: string | null;
    payments_count: number;
    updated_by: {
        id: number;
        name: string;
    } | null;
    created_at: string;
    is_primary: boolean;
    description: string | null;
}

interface IndexProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    bankAccounts: {
        data: BankAccount[];
        links: any;
        meta: any;
    };
    filters: {
        search?: string;
        account_type?: string;
    };
    stats: {
        total_balance: number;
        total_accounts: number;
        total_transactions: number;
        savings_balance: number;
    };
}

export default function Index({ society, bankAccounts, filters, stats }: IndexProps) {
    const [search, setSearch] = useState(filters.search || '');
    const [accountType, setAccountType] = useState(filters.account_type || 'all');
    const [isCreateOpen, setIsCreateOpen] = useState(false);
    const [isEditOpen, setIsEditOpen] = useState(false);
    const [isDetailsOpen, setIsDetailsOpen] = useState(false);
    const [selectedAccount, setSelectedAccount] = useState<BankAccount | null>(null);

    const createForm = useForm({
        account_name: '',
        account_number: '',
        ifsc_code: '',
        bank_name: '',
        branch_name: '',
        balance: '',
        account_type: 'current',
        description: '',
        is_primary: false,
    });

    const editForm = useForm({
        account_name: '',
        account_number: '',
        ifsc_code: '',
        bank_name: '',
        branch_name: '',
        balance: '',
        account_type: 'current',
        description: '',
        is_primary: false,
    });

    const handleSearch = () => {
        const filterParams: any = {};
        if (search) filterParams.search = search;
        if (accountType && accountType !== 'all') filterParams.account_type = accountType;

        router.get(`/societies/${society.id}/bank-accounts`, filterParams, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const handleCreateSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        createForm.post(`/societies/${society.id}/bank-accounts`, {
            onSuccess: () => {
                setIsCreateOpen(false);
                createForm.reset();
            },
        });
    };

    const handleEditSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        if (!selectedAccount) return;

        editForm.put(`/societies/${society.id}/bank-accounts/${selectedAccount.id}`, {
            onSuccess: () => {
                setIsEditOpen(false);
                editForm.reset();
                setSelectedAccount(null);
            },
        });
    };

    const openEditModal = (account: BankAccount) => {
        setSelectedAccount(account);
        editForm.setData({
            account_name: account.account_name,
            account_number: account.account_number,
            ifsc_code: account.ifsc_code,
            bank_name: account.bank_name,
            branch_name: account.branch_name || '',
            balance: account.balance.toString(),
            account_type: account.account_type,
        });
        setIsEditOpen(true);
    };

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 0,
        }).format(amount);
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Bank Accounts" />
            <div className="space-y-6 p-6 md:p-8">
                <PageHeader
                    title="Bank Accounts"
                    description="Manage all bank accounts for your society"
                    actions={
                        <Dialog open={isCreateOpen} onOpenChange={setIsCreateOpen}>
                            <DialogTrigger asChild>
                                <Button className="bg-primary hover:bg-primary/90">
                                    <Plus className="mr-2 h-4 w-4" />
                                    Add Bank Account
                                </Button>
                            </DialogTrigger>
                            <DialogContent className="max-w-2xl">
                                <DialogHeader>
                                    <DialogTitle>Add New Bank Account</DialogTitle>
                                    <DialogDescription>
                                        Add a bank account for your society
                                    </DialogDescription>
                                </DialogHeader>
                                <form onSubmit={handleCreateSubmit} className="space-y-4">
                                    <div className="grid grid-cols-2 gap-4">
                                        <div className="space-y-2">
                                            <Label htmlFor="account_name">Account Name *</Label>
                                            <Input
                                                id="account_name"
                                                placeholder="e.g., Main Account"
                                                value={createForm.data.account_name}
                                                onChange={(e) => createForm.setData('account_name', e.target.value)}
                                                required
                                            />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="account_number">Account Number *</Label>
                                            <Input
                                                id="account_number"
                                                placeholder="e.g., 1234567890"
                                                value={createForm.data.account_number}
                                                onChange={(e) => createForm.setData('account_number', e.target.value)}
                                                required
                                            />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="bank_name">Bank Name *</Label>
                                            <Input
                                                id="bank_name"
                                                placeholder="e.g., HDFC Bank"
                                                value={createForm.data.bank_name}
                                                onChange={(e) => createForm.setData('bank_name', e.target.value)}
                                                required
                                            />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="ifsc_code">IFSC Code *</Label>
                                            <Input
                                                id="ifsc_code"
                                                placeholder="e.g., HDFC0001234"
                                                value={createForm.data.ifsc_code}
                                                onChange={(e) => createForm.setData('ifsc_code', e.target.value.toUpperCase())}
                                                required
                                            />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="branch_name">Branch Name</Label>
                                            <Input
                                                id="branch_name"
                                                placeholder="e.g., Main Branch"
                                                value={createForm.data.branch_name}
                                                onChange={(e) => createForm.setData('branch_name', e.target.value)}
                                            />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="account_type">Account Type *</Label>
                                            <Select
                                                value={createForm.data.account_type}
                                                onValueChange={(value) => createForm.setData('account_type', value)}
                                            >
                                                <SelectTrigger id="account_type">
                                                    <SelectValue />
                                                </SelectTrigger>
                                                <SelectContent>
                                                    <SelectItem value="savings">Savings</SelectItem>
                                                    <SelectItem value="current">Current</SelectItem>
                                                    <SelectItem value="fd">Fixed Deposit</SelectItem>
                                                </SelectContent>
                                            </Select>
                                        </div>
                                    <div className="space-y-2 col-span-2">
                                        <Label htmlFor="description">Description</Label>
                                        <Input
                                            id="description"
                                            placeholder="Optional description"
                                            value={createForm.data.description}
                                            onChange={(e) => createForm.setData('description', e.target.value)}
                                        />
                                    </div>
                                    <div className="flex items-center space-x-2 col-span-2">
                                        <input
                                            type="checkbox"
                                            id="is_primary"
                                            className="h-4 w-4 rounded border-gray-300 text-primary focus:ring-primary"
                                            checked={createForm.data.is_primary}
                                            onChange={(e) => createForm.setData('is_primary', e.target.checked)}
                                        />
                                        <Label htmlFor="is_primary">Set as Primary Account (Default for all credits)</Label>
                                    </div>
                                    <div className="space-y-2 col-span-2">
                                        <Label htmlFor="balance">Opening Balance *</Label>
                                            <Input
                                                id="balance"
                                                type="number"
                                                step="0.01"
                                                placeholder="0.00"
                                                value={createForm.data.balance}
                                                onChange={(e) => createForm.setData('balance', e.target.value)}
                                                required
                                            />
                                        </div>
                                    </div>
                                    <DialogFooter>
                                        <Button type="button" variant="outline" onClick={() => setIsCreateOpen(false)}>
                                            Cancel
                                        </Button>
                                        <Button type="submit" disabled={createForm.processing}>
                                            {createForm.processing ? 'Adding...' : 'Add Account'}
                                        </Button>
                                    </DialogFooter>
                                </form>
                            </DialogContent>
                        </Dialog>
                    }
                />

                {/* KPI Cards */}
                <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-4">
                    <GlassCard className="p-6">
                        <div className="flex items-center justify-between">
                            <div className="space-y-1">
                                <p className="text-sm font-medium text-muted-foreground">Total Balance</p>
                                <p className="text-2xl font-bold bg-clip-text text-transparent bg-gradient-to-r from-primary to-primary/60">{formatCurrency(stats?.total_balance || 0)}</p>
                            </div>
                            <div className="rounded-xl bg-primary/10 p-3 ring-1 ring-primary/20">
                                <DollarSign className="h-6 w-6 text-primary" />
                            </div>
                        </div>
                    </GlassCard>
                    <GlassCard className="p-6">
                        <div className="flex items-center justify-between">
                            <div className="space-y-1">
                                <p className="text-sm font-medium text-muted-foreground">Total Accounts</p>
                                <p className="text-2xl font-bold bg-clip-text text-transparent bg-gradient-to-r from-emerald-500 to-emerald-700">{stats?.total_accounts || 0}</p>
                            </div>
                            <div className="rounded-xl bg-emerald-500/10 p-3 ring-1 ring-emerald-500/20">
                                <Building2 className="h-6 w-6 text-emerald-500" />
                            </div>
                        </div>
                    </GlassCard>
                    <GlassCard className="p-6">
                        <div className="flex items-center justify-between">
                            <div className="space-y-1">
                                <p className="text-sm font-medium text-muted-foreground">Total Transactions</p>
                                <p className="text-2xl font-bold bg-clip-text text-transparent bg-gradient-to-r from-blue-500 to-blue-700">{stats?.total_transactions || 0}</p>
                            </div>
                            <div className="rounded-xl bg-blue-500/10 p-3 ring-1 ring-blue-500/20">
                                <CreditCard className="h-6 w-6 text-blue-500" />
                            </div>
                        </div>
                    </GlassCard>
                    <GlassCard className="p-6">
                        <div className="flex items-center justify-between">
                            <div className="space-y-1">
                                <p className="text-sm font-medium text-muted-foreground">Savings Balance</p>
                                <p className="text-2xl font-bold bg-clip-text text-transparent bg-gradient-to-r from-amber-500 to-amber-700">{formatCurrency(stats?.savings_balance || 0)}</p>
                            </div>
                            <div className="rounded-xl bg-amber-500/10 p-3 ring-1 ring-amber-500/20">
                                <FileText className="h-6 w-6 text-amber-500" />
                            </div>
                        </div>
                    </GlassCard>
                </div>

                {/* Filters */}
                <GlassCard className="p-6">
                    <div className="flex flex-col md:flex-row gap-4">
                        <div className="flex-1">
                            <Label htmlFor="search">Search</Label>
                            <div className="relative mt-1">
                                <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                <Input
                                    id="search"
                                    placeholder="Search by account name, number, bank name, or IFSC..."
                                    value={search}
                                    onChange={(e) => setSearch(e.target.value)}
                                    onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                    className="pl-9 bg-background/50"
                                />
                            </div>
                        </div>
                        <div className="w-full md:w-48">
                            <Label htmlFor="account_type">Account Type</Label>
                            <Select value={accountType} onValueChange={setAccountType}>
                                <SelectTrigger id="account_type" className="mt-1 bg-background/50">
                                    <SelectValue placeholder="All Types" />
                                </SelectTrigger>
                                <SelectContent>
                                    <SelectItem value="all">All Types</SelectItem>
                                    <SelectItem value="savings">Savings</SelectItem>
                                    <SelectItem value="current">Current</SelectItem>
                                    <SelectItem value="fd">Fixed Deposit</SelectItem>
                                </SelectContent>
                            </Select>
                        </div>
                        <div className="flex items-end">
                            <Button onClick={handleSearch} variant="secondary">Filter</Button>
                        </div>
                    </div>
                </GlassCard>

                {/* Bank Accounts Table */}
                <GlassCard className="overflow-hidden">
                    <ModernTable>
                        <ModernTableHeader>
                            <ModernTableRow>
                                <ModernTableHead>Account Name</ModernTableHead>
                                <ModernTableHead>Bank</ModernTableHead>
                                <ModernTableHead>Account Number</ModernTableHead>
                                <ModernTableHead>IFSC Code</ModernTableHead>
                                <ModernTableHead>Type</ModernTableHead>
                                <ModernTableHead>Balance</ModernTableHead>
                                <ModernTableHead>Transactions</ModernTableHead>
                                <ModernTableHead className="text-right">Actions</ModernTableHead>
                            </ModernTableRow>
                        </ModernTableHeader>
                        <ModernTableBody>
                            {bankAccounts.data.length > 0 ? (
                                bankAccounts.data.map((account) => (
                                    <ModernTableRow key={account.id}>
                                        <ModernTableCell>
                                            <div className="flex flex-col">
                                                <div className="flex items-center gap-2">
                                                    <div className="p-2 rounded-lg bg-primary/10 text-primary">
                                                        <Building2 className="h-4 w-4" />
                                                    </div>
                                                    <span className="font-medium">{account.account_name}</span>
                                                    {account.is_primary && (
                                                        <Badge variant="default" className="bg-primary/80 hover:bg-primary/90 text-[10px] py-0 h-5">
                                                            Primary
                                                        </Badge>
                                                    )}
                                                </div>
                                                {account.description && (
                                                    <span className="text-xs text-muted-foreground ml-10 truncate max-w-[200px]">
                                                        {account.description}
                                                    </span>
                                                )}
                                            </div>
                                        </ModernTableCell>
                                        <ModernTableCell>
                                            <div className="flex flex-col">
                                                <span className="text-sm font-medium">{account.bank_name}</span>
                                                {account.branch_name && (
                                                    <span className="text-xs text-muted-foreground">
                                                        {account.branch_name}
                                                    </span>
                                                )}
                                            </div>
                                        </ModernTableCell>
                                        <ModernTableCell>
                                            <span className="font-mono text-sm text-muted-foreground tracking-wider">
                                                {account.account_number}
                                            </span>
                                        </ModernTableCell>
                                        <ModernTableCell>
                                            <Badge variant="outline" className="font-mono text-xs">
                                                {account.ifsc_code}
                                            </Badge>
                                        </ModernTableCell>
                                        <ModernTableCell>
                                            <Badge variant="secondary" className="capitalize">
                                                {account.account_type}
                                            </Badge>
                                        </ModernTableCell>
                                        <ModernTableCell>
                                            <div className="flex items-center gap-2">
                                                <TrendingUp className="h-4 w-4 text-emerald-500" />
                                                <span className="font-bold text-emerald-600 dark:text-emerald-400">{formatCurrency(account.balance)}</span>
                                            </div>
                                        </ModernTableCell>
                                        <ModernTableCell>
                                            <span className="text-sm text-muted-foreground">
                                                {account.payments_count} txns
                                            </span>
                                        </ModernTableCell>
                                        <ModernTableCell className="text-right">
                                            <div className="flex items-center justify-end gap-2">
                                                <Button
                                                    variant="ghost"
                                                    size="icon"
                                                    className="h-8 w-8 hover:bg-muted"
                                                    onClick={() => router.visit(`/societies/${society.id}/bank-accounts/${account.id}/transactions`)}
                                                    title="View Transactions"
                                                >
                                                    <Eye className="h-4 w-4" />
                                                </Button>
                                                <Button
                                                    variant="ghost"
                                                    size="icon"
                                                    className="h-8 w-8 hover:bg-muted"
                                                    onClick={() => openEditModal(account)}
                                                >
                                                    <Edit className="h-4 w-4" />
                                                </Button>
                                            </div>
                                        </ModernTableCell>
                                    </ModernTableRow>
                                ))
                            ) : (
                                <ModernTableRow>
                                    <ModernTableCell colSpan={8} className="h-24 text-center text-muted-foreground">
                                        No bank accounts found
                                    </ModernTableCell>
                                </ModernTableRow>
                            )}
                        </ModernTableBody>
                    </ModernTable>
                    
                    {/* Pagination */}
                    {bankAccounts.links && bankAccounts.links.length > 3 && (
                        <div className="flex items-center justify-between p-4 border-t border-border/40">
                            <div className="text-sm text-muted-foreground">
                                Showing {bankAccounts.meta.from} to {bankAccounts.meta.to} of {bankAccounts.meta.total} results
                            </div>
                            <div className="flex gap-2">
                                {bankAccounts.links.map((link: any, index: number) => (
                                    <button
                                        key={index}
                                        onClick={() => link.url && router.visit(link.url)}
                                        disabled={!link.url}
                                        className={cn(
                                            "px-3 py-1 rounded-md text-sm transition-colors",
                                            link.active
                                                ? 'bg-primary text-primary-foreground'
                                                : 'bg-muted hover:bg-muted/80 text-muted-foreground',
                                            !link.url && 'opacity-50 cursor-not-allowed hidden'
                                        )}
                                        dangerouslySetInnerHTML={{ __html: link.label }}
                                    />
                                ))}
                            </div>
                        </div>
                    )}
                </GlassCard>

                {/* Edit Modal */}
                <Dialog open={isEditOpen} onOpenChange={setIsEditOpen}>
                    <DialogContent className="max-w-2xl">
                        <DialogHeader>
                            <DialogTitle>Edit Bank Account</DialogTitle>
                            <DialogDescription>
                                Update bank account information
                            </DialogDescription>
                        </DialogHeader>
                        <form onSubmit={handleEditSubmit} className="space-y-4">
                            <div className="grid grid-cols-2 gap-4">
                                <div className="space-y-2">
                                    <Label htmlFor="edit_account_name">Account Name *</Label>
                                    <Input
                                        id="edit_account_name"
                                        value={editForm.data.account_name}
                                        onChange={(e) => editForm.setData('account_name', e.target.value)}
                                        required
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_account_number">Account Number *</Label>
                                    <Input
                                        id="edit_account_number"
                                        value={editForm.data.account_number}
                                        onChange={(e) => editForm.setData('account_number', e.target.value)}
                                        required
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_bank_name">Bank Name *</Label>
                                    <Input
                                        id="edit_bank_name"
                                        value={editForm.data.bank_name}
                                        onChange={(e) => editForm.setData('bank_name', e.target.value)}
                                        required
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_ifsc_code">IFSC Code *</Label>
                                    <Input
                                        id="edit_ifsc_code"
                                        value={editForm.data.ifsc_code}
                                        onChange={(e) => editForm.setData('ifsc_code', e.target.value.toUpperCase())}
                                        required
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_branch_name">Branch Name</Label>
                                    <Input
                                        id="edit_branch_name"
                                        value={editForm.data.branch_name}
                                        onChange={(e) => editForm.setData('branch_name', e.target.value)}
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_account_type">Account Type *</Label>
                                    <Select
                                        value={editForm.data.account_type}
                                        onValueChange={(value) => editForm.setData('account_type', value)}
                                    >
                                        <SelectTrigger id="edit_account_type">
                                            <SelectValue />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="savings">Savings</SelectItem>
                                            <SelectItem value="current">Current</SelectItem>
                                            <SelectItem value="fd">Fixed Deposit</SelectItem>
                                        </SelectContent>
                                    </Select>
                                </div>
                                <div className="space-y-2 col-span-2">
                                    <Label htmlFor="edit_description">Description</Label>
                                    <Input
                                        id="edit_description"
                                        value={editForm.data.description}
                                        onChange={(e) => editForm.setData('description', e.target.value)}
                                    />
                                </div>
                                <div className="flex items-center space-x-2 col-span-2">
                                    <input
                                        type="checkbox"
                                        id="edit_is_primary"
                                        className="h-4 w-4 rounded border-gray-300 text-primary focus:ring-primary"
                                        checked={editForm.data.is_primary}
                                        onChange={(e) => editForm.setData('is_primary', e.target.checked)}
                                    />
                                    <Label htmlFor="edit_is_primary">Set as Primary Account</Label>
                                </div>
                                <div className="space-y-2 col-span-2">
                                    <Label htmlFor="edit_balance">Current Balance *</Label>
                                    <Input
                                        id="edit_balance"
                                        type="number"
                                        step="0.01"
                                        value={editForm.data.balance}
                                        onChange={(e) => editForm.setData('balance', e.target.value)}
                                        required
                                    />
                                </div>
                            </div>
                            <DialogFooter>
                                <Button type="button" variant="outline" onClick={() => setIsEditOpen(false)}>
                                    Cancel
                                </Button>
                                <Button type="submit" disabled={editForm.processing}>
                                    {editForm.processing ? 'Updating...' : 'Update Account'}
                                </Button>
                            </DialogFooter>
                        </form>
                    </DialogContent>
                </Dialog>
            </div>
        </SocietyAdminLayout>
    );
}
