import { Button } from '@/components/ui/button';
import { GlassCard } from '@/components/ui/glass-card';
import { PageHeader } from '@/components/ui/page-header';
import { ModernTable, ModernTableHeader, ModernTableBody, ModernTableRow, ModernTableHead, ModernTableCell } from '@/components/ui/modern-table';
import { Badge } from '@/components/ui/badge';
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Textarea } from '@/components/ui/textarea';
import { Checkbox } from '@/components/ui/checkbox';
import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, router, useForm } from '@inertiajs/react';
import { Megaphone, Plus, Search, Edit, Trash2, Eye, EyeOff, AlertCircle, Info, Calendar, Wrench, CreditCard, CheckCircle2, BarChart3, Users, Building as BuildingIcon } from 'lucide-react';
import { useState } from 'react';
import { cn } from '@/lib/utils';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Announcements',
        href: '/societies/{society}/announcements',
    },
];

interface Announcement {
    id: number;
    title: string;
    content: string;
    type: string;
    priority: string;
    target_roles: string[] | null;
    target_units: number[] | null;
    target_buildings: number[] | null;
    start_date: string | null;
    end_date: string | null;
    require_acknowledgement: boolean;
    views_count: number;
    acknowledged_count: number;
    is_published: boolean;
    created_at: string;
    created_by: {
        id: number;
        name: string;
    };
}

interface Building {
    id: number;
    name: string;
    building_no: string;
}

interface IndexProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    announcements: {
        data: Announcement[];
        links: any;
        meta: any;
    };
    filters: {
        search?: string;
        type?: string;
        priority?: string;
        is_published?: string;
    };
    stats: {
        total: number;
        active: number;
        total_views: number;
        acknowledgements: number;
    };
    buildings: Building[];
}

const typeConfig = {
    info: { label: 'Info', icon: Info, color: 'bg-blue-500/10 text-blue-600 border-blue-500/20' },
    urgent: { label: 'Urgent', icon: AlertCircle, color: 'bg-red-500/10 text-red-600 border-red-500/20' },
    event: { label: 'Event', icon: Calendar, color: 'bg-emerald-500/10 text-emerald-600 border-emerald-500/20' },
    maintenance: { label: 'Maintenance', icon: Wrench, color: 'bg-orange-500/10 text-orange-600 border-orange-500/20' },
    payment: { label: 'Payment', icon: CreditCard, color: 'bg-purple-500/10 text-purple-600 border-purple-500/20' },
};

const priorityConfig = {
    normal: { label: 'Normal', color: 'bg-slate-500/10 text-slate-600 border-slate-500/20' },
    important: { label: 'Important', color: 'bg-amber-500/10 text-amber-600 border-amber-500/20' },
    urgent: { label: 'Urgent', color: 'bg-red-500/10 text-red-600 border-red-500/20' },
};

export default function Index({ society, announcements, filters, stats, buildings }: IndexProps) {
    const [search, setSearch] = useState(filters.search || '');
    const [type, setType] = useState(filters.type || 'all');
    const [priority, setPriority] = useState(filters.priority || 'all');
    const [isPublished, setIsPublished] = useState(filters.is_published || 'all');
    
    // Modal States
    const [isCreateOpen, setIsCreateOpen] = useState(false);
    const [isEditOpen, setIsEditOpen] = useState(false);
    const [selectedAnnouncement, setSelectedAnnouncement] = useState<Announcement | null>(null);

    const createForm = useForm({
        title: '',
        content: '',
        type: 'info',
        priority: 'normal',
        target_roles: [] as string[],
        target_buildings: [] as number[],
        start_date: '',
        end_date: '',
        require_acknowledgement: false,
        is_published: true,
    });

    const editForm = useForm({
        title: '',
        content: '',
        type: 'info',
        priority: 'normal',
        target_roles: [] as string[],
        target_buildings: [] as number[],
        target_units: [] as number[],
        start_date: '',
        end_date: '',
        require_acknowledgement: false,
        is_published: true,
    });

    const handleSearch = () => {
        const filterParams: any = {};
        if (search) filterParams.search = search;
        if (type && type !== 'all') filterParams.type = type;
        if (priority && priority !== 'all') filterParams.priority = priority;
        if (isPublished && isPublished !== 'all') filterParams.is_published = isPublished;

        router.get(`/societies/${society.id}/announcements`, filterParams, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const handleTogglePublish = (id: number) => {
        router.post(`/societies/${society.id}/announcements/${id}/toggle-publish`, {}, {
            preserveScroll: true,
        });
    };

    const handleDelete = (id: number) => {
        if (confirm('Are you sure you want to delete this announcement?')) {
            router.delete(`/societies/${society.id}/announcements/${id}`, {
                preserveScroll: true,
            });
        }
    };

    // Role & Building Toggles
    const toggleRole = (role: string, isEdit = false) => {
        const currentRoles = isEdit ? editForm.data.target_roles : createForm.data.target_roles;
        const newRoles = currentRoles.includes(role) 
            ? currentRoles.filter(r => r !== role) 
            : [...currentRoles, role];
        
        if (isEdit) editForm.setData('target_roles', newRoles);
        else createForm.setData('target_roles', newRoles);
    };

    const toggleBuilding = (buildingId: number, isEdit = false) => {
        const currentBuildings = isEdit ? editForm.data.target_buildings : createForm.data.target_buildings;
        const newBuildings = currentBuildings.includes(buildingId) 
            ? currentBuildings.filter(b => b !== buildingId) 
            : [...currentBuildings, buildingId];

        if (isEdit) editForm.setData('target_buildings', newBuildings);
        else createForm.setData('target_buildings', newBuildings);
    };

    const handleCreateSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        createForm.post(`/societies/${society.id}/announcements`, {
            onSuccess: () => {
                setIsCreateOpen(false);
                createForm.reset();
            },
        });
    };

    const handleEditSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        if(!selectedAnnouncement) return;
        editForm.put(`/societies/${society.id}/announcements/${selectedAnnouncement.id}`, {
            onSuccess: () => {
                setIsEditOpen(false);
                editForm.reset();
                setSelectedAnnouncement(null);
            },
        });
    };

    const openEditModal = (announcement: Announcement) => {
        setSelectedAnnouncement(announcement);
        editForm.setData({
            title: announcement.title,
            content: announcement.content,
            type: announcement.type,
            priority: announcement.priority,
            target_roles: announcement.target_roles || [],
            target_buildings: announcement.target_buildings || [],
            target_units: announcement.target_units || [],
            start_date: announcement.start_date || '',
            end_date: announcement.end_date || '',
            require_acknowledgement: announcement.require_acknowledgement,
            is_published: announcement.is_published,
        });
        setIsEditOpen(true);
    };

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Announcements" />
            <div className="space-y-6 p-6 md:p-8">
                <PageHeader
                    title="Announcements"
                    description="Manage all announcements for your society"
                    actions={
                        <Dialog open={isCreateOpen} onOpenChange={setIsCreateOpen}>
                            <DialogTrigger asChild>
                                <Button className="bg-primary hover:bg-primary/90">
                                    <Plus className="mr-2 h-4 w-4" />
                                    New Announcement
                                </Button>
                            </DialogTrigger>
                            <DialogContent className="max-w-3xl max-h-[90vh] overflow-y-auto">
                                <DialogHeader>
                                    <DialogTitle>Create New Announcement</DialogTitle>
                                    <DialogDescription>Broadcast information to residents and staff</DialogDescription>
                                </DialogHeader>
                                <form onSubmit={handleCreateSubmit} className="space-y-4">
                                    <div className="grid grid-cols-2 gap-4">
                                        <div className="col-span-2 space-y-2">
                                            <Label htmlFor="title">Title *</Label>
                                            <Input
                                                id="title"
                                                value={createForm.data.title}
                                                onChange={e => createForm.setData('title', e.target.value)}
                                                placeholder="Enter header title"
                                                required
                                            />
                                        </div>
                                        <div className="col-span-2 space-y-2">
                                            <Label htmlFor="content">Content *</Label>
                                            <Textarea
                                                id="content"
                                                value={createForm.data.content}
                                                onChange={e => createForm.setData('content', e.target.value)}
                                                placeholder="Enter announcement details"
                                                rows={4}
                                                required
                                            />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="type">Type</Label>
                                            <Select value={createForm.data.type} onValueChange={v => createForm.setData('type', v)}>
                                                <SelectTrigger><SelectValue /></SelectTrigger>
                                                <SelectContent>
                                                    {Object.entries(typeConfig).map(([k, v]) => (
                                                        <SelectItem key={k} value={k}>{v.label}</SelectItem>
                                                    ))}
                                                </SelectContent>
                                            </Select>
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="priority">Priority</Label>
                                            <Select value={createForm.data.priority} onValueChange={v => createForm.setData('priority', v)}>
                                                <SelectTrigger><SelectValue /></SelectTrigger>
                                                <SelectContent>
                                                    {Object.entries(priorityConfig).map(([k, v]) => (
                                                        <SelectItem key={k} value={k}>{v.label}</SelectItem>
                                                    ))}
                                                </SelectContent>
                                            </Select>
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="start_date">Start Date (Optional)</Label>
                                            <Input 
                                                type="datetime-local" 
                                                value={createForm.data.start_date}
                                                onChange={e => createForm.setData('start_date', e.target.value)}
                                            />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="end_date">End Date (Optional)</Label>
                                            <Input 
                                                type="datetime-local" 
                                                value={createForm.data.end_date}
                                                onChange={e => createForm.setData('end_date', e.target.value)}
                                            />
                                        </div>
                                    </div>
                                    
                                    <div className="space-y-2 border-t pt-4">
                                        <Label>Target Audience (Optional - Leave blank for all)</Label>
                                        <div className="grid grid-cols-2 gap-4">
                                            <div>
                                                <Label className="text-xs text-muted-foreground mb-2 block">Roles</Label>
                                                <div className="flex flex-wrap gap-2">
                                                    {['resident', 'guard', 'vendor', 'staff'].map(role => (
                                                        <div key={role} className="flex items-center space-x-2">
                                                            <Checkbox 
                                                                id={`create_role_${role}`} 
                                                                checked={createForm.data.target_roles.includes(role)}
                                                                onCheckedChange={() => toggleRole(role)}
                                                            />
                                                            <Label htmlFor={`create_role_${role}`} className="capitalize">{role}</Label>
                                                        </div>
                                                    ))}
                                                </div>
                                            </div>
                                            <div>
                                                <Label className="text-xs text-muted-foreground mb-2 block">Specific Buildings</Label>
                                                <div className="border rounded-md p-2 h-32 overflow-y-auto space-y-2">
                                                    {buildings.length > 0 ? buildings.map(building => (
                                                        <div key={building.id} className="flex items-center space-x-2">
                                                            <Checkbox 
                                                                id={`create_building_${building.id}`}
                                                                checked={createForm.data.target_buildings.includes(building.id)}
                                                                onCheckedChange={() => toggleBuilding(building.id)}
                                                            />
                                                            <Label htmlFor={`create_building_${building.id}`}>
                                                                {building.name} {building.building_no ? `(${building.building_no})` : ''}
                                                            </Label>
                                                        </div>
                                                    )) : (
                                                        <div className="text-sm text-muted-foreground p-2">No buildings found</div>
                                                    )}
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    <div className="flex gap-4 pt-2">
                                        <div className="flex items-center space-x-2">
                                            <Checkbox 
                                                id="create_ack" 
                                                checked={createForm.data.require_acknowledgement}
                                                onCheckedChange={(checked) => createForm.setData('require_acknowledgement', checked === true)}
                                            />
                                            <Label htmlFor="create_ack">Require Acknowledgement</Label>
                                        </div>
                                        <div className="flex items-center space-x-2">
                                            <Checkbox 
                                                id="create_pub" 
                                                checked={createForm.data.is_published}
                                                onCheckedChange={(checked) => createForm.setData('is_published', checked === true)}
                                            />
                                            <Label htmlFor="create_pub">Publish Immediately</Label>
                                        </div>
                                    </div>

                                    <DialogFooter>
                                        <Button type="button" variant="outline" onClick={() => setIsCreateOpen(false)}>Cancel</Button>
                                        <Button type="submit" disabled={createForm.processing}>
                                            {createForm.processing ? 'Creating...' : 'Create Announcement'}
                                        </Button>
                                    </DialogFooter>
                                </form>
                            </DialogContent>
                        </Dialog>
                    }
                />

                {/* KPI Cards */}
                <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-4">
                    <GlassCard className="p-6">
                        <div className="flex items-center justify-between">
                            <div className="space-y-1">
                                <p className="text-sm font-medium text-muted-foreground">Total Announcements</p>
                                <p className="text-2xl font-bold bg-clip-text text-transparent bg-gradient-to-r from-primary to-primary/60">{stats.total}</p>
                            </div>
                            <div className="rounded-xl bg-primary/10 p-3 ring-1 ring-primary/20">
                                <Megaphone className="h-6 w-6 text-primary" />
                            </div>
                        </div>
                    </GlassCard>
                    <GlassCard className="p-6">
                        <div className="flex items-center justify-between">
                            <div className="space-y-1">
                                <p className="text-sm font-medium text-muted-foreground">Active Now</p>
                                <p className="text-2xl font-bold bg-clip-text text-transparent bg-gradient-to-r from-emerald-500 to-emerald-700">{stats.active}</p>
                            </div>
                            <div className="rounded-xl bg-emerald-500/10 p-3 ring-1 ring-emerald-500/20">
                                <CheckCircle2 className="h-6 w-6 text-emerald-500" />
                            </div>
                        </div>
                    </GlassCard>
                    <GlassCard className="p-6">
                        <div className="flex items-center justify-between">
                            <div className="space-y-1">
                                <p className="text-sm font-medium text-muted-foreground">Total Views</p>
                                <p className="text-2xl font-bold bg-clip-text text-transparent bg-gradient-to-r from-blue-500 to-blue-700">{stats.total_views}</p>
                            </div>
                            <div className="rounded-xl bg-blue-500/10 p-3 ring-1 ring-blue-500/20">
                                <Eye className="h-6 w-6 text-blue-500" />
                            </div>
                        </div>
                    </GlassCard>
                    <GlassCard className="p-6">
                        <div className="flex items-center justify-between">
                            <div className="space-y-1">
                                <p className="text-sm font-medium text-muted-foreground">Acknowledgements</p>
                                <p className="text-2xl font-bold bg-clip-text text-transparent bg-gradient-to-r from-orange-500 to-orange-700">{stats.acknowledgements}</p>
                            </div>
                            <div className="rounded-xl bg-orange-500/10 p-3 ring-1 ring-orange-500/20">
                                <Users className="h-6 w-6 text-orange-500" />
                            </div>
                        </div>
                    </GlassCard>
                </div>

                {/* Filters */}
                <GlassCard className="p-6">
                    <div className="flex flex-col md:flex-row gap-4 flex-wrap">
                        <div className="flex-1 min-w-[200px]">
                            <Label htmlFor="search">Search</Label>
                            <div className="relative mt-1">
                                <Search className="absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2 text-muted-foreground" />
                                <Input
                                    id="search"
                                    placeholder="Search by title or content..."
                                    value={search}
                                    onChange={(e) => setSearch(e.target.value)}
                                    onKeyDown={(e) => e.key === 'Enter' && handleSearch()}
                                    className="pl-9 bg-background/50"
                                />
                            </div>
                        </div>
                        <div className="w-full md:w-40">
                            <Label htmlFor="type">Type</Label>
                            <Select value={type} onValueChange={setType}>
                                <SelectTrigger id="type" className="mt-1 bg-background/50">
                                    <SelectValue placeholder="All Types" />
                                </SelectTrigger>
                                <SelectContent>
                                    <SelectItem value="all">All Types</SelectItem>
                                    {Object.entries(typeConfig).map(([t, c]) => (
                                        <SelectItem key={t} value={t}>
                                            {c.label}
                                        </SelectItem>
                                    ))}
                                </SelectContent>
                            </Select>
                        </div>
                        <div className="w-full md:w-40">
                            <Label htmlFor="priority">Priority</Label>
                            <Select value={priority} onValueChange={setPriority}>
                                <SelectTrigger id="priority" className="mt-1 bg-background/50">
                                    <SelectValue placeholder="All Priorities" />
                                </SelectTrigger>
                                <SelectContent>
                                    <SelectItem value="all">All Priorities</SelectItem>
                                    {Object.entries(priorityConfig).map(([p, c]) => (
                                        <SelectItem key={p} value={p}>
                                            {c.label}
                                        </SelectItem>
                                    ))}
                                </SelectContent>
                            </Select>
                        </div>
                        <div className="w-full md:w-40">
                            <Label htmlFor="is_published">Status</Label>
                            <Select value={isPublished} onValueChange={setIsPublished}>
                                <SelectTrigger id="is_published" className="mt-1 bg-background/50">
                                    <SelectValue placeholder="All Status" />
                                </SelectTrigger>
                                <SelectContent>
                                    <SelectItem value="all">All Status</SelectItem>
                                    <SelectItem value="1">Published</SelectItem>
                                    <SelectItem value="0">Unpublished</SelectItem>
                                </SelectContent>
                            </Select>
                        </div>
                        <div className="flex items-end">
                            <Button onClick={handleSearch} variant="secondary">Filter</Button>
                        </div>
                    </div>
                </GlassCard>

                {/* Announcements Table */}
                <GlassCard className="overflow-hidden">
                    <ModernTable>
                        <ModernTableHeader>
                            <ModernTableRow>
                                <ModernTableHead>Title</ModernTableHead>
                                <ModernTableHead>Type</ModernTableHead>
                                <ModernTableHead>Priority</ModernTableHead>
                                <ModernTableHead>Target</ModernTableHead>
                                <ModernTableHead>Schedule</ModernTableHead>
                                <ModernTableHead>Views/Ack</ModernTableHead>
                                <ModernTableHead>Status</ModernTableHead>
                                <ModernTableHead>Created By</ModernTableHead>
                                <ModernTableHead className="text-right">Actions</ModernTableHead>
                            </ModernTableRow>
                        </ModernTableHeader>
                        <ModernTableBody>
                            {announcements.data.length > 0 ? (
                                announcements.data.map((announcement) => {
                                    const TypeIcon = typeConfig[announcement.type as keyof typeof typeConfig].icon;
                                    return (
                                        <ModernTableRow key={announcement.id}>
                                            <ModernTableCell>
                                                <div className="flex items-center gap-3">
                                                    <div className={cn("p-2 rounded-lg", typeConfig[announcement.type as keyof typeof typeConfig].color.split(' ')[0])}>
                                                        <Megaphone className={cn("h-4 w-4", typeConfig[announcement.type as keyof typeof typeConfig].color.split(' ')[1])} />
                                                    </div>
                                                    <span className="font-medium">{announcement.title}</span>
                                                </div>
                                            </ModernTableCell>
                                            <ModernTableCell>
                                                <div className="flex items-center gap-2">
                                                    <Badge variant="outline" className={cn("font-medium", typeConfig[announcement.type as keyof typeof typeConfig].color)}>
                                                        {TypeIcon && <TypeIcon className="w-3 h-3 mr-1" />}
                                                        {typeConfig[announcement.type as keyof typeof typeConfig].label}
                                                    </Badge>
                                                </div>
                                            </ModernTableCell>
                                            <ModernTableCell>
                                                <Badge variant="outline" className={cn("font-medium", priorityConfig[announcement.priority as keyof typeof priorityConfig].color)}>
                                                    {priorityConfig[announcement.priority as keyof typeof priorityConfig].label}
                                                </Badge>
                                            </ModernTableCell>
                                            <ModernTableCell>
                                                <div className="text-sm text-muted-foreground">
                                                    {(!announcement.target_roles?.length && !announcement.target_buildings?.length && !announcement.target_units?.length) ? (
                                                        <Badge variant="secondary">All</Badge>
                                                    ) : (
                                                        <div className="flex flex-col gap-1">
                                                            {announcement.target_roles?.length ? (
                                                                <span className="text-xs">Roles: {announcement.target_roles.join(', ')}</span>
                                                            ) : <span className="text-xs">Roles: All</span>}
                                                            
                                                            {announcement.target_buildings && announcement.target_buildings.length > 0 && (
                                                                <span className="text-xs">Buildings: {announcement.target_buildings.length}</span>
                                                            )}
                                                            
                                                            {(!announcement.target_buildings || announcement.target_buildings.length === 0) && announcement.target_units && announcement.target_units.length > 0 && (
                                                                <span className="text-xs">Units: {announcement.target_units.length}</span>
                                                            )}
                                                        </div>
                                                    )}
                                                </div>
                                            </ModernTableCell>
                                            <ModernTableCell>
                                                <div className="flex flex-col gap-1 text-sm text-muted-foreground">
                                                    {announcement.start_date ? (
                                                        <>
                                                            <span className="text-nowrap text-xs">Start: {new Date(announcement.start_date).toLocaleDateString()}</span>
                                                            {announcement.end_date && (
                                                                <span className="text-nowrap text-xs">End: {new Date(announcement.end_date).toLocaleDateString()}</span>
                                                            )}
                                                        </>
                                                    ) : (
                                                        <span className="text-xs">Immediate</span>
                                                    )}
                                                </div>
                                            </ModernTableCell>
                                            <ModernTableCell>
                                                <div className="flex flex-col gap-1">
                                                    <div className="flex items-center text-xs">
                                                        <Eye className="w-3 h-3 mr-1 text-muted-foreground" />
                                                        {announcement.views_count}
                                                    </div>
                                                    {announcement.require_acknowledgement && (
                                                        <div className="flex items-center text-xs text-green-600 dark:text-green-400">
                                                            <CheckCircle2 className="w-3 h-3 mr-1" />
                                                            {announcement.acknowledged_count}
                                                        </div>
                                                    )}
                                                </div>
                                            </ModernTableCell>
                                            <ModernTableCell>
                                                <Badge
                                                    variant={announcement.is_published ? 'default' : 'secondary'}
                                                    className={cn(
                                                        announcement.is_published 
                                                        ? 'bg-emerald-500/10 text-emerald-600 border-emerald-500/20 hover:bg-emerald-500/20' 
                                                        : 'bg-slate-500/10 text-slate-600 border-slate-500/20 hover:bg-slate-500/20'
                                                    )}
                                                >
                                                    {announcement.is_published ? 'Published' : 'Unpublished'}
                                                </Badge>
                                            </ModernTableCell>
                                            <ModernTableCell>
                                                <span className="text-sm font-medium">{announcement.created_by.name}</span>
                                            </ModernTableCell>
                                            <ModernTableCell className="text-right">
                                                <div className="flex items-center justify-end gap-2">
                                                    <Button
                                                        variant="ghost"
                                                        size="icon"
                                                        className="h-8 w-8 hover:bg-muted"
                                                        onClick={() => handleTogglePublish(announcement.id)}
                                                        title={announcement.is_published ? "Unpublish" : "Publish"}
                                                    >
                                                        {announcement.is_published ? (
                                                            <EyeOff className="h-4 w-4 text-orange-500" />
                                                        ) : (
                                                            <Eye className="h-4 w-4 text-green-500" />
                                                        )}
                                                    </Button>
                                                    <Button 
                                                        variant="ghost" 
                                                        size="icon"
                                                        className="h-8 w-8 hover:bg-muted"
                                                        onClick={() => openEditModal(announcement)}
                                                    >
                                                        <Edit className="h-4 w-4" />
                                                    </Button>
                                                    <Button
                                                        variant="ghost"
                                                        size="icon"
                                                        className="h-8 w-8 hover:bg-destructive/10 text-destructive"
                                                        onClick={() => handleDelete(announcement.id)}
                                                    >
                                                        <Trash2 className="h-4 w-4" />
                                                    </Button>
                                                </div>
                                            </ModernTableCell>
                                        </ModernTableRow>
                                    );
                                })
                            ) : (
                                <ModernTableRow>
                                    <ModernTableCell colSpan={9} className="h-24 text-center text-muted-foreground">
                                        No announcements found
                                    </ModernTableCell>
                                </ModernTableRow>
                            )}
                        </ModernTableBody>
                    </ModernTable>
                    
                    {/* Pagination */}
                    {announcements.links && announcements.links.length > 3 && (
                        <div className="flex items-center justify-between p-4 border-t border-border/40">
                            <div className="text-sm text-muted-foreground">
                                Showing {announcements.meta.from} to {announcements.meta.to} of {announcements.meta.total} results
                            </div>
                            <div className="flex gap-2">
                                {announcements.links.map((link: any, index: number) => (
                                    <button
                                        key={index}
                                        onClick={() => link.url && router.visit(link.url)}
                                        disabled={!link.url}
                                        className={cn(
                                            "px-3 py-1 rounded-md text-sm transition-colors",
                                            link.active
                                                ? 'bg-primary text-primary-foreground'
                                                : 'bg-muted hover:bg-muted/80 text-muted-foreground',
                                            !link.url && 'opacity-50 cursor-not-allowed hidden'
                                        )}
                                        dangerouslySetInnerHTML={{ __html: link.label }}
                                    />
                                ))}
                            </div>
                        </div>
                    )}
                </GlassCard>

                {/* Edit Modal */}
                <Dialog open={isEditOpen} onOpenChange={setIsEditOpen}>
                    <DialogContent className="max-w-3xl max-h-[90vh] overflow-y-auto">
                        <DialogHeader>
                            <DialogTitle>Edit Announcement</DialogTitle>
                            <DialogDescription>Update announcement details</DialogDescription>
                        </DialogHeader>
                        <form onSubmit={handleEditSubmit} className="space-y-4">
                            <div className="grid grid-cols-2 gap-4">
                                <div className="col-span-2 space-y-2">
                                    <Label htmlFor="edit_title">Title *</Label>
                                    <Input
                                        id="edit_title"
                                        value={editForm.data.title}
                                        onChange={e => editForm.setData('title', e.target.value)}
                                        required
                                    />
                                </div>
                                <div className="col-span-2 space-y-2">
                                    <Label htmlFor="edit_content">Content *</Label>
                                    <Textarea
                                        id="edit_content"
                                        value={editForm.data.content}
                                        onChange={e => editForm.setData('content', e.target.value)}
                                        rows={4}
                                        required
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_type">Type</Label>
                                    <Select value={editForm.data.type} onValueChange={v => editForm.setData('type', v)}>
                                        <SelectTrigger id="edit_type"><SelectValue /></SelectTrigger>
                                        <SelectContent>
                                            {Object.entries(typeConfig).map(([k, v]) => (
                                                <SelectItem key={k} value={k}>{v.label}</SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_priority">Priority</Label>
                                    <Select value={editForm.data.priority} onValueChange={v => editForm.setData('priority', v)}>
                                        <SelectTrigger id="edit_priority"><SelectValue /></SelectTrigger>
                                        <SelectContent>
                                            {Object.entries(priorityConfig).map(([k, v]) => (
                                                <SelectItem key={k} value={k}>{v.label}</SelectItem>
                                            ))}
                                        </SelectContent>
                                    </Select>
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_start_date">Start Date (Optional)</Label>
                                    <Input 
                                        type="datetime-local" 
                                        value={editForm.data.start_date || ''}
                                        onChange={e => editForm.setData('start_date', e.target.value)}
                                    />
                                </div>
                                <div className="space-y-2">
                                    <Label htmlFor="edit_end_date">End Date (Optional)</Label>
                                    <Input 
                                        type="datetime-local" 
                                        value={editForm.data.end_date || ''}
                                        onChange={e => editForm.setData('end_date', e.target.value)}
                                    />
                                </div>
                            </div>

                            <div className="space-y-2 border-t pt-4">
                                <Label>Target Audience</Label>
                                <div className="grid grid-cols-2 gap-4">
                                    <div>
                                        <Label className="text-xs text-muted-foreground mb-2 block">Roles (Empty = All)</Label>
                                        <div className="flex flex-wrap gap-2">
                                            {['resident', 'guard', 'vendor', 'staff'].map(role => (
                                                <div key={role} className="flex items-center space-x-2">
                                                    <Checkbox 
                                                        id={`edit_role_${role}`} 
                                                        checked={editForm.data.target_roles.includes(role)}
                                                        onCheckedChange={() => toggleRole(role, true)}
                                                    />
                                                    <Label htmlFor={`edit_role_${role}`} className="capitalize">{role}</Label>
                                                </div>
                                            ))}
                                        </div>
                                    </div>
                                    <div>
                                        <Label className="text-xs text-muted-foreground mb-2 block">Specific Buildings (Empty = All)</Label>
                                        <div className="border rounded p-2 h-32 overflow-y-auto space-y-2">
                                            {buildings.length > 0 ? buildings.map(building => (
                                                <div key={building.id} className="flex items-center space-x-2">
                                                    <Checkbox 
                                                        id={`edit_building_${building.id}`}
                                                        checked={editForm.data.target_buildings.includes(building.id)}
                                                        onCheckedChange={() => toggleBuilding(building.id, true)}
                                                    />
                                                    <Label htmlFor={`edit_building_${building.id}`}>
                                                        {building.name} {building.building_no ? `(${building.building_no})` : ''}
                                                    </Label>
                                                </div>
                                            )) : (
                                                <div className="text-sm text-muted-foreground p-2">No buildings found</div>
                                            )}
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div className="flex gap-4 pt-2">
                                <div className="flex items-center space-x-2">
                                    <Checkbox 
                                        id="edit_ack" 
                                        checked={editForm.data.require_acknowledgement}
                                        onCheckedChange={(checked) => editForm.setData('require_acknowledgement', checked === true)}
                                    />
                                    <Label htmlFor="edit_ack">Require Acknowledgement</Label>
                                </div>
                            </div>
                            
                            <DialogFooter>
                                <Button type="button" variant="outline" onClick={() => setIsEditOpen(false)}>Cancel</Button>
                                <Button type="submit" disabled={editForm.processing}>
                                    {editForm.processing ? 'Updating...' : 'Update Announcement'}
                                </Button>
                            </DialogFooter>
                        </form>
                    </DialogContent>
                </Dialog>
            </div>
        </SocietyAdminLayout>
    );
}
