import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import PublicLayout from '@/layouts/public-layout';
import { Head, Link, usePage } from '@inertiajs/react';
import { Check, X, HelpCircle } from 'lucide-react';
import { useState } from 'react';
import { Switch } from '@/components/ui/switch';
import { Label } from '@/components/ui/label';
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from '@/components/ui/tooltip';

interface Plan {
    id: number;
    name: string;
    display_name: string;
    description: string | null;
    monthly_price: number;
    quarterly_price: number | null;
    annual_price: number | null;
    max_users: number;
    max_units: number;
    max_buildings: number;
    storage_gb: number;
    validity_days: number | null;
    // Features
    sms_enabled: boolean;
    whatsapp_enabled: boolean;
    email_enabled: boolean;
    fcm_enabled: boolean;
    advanced_reports: boolean;
    api_access: boolean;
    custom_branding: boolean;
    priority_support: boolean;
    white_label: boolean;
}

interface PricingProps {
    plans: Plan[];
}

export default function Pricing({ plans = [] }: PricingProps) {
    const { settings } = usePage().props as any;
    const [billingCycle, setBillingCycle] = useState<'monthly' | 'yearly'>('monthly');

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 0,
        }).format(amount);
    };

    return (
        <PublicLayout>
            <Head title="Pricing" />

            <section className="py-24 bg-background">
                <div className="container mx-auto px-4">
                    <div className="text-center max-w-3xl mx-auto mb-12">
                        <h1 className="text-4xl md:text-5xl font-bold mb-6">Simple, Transparent Pricing</h1>
                        <p className="text-muted-foreground text-lg mb-8">
                            Choose the plan that fits your society's needs. No hidden fees.
                        </p>

                        <div className="flex items-center justify-center gap-4">
                            <Label htmlFor="billing-toggle" className={`text-lg cursor-pointer ${billingCycle === 'monthly' ? 'text-foreground font-semibold' : 'text-muted-foreground'}`}>Monthly</Label>
                            <Switch
                                id="billing-toggle"
                                checked={billingCycle === 'yearly'}
                                onCheckedChange={(checked) => setBillingCycle(checked ? 'yearly' : 'monthly')}
                            />
                            <Label htmlFor="billing-toggle" className={`text-lg cursor-pointer ${billingCycle === 'yearly' ? 'text-foreground font-semibold' : 'text-muted-foreground'}`}>
                                Yearly <span className="text-xs text-emerald-600 font-bold ml-1">(Save up to 20%)</span>
                            </Label>
                        </div>
                    </div>

                    <div className="grid md:grid-cols-3 gap-8 max-w-7xl mx-auto">
                        {plans.map((plan) => {
                            const price = billingCycle === 'monthly' ? plan.monthly_price : (plan.annual_price ? Math.round(plan.annual_price / 12) : plan.monthly_price);
                            const isYearly = billingCycle === 'yearly' && plan.annual_price;
                            
                            return (
                                <Card key={plan.id} className={`flex flex-col relative overflow-hidden transition-all duration-300 hover:shadow-xl ${plan.name === 'premium' ? 'border-emerald-500 shadow-lg scale-105 z-10 ring-1 ring-emerald-500/50' : 'hover:-translate-y-1'}`}>
                                    {plan.name === 'premium' && (
                                        <div className="absolute top-0 inset-x-0 h-1.5 bg-gradient-to-r from-emerald-500 to-teal-500" />
                                    )}
                                    {plan.name === 'premium' && (
                                        <Badge className="absolute top-4 right-4 bg-emerald-500 hover:bg-emerald-600">Most Popular</Badge>
                                    )}
                                    
                                    <CardHeader>
                                        <CardTitle className="text-2xl">{plan.display_name}</CardTitle>
                                        <CardDescription className="min-h-[40px]">{plan.description}</CardDescription>
                                    </CardHeader>
                                    <CardContent className="flex-1">
                                        <div className="mb-8">
                                            <div className="flex items-baseline">
                                                <span className="text-4xl font-bold">{formatCurrency(price)}</span>
                                                <span className="text-muted-foreground ml-2">/month</span>
                                            </div>
                                            {isYearly && (
                                                <p className="text-sm text-emerald-600 mt-2 font-medium bg-emerald-50 inline-block px-2 py-1 rounded">
                                                    Billed {formatCurrency(plan.annual_price!)} yearly
                                                </p>
                                            )}
                                        </div>

                                        <div className="space-y-4 text-sm">
                                            <div className="font-semibold text-foreground mb-2">What's included:</div>
                                            
                                            <div className="flex items-center gap-3">
                                                <div className="h-6 w-6 rounded-full bg-emerald-100 flex items-center justify-center shrink-0">
                                                    <Check className="h-3.5 w-3.5 text-emerald-600" />
                                                </div>
                                                <span>{plan.max_units === 9999 ? 'Unlimited' : plan.max_units} Units</span>
                                            </div>
                                            <div className="flex items-center gap-3">
                                                <div className="h-6 w-6 rounded-full bg-emerald-100 flex items-center justify-center shrink-0">
                                                    <Check className="h-3.5 w-3.5 text-emerald-600" />
                                                </div>
                                                <span>{plan.max_users === 9999 ? 'Unlimited' : plan.max_users} Users</span>
                                            </div>
                                            <div className="flex items-center gap-3">
                                                <div className="h-6 w-6 rounded-full bg-emerald-100 flex items-center justify-center shrink-0">
                                                    <Check className="h-3.5 w-3.5 text-emerald-600" />
                                                </div>
                                                <span>{plan.storage_gb} GB Storage</span>
                                            </div>
                                            
                                            <div className="h-px bg-border my-4" />
                                            
                                            {/* Feature List */}
                                            {[
                                                { key: 'email_enabled', label: 'Email Notifications' },
                                                { key: 'sms_enabled', label: 'SMS Notifications' },
                                                { key: 'whatsapp_enabled', label: 'WhatsApp Integration' },
                                                { key: 'fcm_enabled', label: 'Mobile App Access' },
                                                { key: 'advanced_reports', label: 'Advanced Reports' },
                                                { key: 'api_access', label: 'API Access' },
                                                { key: 'custom_branding', label: 'Custom Branding' },
                                                { key: 'priority_support', label: 'Priority Support' },
                                            ].map((feature) => (
                                                <div key={feature.key} className="flex items-center gap-3">
                                                    {plan[feature.key as keyof Plan] ? (
                                                        <div className="h-6 w-6 rounded-full bg-emerald-100 flex items-center justify-center shrink-0">
                                                            <Check className="h-3.5 w-3.5 text-emerald-600" />
                                                        </div>
                                                    ) : (
                                                        <div className="h-6 w-6 rounded-full bg-muted flex items-center justify-center shrink-0">
                                                            <X className="h-3.5 w-3.5 text-muted-foreground" />
                                                        </div>
                                                    )}
                                                    <span className={!plan[feature.key as keyof Plan] ? 'text-muted-foreground/60' : ''}>
                                                        {feature.label}
                                                    </span>
                                                </div>
                                            ))}
                                        </div>
                                    </CardContent>
                                    <CardFooter>
                                        {settings?.enable_registration === '1' ? (
                                            <Link href={`/register?plan=${plan.id}&billing=${billingCycle}`} className="w-full">
                                                <Button size="lg" className={`w-full ${plan.name === 'premium' ? 'bg-gradient-to-r from-emerald-600 to-teal-600 hover:from-emerald-700 hover:to-teal-700 shadow-md' : ''}`}>
                                                    Choose {plan.display_name}
                                                </Button>
                                            </Link>
                                        ) : (
                                            <Button disabled className="w-full" variant="secondary">Registration Closed</Button>
                                        )}
                                    </CardFooter>
                                </Card>
                            );
                        })}
                    </div>
                </div>
            </section>
        </PublicLayout>
    );
}
