import { Button } from '@/components/ui/button';
import { useAppearance } from '@/hooks/use-appearance';
import { Head, router } from '@inertiajs/react';
import { 
    Rocket, 
    Database, 
    Shield, 
    Settings, 
    CheckCircle2,
    Moon, 
    Sun, 
    ChevronRight,
    Sparkles
} from 'lucide-react';

export default function Welcome() {
    const { appearance, updateAppearance } = useAppearance();

    const handleStart = () => {
        router.get('/install/requirements');
    };

    const steps = [
        { icon: CheckCircle2, title: 'Requirements', description: 'Check system requirements' },
        { icon: Database, title: 'Database', description: 'Configure your database' },
        { icon: Shield, title: 'Admin', description: 'Create super admin account' },
        { icon: Sparkles, title: 'Finish', description: 'Complete installation' },
    ];

    return (
        <>
            <Head title="Installation Wizard" />
            
            <div className="min-h-screen bg-gradient-to-br from-emerald-50 via-teal-50 to-cyan-50 dark:from-slate-900 dark:via-emerald-950 dark:to-teal-950 flex items-center justify-center p-6">
                {/* Background Orbs */}
                <div className="absolute top-0 left-0 w-96 h-96 bg-emerald-200/40 dark:bg-emerald-500/20 rounded-full blur-3xl animate-pulse" />
                <div className="absolute bottom-0 right-0 w-96 h-96 bg-teal-200/40 dark:bg-teal-500/20 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '2s' }} />
                
                {/* Theme Toggle */}
                <div className="fixed top-6 right-6 z-20">
                    <div className="inline-flex items-center rounded-full bg-white/80 dark:bg-slate-800/80 backdrop-blur-sm p-1 shadow-lg border border-border/50">
                        <button
                            onClick={() => updateAppearance('light')}
                            className={`inline-flex items-center gap-1.5 rounded-full px-3 py-2 text-sm font-medium transition-all ${
                                appearance === 'light'
                                    ? 'bg-white dark:bg-slate-700 text-foreground shadow-sm'
                                    : 'text-muted-foreground hover:text-foreground'
                            }`}
                        >
                            <Sun className="h-4 w-4" />
                        </button>
                        <button
                            onClick={() => updateAppearance('dark')}
                            className={`inline-flex items-center gap-1.5 rounded-full px-3 py-2 text-sm font-medium transition-all ${
                                appearance === 'dark'
                                    ? 'bg-white dark:bg-slate-700 text-foreground shadow-sm'
                                    : 'text-muted-foreground hover:text-foreground'
                            }`}
                        >
                            <Moon className="h-4 w-4" />
                        </button>
                    </div>
                </div>
                
                <div className="relative z-10 w-full max-w-2xl">
                    {/* Main Card */}
                    <div className="rounded-3xl border border-border/50 bg-white/80 dark:bg-slate-900/80 backdrop-blur-xl p-8 md:p-12 shadow-2xl">
                        {/* Logo */}
                        <div className="flex justify-center mb-8">
                            <div className="flex h-20 w-20 items-center justify-center rounded-2xl bg-gradient-to-br from-emerald-500 to-teal-600 shadow-xl">
                                <Rocket className="h-10 w-10 text-white" />
                            </div>
                        </div>
                        
                        {/* Heading */}
                        <div className="text-center mb-8">
                            <h1 className="text-3xl md:text-4xl font-bold mb-3 text-foreground">
                                Welcome to{' '}
                                <span className="bg-gradient-to-r from-emerald-600 to-teal-600 bg-clip-text text-transparent">
                                    Societify
                                </span>
                            </h1>
                            <p className="text-muted-foreground text-lg">
                                Let's get your society management platform set up in just a few steps.
                            </p>
                        </div>
                        
                        {/* Installation Steps Preview */}
                        <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-10">
                            {steps.map((step, index) => (
                                <div 
                                    key={step.title}
                                    className="flex flex-col items-center p-4 rounded-xl bg-muted/50 border border-border/50"
                                >
                                    <div className={`w-10 h-10 rounded-full flex items-center justify-center mb-2 ${
                                        index === 0 
                                            ? 'bg-gradient-to-br from-emerald-500 to-teal-600 text-white' 
                                            : 'bg-muted text-muted-foreground'
                                    }`}>
                                        <step.icon className="w-5 h-5" />
                                    </div>
                                    <span className="text-sm font-medium text-foreground">{step.title}</span>
                                    <span className="text-xs text-muted-foreground text-center">{step.description}</span>
                                </div>
                            ))}
                        </div>
                        
                        {/* Get Started Button */}
                        <div className="flex justify-center">
                            <Button
                                onClick={handleStart}
                                size="lg"
                                className="h-12 px-8 text-base font-medium bg-gradient-to-r from-emerald-600 to-teal-600 hover:from-emerald-700 hover:to-teal-700 text-white shadow-lg hover:shadow-xl transition-all group"
                            >
                                Get Started
                                <ChevronRight className="ml-2 h-5 w-5 group-hover:translate-x-1 transition-transform" />
                            </Button>
                        </div>
                        
                        {/* Footer */}
                        <div className="mt-8 pt-6 border-t border-border/50 text-center">
                            <p className="text-sm text-muted-foreground">
                                <Settings className="w-4 h-4 inline mr-1" />
                                Installation typically takes 2-3 minutes
                            </p>
                        </div>
                    </div>
                    
                    {/* Version Info */}
                    <p className="mt-6 text-center text-sm text-muted-foreground">
                        Societify v1.0.0 • Society Management Platform
                    </p>
                </div>
            </div>
        </>
    );
}
