import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { useAppearance } from '@/hooks/use-appearance';
import { Head, router, usePage } from '@inertiajs/react';
import { 
    CheckCircle2, 
    XCircle, 
    RefreshCw, 
    ChevronRight, 
    ChevronLeft,
    Moon, 
    Sun,
    Server,
    FolderOpen,
    Cpu
} from 'lucide-react';

interface RequirementsProps {
    requirements: {
        php: {
            name: string;
            required: string;
            current: string;
            passed: boolean;
        };
        extensions: Record<string, { name: string; passed: boolean }>;
        directories: Record<string, { name: string; passed: boolean }>;
    };
    allPassed: boolean;
    errors?: {
        requirements?: string;
    };
}

export default function Requirements({ requirements, allPassed, errors }: RequirementsProps) {
    const { appearance, updateAppearance } = useAppearance();

    const handleCheck = () => {
        router.post('/install/requirements');
    };

    const handleRefresh = () => {
        router.reload();
    };

    const handleBack = () => {
        router.get('/install/welcome');
    };

    return (
        <>
            <Head title="Requirements Check - Installation" />
            
            <div className="min-h-screen bg-gradient-to-br from-emerald-50 via-teal-50 to-cyan-50 dark:from-slate-900 dark:via-emerald-950 dark:to-teal-950 flex items-center justify-center p-6">
                {/* Background Orbs */}
                <div className="absolute top-0 left-0 w-96 h-96 bg-emerald-200/40 dark:bg-emerald-500/20 rounded-full blur-3xl animate-pulse" />
                <div className="absolute bottom-0 right-0 w-96 h-96 bg-teal-200/40 dark:bg-teal-500/20 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '2s' }} />
                
                {/* Theme Toggle */}
                <div className="fixed top-6 right-6 z-20">
                    <div className="inline-flex items-center rounded-full bg-white/80 dark:bg-slate-800/80 backdrop-blur-sm p-1 shadow-lg border border-border/50">
                        <button
                            onClick={() => updateAppearance('light')}
                            className={`inline-flex items-center gap-1.5 rounded-full px-3 py-2 text-sm font-medium transition-all ${
                                appearance === 'light'
                                    ? 'bg-white dark:bg-slate-700 text-foreground shadow-sm'
                                    : 'text-muted-foreground hover:text-foreground'
                            }`}
                        >
                            <Sun className="h-4 w-4" />
                        </button>
                        <button
                            onClick={() => updateAppearance('dark')}
                            className={`inline-flex items-center gap-1.5 rounded-full px-3 py-2 text-sm font-medium transition-all ${
                                appearance === 'dark'
                                    ? 'bg-white dark:bg-slate-700 text-foreground shadow-sm'
                                    : 'text-muted-foreground hover:text-foreground'
                            }`}
                        >
                            <Moon className="h-4 w-4" />
                        </button>
                    </div>
                </div>
                
                <div className="relative z-10 w-full max-w-2xl">
                    {/* Progress Bar */}
                    <div className="mb-6">
                        <div className="flex items-center justify-between text-sm text-muted-foreground mb-2">
                            <span>Step 1 of 4</span>
                            <span>25%</span>
                        </div>
                        <div className="h-2 bg-muted rounded-full overflow-hidden">
                            <div className="h-full w-1/4 bg-gradient-to-r from-emerald-500 to-teal-600 rounded-full transition-all duration-500" />
                        </div>
                    </div>
                    
                    {/* Main Card */}
                    <div className="rounded-3xl border border-border/50 bg-white/80 dark:bg-slate-900/80 backdrop-blur-xl p-8 shadow-2xl">
                        {/* Heading */}
                        <div className="flex items-center gap-3 mb-6">
                            <div className="w-12 h-12 rounded-xl bg-gradient-to-br from-emerald-500 to-teal-600 flex items-center justify-center">
                                <Server className="w-6 h-6 text-white" />
                            </div>
                            <div>
                                <h1 className="text-2xl font-bold text-foreground">Requirements Check</h1>
                                <p className="text-muted-foreground">Verify system compatibility</p>
                            </div>
                        </div>
                        
                        {errors?.requirements && (
                            <InputError message={errors.requirements} className="mb-4" />
                        )}
                        
                        {/* PHP Version */}
                        <div className="mb-6">
                            <h3 className="text-sm font-semibold text-foreground flex items-center gap-2 mb-3">
                                <Cpu className="w-4 h-4" />
                                PHP Version
                            </h3>
                            <div className={`flex items-center justify-between p-4 rounded-xl border ${
                                requirements.php.passed 
                                    ? 'bg-emerald-50 dark:bg-emerald-950/30 border-emerald-200 dark:border-emerald-800'
                                    : 'bg-red-50 dark:bg-red-950/30 border-red-200 dark:border-red-800'
                            }`}>
                                <div>
                                    <p className="font-medium text-foreground">PHP {requirements.php.current}</p>
                                    <p className="text-sm text-muted-foreground">Required: {requirements.php.required}+</p>
                                </div>
                                {requirements.php.passed ? (
                                    <CheckCircle2 className="w-6 h-6 text-emerald-600 dark:text-emerald-400" />
                                ) : (
                                    <XCircle className="w-6 h-6 text-red-600 dark:text-red-400" />
                                )}
                            </div>
                        </div>
                        
                        {/* PHP Extensions */}
                        <div className="mb-6">
                            <h3 className="text-sm font-semibold text-foreground flex items-center gap-2 mb-3">
                                <Server className="w-4 h-4" />
                                PHP Extensions
                            </h3>
                            <div className="grid grid-cols-2 md:grid-cols-3 gap-2">
                                {Object.values(requirements.extensions).map((ext) => (
                                    <div 
                                        key={ext.name}
                                        className={`flex items-center gap-2 p-3 rounded-lg border ${
                                            ext.passed 
                                                ? 'bg-emerald-50/50 dark:bg-emerald-950/20 border-emerald-200/50 dark:border-emerald-800/50'
                                                : 'bg-red-50/50 dark:bg-red-950/20 border-red-200/50 dark:border-red-800/50'
                                        }`}
                                    >
                                        {ext.passed ? (
                                            <CheckCircle2 className="w-4 h-4 text-emerald-600 dark:text-emerald-400 flex-shrink-0" />
                                        ) : (
                                            <XCircle className="w-4 h-4 text-red-600 dark:text-red-400 flex-shrink-0" />
                                        )}
                                        <span className="text-sm font-medium text-foreground">{ext.name}</span>
                                    </div>
                                ))}
                            </div>
                        </div>
                        
                        {/* Directory Permissions */}
                        <div className="mb-8">
                            <h3 className="text-sm font-semibold text-foreground flex items-center gap-2 mb-3">
                                <FolderOpen className="w-4 h-4" />
                                Directory Permissions
                            </h3>
                            <div className="space-y-2">
                                {Object.values(requirements.directories).map((dir) => (
                                    <div 
                                        key={dir.name}
                                        className={`flex items-center justify-between p-3 rounded-lg border ${
                                            dir.passed 
                                                ? 'bg-emerald-50/50 dark:bg-emerald-950/20 border-emerald-200/50 dark:border-emerald-800/50'
                                                : 'bg-red-50/50 dark:bg-red-950/20 border-red-200/50 dark:border-red-800/50'
                                        }`}
                                    >
                                        <span className="text-sm font-mono text-foreground">{dir.name}</span>
                                        <div className="flex items-center gap-2">
                                            <span className={`text-xs ${dir.passed ? 'text-emerald-600 dark:text-emerald-400' : 'text-red-600 dark:text-red-400'}`}>
                                                {dir.passed ? 'Writable' : 'Not Writable'}
                                            </span>
                                            {dir.passed ? (
                                                <CheckCircle2 className="w-4 h-4 text-emerald-600 dark:text-emerald-400" />
                                            ) : (
                                                <XCircle className="w-4 h-4 text-red-600 dark:text-red-400" />
                                            )}
                                        </div>
                                    </div>
                                ))}
                            </div>
                        </div>
                        
                        {/* Actions */}
                        <div className="flex items-center justify-between">
                            <div className="flex items-center gap-2">
                                <Button
                                    type="button"
                                    variant="outline"
                                    onClick={handleBack}
                                >
                                    <ChevronLeft className="mr-1 h-4 w-4" />
                                    Back
                                </Button>
                                <Button
                                    type="button"
                                    variant="ghost"
                                    onClick={handleRefresh}
                                >
                                    <RefreshCw className="mr-1 h-4 w-4" />
                                    Refresh
                                </Button>
                            </div>
                            <Button
                                onClick={handleCheck}
                                disabled={!allPassed}
                                className="bg-gradient-to-r from-emerald-600 to-teal-600 hover:from-emerald-700 hover:to-teal-700 text-white shadow-lg"
                            >
                                Continue
                                <ChevronRight className="ml-1 h-4 w-4" />
                            </Button>
                        </div>
                    </div>
                </div>
            </div>
        </>
    );
}
