import { Button } from '@/components/ui/button';
import { useAppearance } from '@/hooks/use-appearance';
import { Head, router } from '@inertiajs/react';
import { 
    CheckCircle2,
    Moon, 
    Sun,
    ExternalLink,
    Copy,
    Check,
    Sparkles,
    PartyPopper
} from 'lucide-react';
import { useState } from 'react';

interface FinishProps {
    loginUrl: string;
    adminEmail: string;
}

export default function Finish({ loginUrl, adminEmail }: FinishProps) {
    const { appearance, updateAppearance } = useAppearance();
    const [copied, setCopied] = useState(false);

    const handleCopy = async () => {
        try {
            await navigator.clipboard.writeText(loginUrl);
            setCopied(true);
            setTimeout(() => setCopied(false), 2000);
        } catch (err) {
            console.error('Failed to copy:', err);
        }
    };

    const handleComplete = () => {
        router.post('/install/complete');
    };

    return (
        <>
            <Head title="Installation Complete" />
            
            <div className="min-h-screen bg-gradient-to-br from-emerald-50 via-teal-50 to-cyan-50 dark:from-slate-900 dark:via-emerald-950 dark:to-teal-950 flex items-center justify-center p-6">
                {/* Background Orbs */}
                <div className="absolute top-0 left-0 w-96 h-96 bg-emerald-200/40 dark:bg-emerald-500/20 rounded-full blur-3xl animate-pulse" />
                <div className="absolute bottom-0 right-0 w-96 h-96 bg-teal-200/40 dark:bg-teal-500/20 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '2s' }} />
                
                {/* Theme Toggle */}
                <div className="fixed top-6 right-6 z-20">
                    <div className="inline-flex items-center rounded-full bg-white/80 dark:bg-slate-800/80 backdrop-blur-sm p-1 shadow-lg border border-border/50">
                        <button
                            onClick={() => updateAppearance('light')}
                            className={`inline-flex items-center gap-1.5 rounded-full px-3 py-2 text-sm font-medium transition-all ${
                                appearance === 'light'
                                    ? 'bg-white dark:bg-slate-700 text-foreground shadow-sm'
                                    : 'text-muted-foreground hover:text-foreground'
                            }`}
                        >
                            <Sun className="h-4 w-4" />
                        </button>
                        <button
                            onClick={() => updateAppearance('dark')}
                            className={`inline-flex items-center gap-1.5 rounded-full px-3 py-2 text-sm font-medium transition-all ${
                                appearance === 'dark'
                                    ? 'bg-white dark:bg-slate-700 text-foreground shadow-sm'
                                    : 'text-muted-foreground hover:text-foreground'
                            }`}
                        >
                            <Moon className="h-4 w-4" />
                        </button>
                    </div>
                </div>
                
                <div className="relative z-10 w-full max-w-2xl">
                    {/* Progress Bar */}
                    <div className="mb-6">
                        <div className="flex items-center justify-between text-sm text-muted-foreground mb-2">
                            <span>Step 4 of 4</span>
                            <span>100%</span>
                        </div>
                        <div className="h-2 bg-muted rounded-full overflow-hidden">
                            <div className="h-full w-full bg-gradient-to-r from-emerald-500 to-teal-600 rounded-full transition-all duration-500" />
                        </div>
                    </div>
                    
                    {/* Main Card */}
                    <div className="rounded-3xl border border-border/50 bg-white/80 dark:bg-slate-900/80 backdrop-blur-xl p-8 shadow-2xl text-center">
                        {/* Success Icon */}
                        <div className="flex justify-center mb-6">
                            <div className="relative">
                                <div className="w-20 h-20 rounded-full bg-gradient-to-br from-emerald-500 to-teal-600 flex items-center justify-center">
                                    <CheckCircle2 className="w-10 h-10 text-white" />
                                </div>
                                <div className="absolute -top-2 -right-2">
                                    <PartyPopper className="w-8 h-8 text-amber-500" />
                                </div>
                            </div>
                        </div>
                        
                        {/* Heading */}
                        <h1 className="text-3xl font-bold text-foreground mb-2">
                            Installation Complete!
                        </h1>
                        <p className="text-muted-foreground text-lg mb-8">
                            <Sparkles className="w-5 h-5 inline mr-1 text-amber-500" />
                            Societify has been successfully installed
                        </p>
                        
                        {/* Admin Email */}
                        <div className="mb-6 p-4 rounded-xl bg-muted/50 border border-border/50">
                            <p className="text-sm text-muted-foreground mb-1">Admin Email</p>
                            <p className="font-semibold text-foreground">{adminEmail}</p>
                        </div>
                        
                        {/* Login URL */}
                        <div className="mb-8">
                            <p className="text-sm text-muted-foreground mb-3">Super Admin Login URL</p>
                            <div className="flex items-center gap-2 p-3 rounded-xl bg-muted/50 border border-border/50">
                                <code className="flex-1 text-sm text-foreground font-mono break-all text-left">
                                    {loginUrl}
                                </code>
                                <Button
                                    type="button"
                                    variant="outline"
                                    size="sm"
                                    onClick={handleCopy}
                                    className="flex-shrink-0"
                                >
                                    {copied ? (
                                        <>
                                            <Check className="w-4 h-4 mr-1 text-emerald-600" />
                                            Copied!
                                        </>
                                    ) : (
                                        <>
                                            <Copy className="w-4 h-4 mr-1" />
                                            Copy
                                        </>
                                    )}
                                </Button>
                            </div>
                        </div>
                        
                        {/* Important Notice */}
                        <div className="mb-8 p-4 rounded-xl bg-amber-50 dark:bg-amber-950/30 border border-amber-200 dark:border-amber-800 text-left">
                            <h3 className="font-semibold text-amber-800 dark:text-amber-200 mb-2">
                                🔐 Important Security Tips
                            </h3>
                            <ul className="text-sm text-amber-700 dark:text-amber-300 space-y-1">
                                <li>• Keep your login credentials secure</li>
                                <li>• Consider changing the password after first login</li>
                                <li>• Set up two-factor authentication if available</li>
                            </ul>
                        </div>
                        
                        {/* Go to Login Button */}
                        <Button
                            onClick={handleComplete}
                            size="lg"
                            className="w-full h-12 text-base font-medium bg-gradient-to-r from-emerald-600 to-teal-600 hover:from-emerald-700 hover:to-teal-700 text-white shadow-lg hover:shadow-xl transition-all"
                        >
                            Go to Admin Login
                            <ExternalLink className="ml-2 h-5 w-5" />
                        </Button>
                        
                        {/* Footer */}
                        <p className="mt-6 text-sm text-muted-foreground">
                            Thank you for choosing Societify!
                        </p>
                    </div>
                </div>
            </div>
        </>
    );
}
