import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Spinner } from '@/components/ui/spinner';
import { useAppearance } from '@/hooks/use-appearance';
import { Head, router, useForm } from '@inertiajs/react';
import { 
    Database, 
    Moon, 
    Sun,
    ChevronRight,
    ChevronLeft,
    Server,
    HardDrive
} from 'lucide-react';
import { useState } from 'react';

interface DatabaseProps {
    currentConnection: string;
    errors?: {
        database?: string;
        db_connection?: string;
        db_host?: string;
        db_port?: string;
        db_database?: string;
        db_username?: string;
        db_password?: string;
    };
}

export default function DatabaseConfig({ currentConnection, errors }: DatabaseProps) {
    const { appearance, updateAppearance } = useAppearance();
    const [dbType, setDbType] = useState<'sqlite' | 'mysql'>(currentConnection as 'sqlite' | 'mysql' || 'sqlite');
    
    const { data, setData, post, processing } = useForm({
        db_connection: dbType,
        db_host: '127.0.0.1',
        db_port: '3306',
        db_database: '',
        db_username: 'root',
        db_password: '',
    });

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        data.db_connection = dbType;
        post('/install/database');
    };

    const handleBack = () => {
        router.get('/install/requirements');
    };

    return (
        <>
            <Head title="Database Configuration - Installation" />
            
            <div className="min-h-screen bg-gradient-to-br from-emerald-50 via-teal-50 to-cyan-50 dark:from-slate-900 dark:via-emerald-950 dark:to-teal-950 flex items-center justify-center p-6">
                {/* Background Orbs */}
                <div className="absolute top-0 left-0 w-96 h-96 bg-emerald-200/40 dark:bg-emerald-500/20 rounded-full blur-3xl animate-pulse" />
                <div className="absolute bottom-0 right-0 w-96 h-96 bg-teal-200/40 dark:bg-teal-500/20 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '2s' }} />
                
                {/* Theme Toggle */}
                <div className="fixed top-6 right-6 z-20">
                    <div className="inline-flex items-center rounded-full bg-white/80 dark:bg-slate-800/80 backdrop-blur-sm p-1 shadow-lg border border-border/50">
                        <button
                            onClick={() => updateAppearance('light')}
                            className={`inline-flex items-center gap-1.5 rounded-full px-3 py-2 text-sm font-medium transition-all ${
                                appearance === 'light'
                                    ? 'bg-white dark:bg-slate-700 text-foreground shadow-sm'
                                    : 'text-muted-foreground hover:text-foreground'
                            }`}
                        >
                            <Sun className="h-4 w-4" />
                        </button>
                        <button
                            onClick={() => updateAppearance('dark')}
                            className={`inline-flex items-center gap-1.5 rounded-full px-3 py-2 text-sm font-medium transition-all ${
                                appearance === 'dark'
                                    ? 'bg-white dark:bg-slate-700 text-foreground shadow-sm'
                                    : 'text-muted-foreground hover:text-foreground'
                            }`}
                        >
                            <Moon className="h-4 w-4" />
                        </button>
                    </div>
                </div>
                
                <div className="relative z-10 w-full max-w-2xl">
                    {/* Progress Bar */}
                    <div className="mb-6">
                        <div className="flex items-center justify-between text-sm text-muted-foreground mb-2">
                            <span>Step 2 of 4</span>
                            <span>50%</span>
                        </div>
                        <div className="h-2 bg-muted rounded-full overflow-hidden">
                            <div className="h-full w-1/2 bg-gradient-to-r from-emerald-500 to-teal-600 rounded-full transition-all duration-500" />
                        </div>
                    </div>
                    
                    {/* Main Card */}
                    <div className="rounded-3xl border border-border/50 bg-white/80 dark:bg-slate-900/80 backdrop-blur-xl p-8 shadow-2xl">
                        {/* Heading */}
                        <div className="flex items-center gap-3 mb-6">
                            <div className="w-12 h-12 rounded-xl bg-gradient-to-br from-emerald-500 to-teal-600 flex items-center justify-center">
                                <Database className="w-6 h-6 text-white" />
                            </div>
                            <div>
                                <h1 className="text-2xl font-bold text-foreground">Database Configuration</h1>
                                <p className="text-muted-foreground">Choose and configure your database</p>
                            </div>
                        </div>
                        
                        {errors?.database && (
                            <InputError message={errors.database} className="mb-4" />
                        )}
                        
                        <form onSubmit={handleSubmit}>
                            {/* Database Type Selection */}
                            <div className="mb-6">
                                <Label className="text-sm font-semibold mb-3 block">Select Database Type</Label>
                                <div className="grid grid-cols-2 gap-4">
                                    <button
                                        type="button"
                                        onClick={() => setDbType('sqlite')}
                                        className={`p-4 rounded-xl border-2 transition-all ${
                                            dbType === 'sqlite'
                                                ? 'border-emerald-500 bg-emerald-50 dark:bg-emerald-950/30'
                                                : 'border-border hover:border-emerald-300 dark:hover:border-emerald-700'
                                        }`}
                                    >
                                        <HardDrive className={`w-8 h-8 mx-auto mb-2 ${
                                            dbType === 'sqlite' ? 'text-emerald-600' : 'text-muted-foreground'
                                        }`} />
                                        <h3 className="font-semibold text-foreground">SQLite</h3>
                                        <p className="text-xs text-muted-foreground mt-1">No configuration needed</p>
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() => setDbType('mysql')}
                                        className={`p-4 rounded-xl border-2 transition-all ${
                                            dbType === 'mysql'
                                                ? 'border-emerald-500 bg-emerald-50 dark:bg-emerald-950/30'
                                                : 'border-border hover:border-emerald-300 dark:hover:border-emerald-700'
                                        }`}
                                    >
                                        <Server className={`w-8 h-8 mx-auto mb-2 ${
                                            dbType === 'mysql' ? 'text-emerald-600' : 'text-muted-foreground'
                                        }`} />
                                        <h3 className="font-semibold text-foreground">MySQL</h3>
                                        <p className="text-xs text-muted-foreground mt-1">For production use</p>
                                    </button>
                                </div>
                            </div>
                            
                            {/* MySQL Configuration Fields */}
                            {dbType === 'mysql' && (
                                <div className="space-y-4 mb-6 p-4 rounded-xl bg-muted/50 border border-border/50">
                                    <div className="grid grid-cols-2 gap-4">
                                        <div className="space-y-2">
                                            <Label htmlFor="db_host">Host</Label>
                                            <Input
                                                id="db_host"
                                                value={data.db_host}
                                                onChange={(e) => setData('db_host', e.target.value)}
                                                placeholder="127.0.0.1"
                                            />
                                            <InputError message={errors?.db_host} />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="db_port">Port</Label>
                                            <Input
                                                id="db_port"
                                                value={data.db_port}
                                                onChange={(e) => setData('db_port', e.target.value)}
                                                placeholder="3306"
                                            />
                                            <InputError message={errors?.db_port} />
                                        </div>
                                    </div>
                                    <div className="space-y-2">
                                        <Label htmlFor="db_database">Database Name</Label>
                                        <Input
                                            id="db_database"
                                            value={data.db_database}
                                            onChange={(e) => setData('db_database', e.target.value)}
                                            placeholder="societify"
                                        />
                                        <InputError message={errors?.db_database} />
                                    </div>
                                    <div className="grid grid-cols-2 gap-4">
                                        <div className="space-y-2">
                                            <Label htmlFor="db_username">Username</Label>
                                            <Input
                                                id="db_username"
                                                value={data.db_username}
                                                onChange={(e) => setData('db_username', e.target.value)}
                                                placeholder="root"
                                            />
                                            <InputError message={errors?.db_username} />
                                        </div>
                                        <div className="space-y-2">
                                            <Label htmlFor="db_password">Password</Label>
                                            <Input
                                                id="db_password"
                                                type="password"
                                                value={data.db_password}
                                                onChange={(e) => setData('db_password', e.target.value)}
                                                placeholder="••••••••"
                                            />
                                            <InputError message={errors?.db_password} />
                                        </div>
                                    </div>
                                </div>
                            )}
                            
                            {/* SQLite Info */}
                            {dbType === 'sqlite' && (
                                <div className="mb-6 p-4 rounded-xl bg-emerald-50 dark:bg-emerald-950/30 border border-emerald-200 dark:border-emerald-800">
                                    <p className="text-sm text-foreground">
                                        <HardDrive className="w-4 h-4 inline mr-2 text-emerald-600" />
                                        SQLite database will be created automatically at <code className="text-xs bg-muted px-1 py-0.5 rounded">database/database.sqlite</code>
                                    </p>
                                </div>
                            )}

                            {/* Warning */}
                            <div className="mb-6 p-4 rounded-xl bg-amber-50 dark:bg-amber-950/30 border border-amber-200 dark:border-amber-800">
                                <p className="text-sm text-amber-800 dark:text-amber-200">
                                    ⚠️ This will run database migrations. Make sure your database is empty or backed up.
                                </p>
                            </div>
                            
                            {/* Actions */}
                            <div className="flex items-center justify-between">
                                <Button
                                    type="button"
                                    variant="outline"
                                    onClick={handleBack}
                                    disabled={processing}
                                >
                                    <ChevronLeft className="mr-1 h-4 w-4" />
                                    Back
                                </Button>
                                <Button
                                    type="submit"
                                    disabled={processing}
                                    className="bg-gradient-to-r from-emerald-600 to-teal-600 hover:from-emerald-700 hover:to-teal-700 text-white shadow-lg"
                                >
                                    {processing && <Spinner className="mr-2" />}
                                    {processing ? 'Configuring...' : 'Configure & Migrate'}
                                    {!processing && <ChevronRight className="ml-1 h-4 w-4" />}
                                </Button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </>
    );
}
