import InputError from '@/components/input-error';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Spinner } from '@/components/ui/spinner';
import { useAppearance } from '@/hooks/use-appearance';
import { Head, router, useForm } from '@inertiajs/react';
import { 
    Shield,
    Moon, 
    Sun,
    ChevronRight,
    ChevronLeft,
    User,
    Mail,
    Lock
} from 'lucide-react';

interface AdminProps {
    errors?: {
        admin?: string;
        name?: string;
        email?: string;
        password?: string;
        password_confirmation?: string;
    };
}

export default function Admin({ errors }: AdminProps) {
    const { appearance, updateAppearance } = useAppearance();
    
    const { data, setData, post, processing } = useForm({
        name: '',
        email: '',
        password: '',
        password_confirmation: '',
    });

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        post('/install/admin');
    };

    const handleBack = () => {
        router.get('/install/database');
    };

    return (
        <>
            <Head title="Create Admin - Installation" />
            
            <div className="min-h-screen bg-gradient-to-br from-emerald-50 via-teal-50 to-cyan-50 dark:from-slate-900 dark:via-emerald-950 dark:to-teal-950 flex items-center justify-center p-6">
                {/* Background Orbs */}
                <div className="absolute top-0 left-0 w-96 h-96 bg-emerald-200/40 dark:bg-emerald-500/20 rounded-full blur-3xl animate-pulse" />
                <div className="absolute bottom-0 right-0 w-96 h-96 bg-teal-200/40 dark:bg-teal-500/20 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '2s' }} />
                
                {/* Theme Toggle */}
                <div className="fixed top-6 right-6 z-20">
                    <div className="inline-flex items-center rounded-full bg-white/80 dark:bg-slate-800/80 backdrop-blur-sm p-1 shadow-lg border border-border/50">
                        <button
                            onClick={() => updateAppearance('light')}
                            className={`inline-flex items-center gap-1.5 rounded-full px-3 py-2 text-sm font-medium transition-all ${
                                appearance === 'light'
                                    ? 'bg-white dark:bg-slate-700 text-foreground shadow-sm'
                                    : 'text-muted-foreground hover:text-foreground'
                            }`}
                        >
                            <Sun className="h-4 w-4" />
                        </button>
                        <button
                            onClick={() => updateAppearance('dark')}
                            className={`inline-flex items-center gap-1.5 rounded-full px-3 py-2 text-sm font-medium transition-all ${
                                appearance === 'dark'
                                    ? 'bg-white dark:bg-slate-700 text-foreground shadow-sm'
                                    : 'text-muted-foreground hover:text-foreground'
                            }`}
                        >
                            <Moon className="h-4 w-4" />
                        </button>
                    </div>
                </div>
                
                <div className="relative z-10 w-full max-w-2xl">
                    {/* Progress Bar */}
                    <div className="mb-6">
                        <div className="flex items-center justify-between text-sm text-muted-foreground mb-2">
                            <span>Step 3 of 4</span>
                            <span>75%</span>
                        </div>
                        <div className="h-2 bg-muted rounded-full overflow-hidden">
                            <div className="h-full w-3/4 bg-gradient-to-r from-emerald-500 to-teal-600 rounded-full transition-all duration-500" />
                        </div>
                    </div>
                    
                    {/* Main Card */}
                    <div className="rounded-3xl border border-border/50 bg-white/80 dark:bg-slate-900/80 backdrop-blur-xl p-8 shadow-2xl">
                        {/* Heading */}
                        <div className="flex items-center gap-3 mb-6">
                            <div className="w-12 h-12 rounded-xl bg-gradient-to-br from-emerald-500 to-teal-600 flex items-center justify-center">
                                <Shield className="w-6 h-6 text-white" />
                            </div>
                            <div>
                                <h1 className="text-2xl font-bold text-foreground">Create Super Admin</h1>
                                <p className="text-muted-foreground">Set up your administrator account</p>
                            </div>
                        </div>
                        
                        {errors?.admin && (
                            <InputError message={errors.admin} className="mb-4" />
                        )}
                        
                        <form onSubmit={handleSubmit} className="space-y-5">
                            {/* Name Field */}
                            <div className="space-y-2">
                                <Label htmlFor="name" className="text-sm font-medium">
                                    Full Name
                                </Label>
                                <div className="relative">
                                    <User className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-muted-foreground" />
                                    <Input
                                        id="name"
                                        value={data.name}
                                        onChange={(e) => setData('name', e.target.value)}
                                        placeholder="Super Admin"
                                        className="pl-10 h-11"
                                        required
                                    />
                                </div>
                                <InputError message={errors?.name} />
                            </div>
                            
                            {/* Email Field */}
                            <div className="space-y-2">
                                <Label htmlFor="email" className="text-sm font-medium">
                                    Email Address
                                </Label>
                                <div className="relative">
                                    <Mail className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-muted-foreground" />
                                    <Input
                                        id="email"
                                        type="email"
                                        value={data.email}
                                        onChange={(e) => setData('email', e.target.value)}
                                        placeholder="admin@yourcompany.com"
                                        className="pl-10 h-11"
                                        required
                                    />
                                </div>
                                <InputError message={errors?.email} />
                            </div>
                            
                            {/* Password Field */}
                            <div className="space-y-2">
                                <Label htmlFor="password" className="text-sm font-medium">
                                    Password
                                </Label>
                                <div className="relative">
                                    <Lock className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-muted-foreground" />
                                    <Input
                                        id="password"
                                        type="password"
                                        value={data.password}
                                        onChange={(e) => setData('password', e.target.value)}
                                        placeholder="••••••••"
                                        className="pl-10 h-11"
                                        required
                                        minLength={8}
                                    />
                                </div>
                                <p className="text-xs text-muted-foreground">Minimum 8 characters</p>
                                <InputError message={errors?.password} />
                            </div>
                            
                            {/* Confirm Password Field */}
                            <div className="space-y-2">
                                <Label htmlFor="password_confirmation" className="text-sm font-medium">
                                    Confirm Password
                                </Label>
                                <div className="relative">
                                    <Lock className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-muted-foreground" />
                                    <Input
                                        id="password_confirmation"
                                        type="password"
                                        value={data.password_confirmation}
                                        onChange={(e) => setData('password_confirmation', e.target.value)}
                                        placeholder="••••••••"
                                        className="pl-10 h-11"
                                        required
                                    />
                                </div>
                                <InputError message={errors?.password_confirmation} />
                            </div>
                            
                            {/* Info */}
                            <div className="p-4 rounded-xl bg-blue-50 dark:bg-blue-950/30 border border-blue-200 dark:border-blue-800">
                                <p className="text-sm text-blue-800 dark:text-blue-200">
                                    💡 This account will have full administrative access to the platform.
                                </p>
                            </div>
                            
                            {/* Actions */}
                            <div className="flex items-center justify-between pt-2">
                                <Button
                                    type="button"
                                    variant="outline"
                                    onClick={handleBack}
                                    disabled={processing}
                                >
                                    <ChevronLeft className="mr-1 h-4 w-4" />
                                    Back
                                </Button>
                                <Button
                                    type="submit"
                                    disabled={processing}
                                    className="bg-gradient-to-r from-emerald-600 to-teal-600 hover:from-emerald-700 hover:to-teal-700 text-white shadow-lg"
                                >
                                    {processing && <Spinner className="mr-2" />}
                                    {processing ? 'Creating...' : 'Create Admin'}
                                    {!processing && <ChevronRight className="ml-1 h-4 w-4" />}
                                </Button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </>
    );
}
