import { Head, Link } from '@inertiajs/react';
import { Button } from '@/components/ui/button';
import { FileQuestion, AlertTriangle, ShieldAlert, ServerCrash, Clock, Ban, Home } from 'lucide-react';

interface Props {
    status: number;
}

export default function ErrorPage({ status }: Props) {
    const title = {
        503: '503: Service Unavailable',
        500: '500: Server Error',
        404: '404: Page Not Found',
        403: '403: Forbidden',
        401: '401: Unauthorized',
        429: '429: Too Many Requests',
        419: '419: Page Expired',
    }[status];

    const description = {
        503: 'Sorry, we are doing some maintenance. Please check back soon.',
        500: 'Whoops, something went wrong on our servers.',
        404: 'Sorry, the page you are looking for could not be found.',
        403: 'Sorry, you are forbidden from accessing this page.',
        401: 'Sorry, you are unauthorized to access this page.',
        429: 'Too many requests. Please try again later.',
        419: 'The page has expired due to inactivity. Please refresh and try again.',
    }[status];

    const Icon = {
        503: Clock,
        500: ServerCrash,
        404: FileQuestion,
        403: ShieldAlert,
        401: ShieldAlert,
        429: Ban,
        419: Clock,
    }[status] || AlertTriangle;

    return (
        <>
            <Head title={title} />
            <div className="min-h-screen w-full flex items-center justify-center bg-background relative overflow-hidden">
                {/* Background Decor */}
                <div className="absolute top-0 left-0 w-full h-full overflow-hidden pointer-events-none">
                    <div className="absolute top-[-10%] left-[-10%] w-[40%] h-[40%] bg-cyan-500/10 rounded-full blur-[100px] animate-pulse" />
                    <div className="absolute bottom-[-10%] right-[-10%] w-[40%] h-[40%] bg-teal-500/10 rounded-full blur-[100px] animate-pulse" />
                </div>

                <div className="relative z-10 max-w-lg w-full px-6 text-center">
                    <div className="mb-8 flex justify-center">
                        <div className="relative">
                            <div className="absolute inset-0 bg-gradient-to-br from-sky-500 to-teal-500 blur-xl opacity-30 rounded-full" />
                            <div className="relative w-24 h-24 bg-card border border-border/50 rounded-3xl flex items-center justify-center shadow-2xl">
                                <Icon className="w-12 h-12 text-cyan-500" />
                            </div>
                        </div>
                    </div>

                    <h1 className="text-8xl font-black mb-2 tracking-tighter bg-gradient-to-br from-sky-600 via-teal-600 to-teal-600 bg-clip-text text-transparent">
                        {status}
                    </h1>
                    
                    <h2 className="text-2xl font-bold mb-4 text-foreground">
                        {title?.split(': ')[1] || 'Error'}
                    </h2>

                    <p className="text-muted-foreground text-lg mb-8 leading-relaxed">
                        {description}
                    </p>

                    <div className="flex flex-col sm:flex-row gap-4 justify-center">
                        <Button asChild size="lg" className="bg-gradient-to-r from-sky-600 to-teal-600 hover:from-sky-700 hover:to-teal-700 shadow-lg shadow-cyan-500/20">
                            <Link href="/">
                                <Home className="w-4 h-4 mr-2" />
                                Back to Home
                            </Link>
                        </Button>
                        
                        <Button asChild variant="outline" size="lg" className="border-cyan-200 hover:bg-cyan-50 dark:border-cyan-800 dark:hover:bg-cyan-900/20">
                            <Link href="#" onClick={() => window.history.back()}>
                                Go Back
                            </Link>
                        </Button>
                    </div>
                </div>
            </div>
        </>
    );
}
