import SocietyAdminLayout from '@/layouts/society-admin-layout';
import { type BreadcrumbItem } from '@/types';
import { Head, usePage } from '@inertiajs/react';
import {
    DollarSign,
    Receipt,
    TrendingUp,
    Banknote,
    AlertCircle,
    Building2,
    Clock,
    ArrowDownRight,
    ArrowUpRight,
    Activity,
} from 'lucide-react';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import {
    BarChart,
    Bar,
    XAxis,
    YAxis,
    CartesianGrid,
    Tooltip,
    ResponsiveContainer,
    Legend,
} from 'recharts';

const breadcrumbs: BreadcrumbItem[] = [
    {
        title: 'Dashboard',
        href: '/societies/{society}/accountant/dashboard',
    },
];

interface DashboardProps {
    society: {
        id: number;
        name: string;
        code?: string;
    };
    kpis: {
        total_collections_mtd: number;
        outstanding_dues: number;
        month_expenses: number;
        collection_rate: number;
        bank_balance: number;
        pending_reconciliation: number;
    };
    recentActivity: {
        payments: Array<{
            id: number;
            payment_no: string;
            amount: number;
            user_name: string;
            bill_no: string | null;
            is_reconciled: boolean;
            created_at: string;
        }>;
        bills: Array<{
            id: number;
            bill_no: string;
            amount: number;
            status: string;
            user_name: string;
            unit_no: string;
            created_at: string;
        }>;
        expenses: Array<{
            id: number;
            expense_no: string;
            amount: number;
            category: string | null;
            created_by_name: string;
            created_at: string;
        }>;
    };
    charts: {
        monthly_collections: Array<{
            month: string;
            total: number;
        }>;
        monthly_expenses: Array<{
            month: string;
            total: number;
        }>;
    };
}

export default function Dashboard({
    society,
    kpis,
    recentActivity,
    charts,
}: DashboardProps) {
    const { auth } = usePage().props as any;
    const user = auth.user;

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            maximumFractionDigits: 0,
        }).format(amount);
    };

    // Prepare data for the combined chart
    const combinedChartData = charts.monthly_collections.map((item, index) => ({
        month: item.month,
        collections: item.total,
        expenses: charts.monthly_expenses[index]?.total || 0,
    }));

    return (
        <SocietyAdminLayout breadcrumbs={breadcrumbs} society={society}>
            <Head title="Accountant Dashboard" />
            <div className="flex flex-1 flex-col gap-8 p-6 md:p-8 overflow-x-hidden bg-muted/10">
                
                {/* Welcome & Society Info Gradient Banner */}
                <div className="relative overflow-hidden rounded-2xl bg-gradient-to-br from-emerald-600 via-teal-600 to-cyan-600 p-8 text-white shadow-lg">
                    <div className="absolute top-0 right-0 -mt-10 -mr-10 h-64 w-64 rounded-full bg-white/10 blur-3xl" />
                    <div className="absolute bottom-0 left-0 -mb-10 -ml-10 h-64 w-64 rounded-full bg-white/10 blur-3xl" />
                    
                    <div className="relative z-10 flex flex-col md:flex-row justify-between items-start md:items-center gap-6">
                        <div>
                            <h1 className="text-3xl font-bold tracking-tight text-white">
                                Financial Dashboard
                            </h1>
                            <p className="mt-2 text-emerald-100 text-lg">
                                Overview of financial operations and metrics
                            </p>
                        </div>
                        <div className="flex items-center gap-3 bg-white/10 backdrop-blur-md p-3 rounded-xl border border-white/20 shadow-inner">
                            <div className="h-12 w-12 rounded-lg bg-white/20 flex items-center justify-center text-white shadow-sm">
                                <Building2 className="h-6 w-6" />
                            </div>
                            <div className="pr-2">
                                <p className="text-base font-bold text-white">{society.name}</p>
                                <p className="text-xs text-emerald-100 font-medium tracking-wide">Welcome, {user.name}</p>
                            </div>
                        </div>
                    </div>
                </div>

                {/* Financial KPIs */}
                <div>
                    <h3 className="text-lg font-bold mb-4 flex items-center gap-2 text-foreground/80">
                        <div className="p-1.5 rounded-md bg-emerald-100 text-emerald-600 dark:bg-emerald-900/30 dark:text-emerald-400">
                            <DollarSign className="h-5 w-5" />
                        </div>
                        Financial Overview
                    </h3>
                    <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
                        <Card className="border-t-4 border-t-emerald-500 shadow-sm hover:shadow-lg transition-all duration-300">
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium text-muted-foreground">Total Collections (MTD)</CardTitle>
                                <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-emerald-400 to-green-600 flex items-center justify-center text-white shadow-md shadow-emerald-500/20">
                                    <DollarSign className="h-5 w-5" />
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="text-2xl font-bold text-foreground">{formatCurrency(kpis.total_collections_mtd)}</div>
                                <p className="text-xs text-emerald-600 font-medium mt-1 flex items-center gap-1">
                                    <ArrowDownRight className="h-3 w-3" />
                                    Month to date revenue
                                </p>
                            </CardContent>
                        </Card>

                        <Card className="border-t-4 border-t-rose-500 shadow-sm hover:shadow-lg transition-all duration-300">
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium text-muted-foreground">Outstanding Dues</CardTitle>
                                <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-rose-400 to-red-600 flex items-center justify-center text-white shadow-md shadow-rose-500/20">
                                    <AlertCircle className="h-5 w-5" />
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="text-2xl font-bold text-rose-600 dark:text-rose-400">
                                    {formatCurrency(kpis.outstanding_dues)}
                                </div>
                                <p className="text-xs text-muted-foreground mt-1">
                                    Pending from residents
                                </p>
                            </CardContent>
                        </Card>

                        <Card className="border-t-4 border-t-orange-500 shadow-sm hover:shadow-lg transition-all duration-300">
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium text-muted-foreground">Monthly Expenses</CardTitle>
                                <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-orange-400 to-amber-600 flex items-center justify-center text-white shadow-md shadow-orange-500/20">
                                    <Receipt className="h-5 w-5" />
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="text-2xl font-bold text-foreground">{formatCurrency(kpis.month_expenses)}</div>
                                <p className="text-xs text-muted-foreground mt-1">
                                    For {new Date().toLocaleString('default', { month: 'long' })}
                                </p>
                            </CardContent>
                        </Card>

                        <Card className="border-t-4 border-t-blue-500 shadow-sm hover:shadow-lg transition-all duration-300">
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium text-muted-foreground">Collection Rate</CardTitle>
                                <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-blue-400 to-indigo-600 flex items-center justify-center text-white shadow-md shadow-blue-500/20">
                                    <Activity className="h-5 w-5" />
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="text-2xl font-bold text-foreground">{kpis.collection_rate}%</div>
                                <div className="w-full bg-secondary h-2 rounded-full mt-2 overflow-hidden">
                                    <div 
                                        className="bg-gradient-to-r from-blue-500 to-indigo-600 h-full rounded-full transition-all duration-500 shadow-[0_0_10px_rgba(59,130,246,0.5)]" 
                                        style={{ width: `${kpis.collection_rate}%` }}
                                    />
                                </div>
                            </CardContent>
                        </Card>

                        <Card className="border-t-4 border-t-purple-500 shadow-sm hover:shadow-lg transition-all duration-300">
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium text-muted-foreground">Bank Balance</CardTitle>
                                <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-cyan-400 to-cyan-600 flex items-center justify-center text-white shadow-md shadow-cyan-500/20">
                                    <Banknote className="h-5 w-5" />
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="text-2xl font-bold text-foreground">{formatCurrency(kpis.bank_balance)}</div>
                                <p className="text-xs text-muted-foreground mt-1">
                                    Total across all accounts
                                </p>
                            </CardContent>
                        </Card>

                        <Card className="border-t-4 border-t-amber-500 shadow-sm hover:shadow-lg transition-all duration-300">
                            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                                <CardTitle className="text-sm font-medium text-muted-foreground">Pending Reconciliation</CardTitle>
                                <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-amber-400 to-yellow-600 flex items-center justify-center text-white shadow-md shadow-amber-500/20">
                                    <TrendingUp className="h-5 w-5" />
                                </div>
                            </CardHeader>
                            <CardContent>
                                <div className="text-2xl font-bold text-amber-600 dark:text-amber-400">{kpis.pending_reconciliation}</div>
                                <p className="text-xs text-muted-foreground mt-1">
                                    Payments to reconcile
                                </p>
                            </CardContent>
                        </Card>
                    </div>
                </div>

                {/* Charts Section */}
                <Card className="shadow-sm border-none ring-1 ring-border/50">
                    <CardHeader>
                        <CardTitle className="text-lg font-bold text-foreground/80">Financial Performance</CardTitle>
                        <CardDescription>Collections vs Expenses over the last 6 months</CardDescription>
                    </CardHeader>
                    <CardContent className="pl-0">
                        <div className="h-[350px] w-full">
                            <ResponsiveContainer width="100%" height="100%">
                                <BarChart data={combinedChartData} margin={{ top: 20, right: 30, left: 0, bottom: 0 }}>
                                    <CartesianGrid strokeDasharray="3 3" vertical={false} className="stroke-muted/50" />
                                    <XAxis 
                                        dataKey="month" 
                                        axisLine={false} 
                                        tickLine={false} 
                                        tick={{ fill: 'hsl(var(--muted-foreground))', fontSize: 12, fontWeight: 500 }} 
                                        dy={10}
                                    />
                                    <YAxis 
                                        axisLine={false} 
                                        tickLine={false} 
                                        tick={{ fill: 'hsl(var(--muted-foreground))', fontSize: 12, fontWeight: 500 }}
                                        tickFormatter={(value) => `₹${value / 1000}k`}
                                    />
                                    <Tooltip 
                                        cursor={{ fill: 'hsl(var(--muted))', opacity: 0.2 }}
                                        contentStyle={{ 
                                            backgroundColor: 'hsl(var(--popover))', 
                                            borderColor: 'hsl(var(--border))',
                                            borderRadius: '12px',
                                            boxShadow: '0 10px 30px -10px rgba(0,0,0,0.2)',
                                            padding: '12px'
                                        }}
                                        formatter={(value: number) => [formatCurrency(value), '']}
                                    />
                                    <Legend wrapperStyle={{ paddingTop: '20px' }} iconType="circle" />
                                    <Bar 
                                        dataKey="collections" 
                                        name="Collections" 
                                        fill="url(#colorCollections)" 
                                        radius={[6, 6, 0, 0]} 
                                        barSize={20}
                                    >
                                        <defs>
                                            <linearGradient id="colorCollections" x1="0" y1="0" x2="0" y2="1">
                                                <stop offset="0%" stopColor="#10b981" stopOpacity={1}/>
                                                <stop offset="100%" stopColor="#059669" stopOpacity={0.8}/>
                                            </linearGradient>
                                        </defs>
                                    </Bar>
                                    <Bar 
                                        dataKey="expenses" 
                                        name="Expenses" 
                                        fill="url(#colorExpenses)" 
                                        radius={[6, 6, 0, 0]} 
                                        barSize={20}
                                    >
                                        <defs>
                                            <linearGradient id="colorExpenses" x1="0" y1="0" x2="0" y2="1">
                                                <stop offset="0%" stopColor="#f43f5e" stopOpacity={1}/>
                                                <stop offset="100%" stopColor="#e11d48" stopOpacity={0.8}/>
                                            </linearGradient>
                                        </defs>
                                    </Bar>
                                </BarChart>
                            </ResponsiveContainer>
                        </div>
                    </CardContent>
                </Card>

                {/* Recent Activity Tabs */}
                <Card className="shadow-md border-none ring-1 ring-border/50">
                    <CardHeader className="bg-muted/30 border-b border-border/50">
                        <div className="flex items-center justify-between">
                            <div>
                                <CardTitle className="text-lg font-bold">Recent Activity</CardTitle>
                                <CardDescription>Latest financial transactions</CardDescription>
                            </div>
                            <div className="p-2 bg-background rounded-full shadow-sm">
                                <Clock className="h-5 w-5 text-muted-foreground" />
                            </div>
                        </div>
                    </CardHeader>
                    <CardContent className="p-6">
                        <Tabs defaultValue="payments" className="w-full">
                            <TabsList className="grid w-full grid-cols-3 lg:w-[400px] mb-6 p-1 bg-muted/50 rounded-xl">
                                <TabsTrigger value="payments" className="rounded-lg data-[state=active]:bg-white data-[state=active]:text-green-600 data-[state=active]:shadow-sm font-medium">Payments</TabsTrigger>
                                <TabsTrigger value="bills" className="rounded-lg data-[state=active]:bg-white data-[state=active]:text-blue-600 data-[state=active]:shadow-sm font-medium">Bills</TabsTrigger>
                                <TabsTrigger value="expenses" className="rounded-lg data-[state=active]:bg-white data-[state=active]:text-orange-600 data-[state=active]:shadow-sm font-medium">Expenses</TabsTrigger>
                            </TabsList>
                            
                            <TabsContent value="payments" className="mt-0">
                                <div className="space-y-4">
                                    {recentActivity.payments.length > 0 ? (
                                        recentActivity.payments.map((payment) => (
                                            <div key={payment.id} className="group flex items-center justify-between p-3 rounded-xl hover:bg-muted/50 transition-colors border border-transparent hover:border-border/50">
                                                <div className="flex items-center gap-4">
                                                    <div className="h-10 w-10 rounded-full bg-green-100 flex items-center justify-center text-green-600 group-hover:scale-110 transition-transform">
                                                        <ArrowDownRight className="h-5 w-5" />
                                                    </div>
                                                    <div>
                                                        <p className="text-sm font-bold text-foreground">{payment.user_name}</p>
                                                        <p className="text-xs text-muted-foreground font-medium">{payment.payment_no} • {new Date(payment.created_at).toLocaleDateString()}</p>
                                                    </div>
                                                </div>
                                                <div className="text-right flex flex-col items-end gap-1">
                                                    <div className="font-bold text-green-600 bg-green-50 px-3 py-1 rounded-lg border border-green-100">
                                                        +{formatCurrency(payment.amount)}
                                                    </div>
                                                    {!payment.is_reconciled && (
                                                        <span className="text-[10px] px-2 py-0.5 rounded-full font-bold uppercase tracking-wide bg-yellow-100 text-yellow-800">
                                                            Pending
                                                        </span>
                                                    )}
                                                </div>
                                            </div>
                                        ))
                                    ) : (
                                        <div className="text-center py-12 text-muted-foreground bg-muted/20 rounded-xl border border-dashed">No recent payments</div>
                                    )}
                                </div>
                            </TabsContent>

                            <TabsContent value="bills" className="mt-0">
                                <div className="space-y-4">
                                    {recentActivity.bills.length > 0 ? (
                                        recentActivity.bills.map((bill) => (
                                            <div key={bill.id} className="group flex items-center justify-between p-3 rounded-xl hover:bg-muted/50 transition-colors border border-transparent hover:border-border/50">
                                                <div className="flex items-center gap-4">
                                                    <div className="h-10 w-10 rounded-full bg-blue-100 flex items-center justify-center text-blue-600 group-hover:scale-110 transition-transform">
                                                        <Receipt className="h-5 w-5" />
                                                    </div>
                                                    <div>
                                                        <p className="text-sm font-bold text-foreground">{bill.user_name} <span className="text-muted-foreground font-normal">({bill.unit_no})</span></p>
                                                        <p className="text-xs text-muted-foreground font-medium">{bill.bill_no} • {new Date(bill.created_at).toLocaleDateString()}</p>
                                                    </div>
                                                </div>
                                                <div className="text-right flex flex-col items-end gap-1">
                                                    <div className="font-bold text-foreground">{formatCurrency(bill.amount)}</div>
                                                    <span className={`text-[10px] px-2 py-0.5 rounded-full font-bold uppercase tracking-wide ${
                                                        bill.status === 'paid' ? 'bg-green-100 text-green-700' :
                                                        bill.status === 'overdue' ? 'bg-red-100 text-red-700' :
                                                        'bg-yellow-100 text-yellow-700'
                                                    }`}>
                                                        {bill.status}
                                                    </span>
                                                </div>
                                            </div>
                                        ))
                                    ) : (
                                        <div className="text-center py-12 text-muted-foreground bg-muted/20 rounded-xl border border-dashed">No recent bills</div>
                                    )}
                                </div>
                            </TabsContent>

                            <TabsContent value="expenses" className="mt-0">
                                <div className="space-y-4">
                                    {recentActivity.expenses.length > 0 ? (
                                        recentActivity.expenses.map((expense) => (
                                            <div key={expense.id} className="group flex items-center justify-between p-3 rounded-xl hover:bg-muted/50 transition-colors border border-transparent hover:border-border/50">
                                                <div className="flex items-center gap-4">
                                                    <div className="h-10 w-10 rounded-full bg-orange-100 flex items-center justify-center text-orange-600 group-hover:scale-110 transition-transform">
                                                        <ArrowUpRight className="h-5 w-5" />
                                                    </div>
                                                    <div>
                                                        <p className="text-sm font-bold text-foreground">{expense.category || 'Uncategorized'}</p>
                                                        <p className="text-xs text-muted-foreground font-medium">By {expense.created_by_name} • {new Date(expense.created_at).toLocaleDateString()}</p>
                                                    </div>
                                                </div>
                                                <div className="font-bold text-red-600 bg-red-50 px-3 py-1 rounded-lg border border-red-100">
                                                    -{formatCurrency(expense.amount)}
                                                </div>
                                            </div>
                                        ))
                                    ) : (
                                        <div className="text-center py-12 text-muted-foreground bg-muted/20 rounded-xl border border-dashed">No recent expenses</div>
                                    )}
                                </div>
                            </TabsContent>
                        </Tabs>
                    </CardContent>
                </Card>
            </div>
        </SocietyAdminLayout>
    );
}
