import { AppContent } from '@/components/app-content';
import { AppShell } from '@/components/app-shell';
import { AppSidebarHeader } from '@/components/app-sidebar-header';
import {
    Sidebar,
    SidebarContent,
    SidebarFooter,
    SidebarHeader,
    SidebarMenu,
    SidebarMenuButton,
    SidebarMenuItem,
    SidebarSeparator,
    SidebarGroup,
    SidebarGroupLabel,
    SidebarGroupContent,
} from '@/components/ui/sidebar';
import { NavMain } from '@/components/nav-main';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { Badge } from '@/components/ui/badge';
import FlashMessagesProvider from '@/components/flash-messages-provider';
import { type BreadcrumbItem, type NavItem } from '@/types';
import { Link, router, usePage } from '@inertiajs/react';
import { 
    Building2, 
    LayoutGrid, 
    Users, 
    CreditCard,
    Crown,
    Settings,
    LogOut,
    ChevronRight
} from 'lucide-react';
import { type PropsWithChildren } from 'react';
import AppLogo from '@/components/app-logo';
import { getSuperAdminMenu } from '@/config/menus';



interface SuperAdminLayoutProps {
    breadcrumbs?: BreadcrumbItem[];
}

export default function SuperAdminLayout({
    children,
    breadcrumbs = [],
}: PropsWithChildren<SuperAdminLayoutProps>) {
    const { auth } = usePage().props as any;
    const user = auth?.user;

    const handleLogout = () => {
        router.post('/super-admin/logout');
    };

    const getInitials = (name: string) => {
        return name
            .split(' ')
            .map(n => n[0])
            .join('')
            .toUpperCase()
            .slice(0, 2);
    };

    return (
        <FlashMessagesProvider>
            {/* Global Gradient Background */}
            <div className="fixed inset-0 -z-10 bg-societify-gradient-animated opacity-10 dark:opacity-5 pointer-events-none" />
            
            <AppShell variant="sidebar">
                <Sidebar collapsible="icon" variant="inset" className="bg-white/60 dark:bg-black/60 backdrop-blur-xl border-r border-white/20 dark:border-white/10 shadow-xl">
                    {/* Enhanced Header with Branding */}
                    <SidebarHeader className="border-b border-sidebar-border/50">
                        <SidebarMenu>
                            <SidebarMenuItem>
                                <SidebarMenuButton size="lg" asChild className="group">
                                    <Link href="/super-admin/dashboard" prefetch>
                                        <div className="flex items-center gap-3 w-full">
                                            {/* Gradient Logo Container */}
                                            <div className="flex h-10 w-10 shrink-0 items-center justify-center rounded-xl bg-societify-gradient shadow-lg shadow-primary/20 group-hover:shadow-primary/40 transition-shadow">
                                                <Crown className="h-5 w-5 text-white" />
                                            </div>
                                            {/* Brand Text */}
                                            <div className="flex flex-col items-start group-data-[collapsible=icon]:hidden">
                                                <span className="text-sm font-bold text-societify-gradient">
                                                    Super Admin
                                                </span>
                                                <span className="text-xs text-muted-foreground">
                                                    Control Panel
                                                </span>
                                            </div>
                                        </div>
                                    </Link>
                                </SidebarMenuButton>
                            </SidebarMenuItem>
                        </SidebarMenu>
                    </SidebarHeader>

                    {/* Menu Content with Sections */}
                    <SidebarContent className="px-2 py-4 overflow-y-auto scrollbar-hide">
                        {getSuperAdminMenu().map((group, index) => (
                            <div key={group.label}>
                                <SidebarGroup>
                                    <SidebarGroupLabel className="text-xs font-semibold text-muted-foreground/80 px-2 mb-2">
                                        {group.label}
                                    </SidebarGroupLabel>
                                    <SidebarGroupContent>
                                        <NavMain items={group.items} />
                                    </SidebarGroupContent>
                                </SidebarGroup>
                                {index < getSuperAdminMenu().length - 1 && <SidebarSeparator className="my-2" />}
                            </div>
                        ))}
                    </SidebarContent>

                    {/* Floating Profile Card Footer */}
                    <SidebarFooter className="p-3">
                        <div className="relative group">
                            {/* Gradient Glow Effect */}
                            <div className="absolute -inset-0.5 bg-gradient-to-r from-sky-500 via-cyan-500 to-teal-500 rounded-xl blur opacity-20 group-hover:opacity-40 transition duration-300" />
                            
                            {/* Profile Card */}
                            <div className="relative rounded-xl border border-border/50 bg-gradient-to-br from-background to-muted/30 backdrop-blur-sm p-3 shadow-lg">
                                <div className="flex flex-col gap-3 group-data-[collapsible=icon]:hidden">
                                    {/* Avatar + User Info Row */}
                                    <div className="flex items-center gap-3">
                                        <div className="relative shrink-0">
                                            <div className="absolute -inset-1 bg-gradient-to-r from-sky-500 to-teal-500 rounded-full blur opacity-50" />
                                            <Avatar className="h-10 w-10 relative border-2 border-background">
                                                <AvatarImage src={user?.avatar} alt={user?.name} />
                                                <AvatarFallback className="bg-gradient-to-br from-sky-500 to-teal-500 text-white font-bold text-sm">
                                                    {user?.name ? getInitials(user.name) : 'SA'}
                                                </AvatarFallback>
                                            </Avatar>
                                        </div>
                                        
                                        {/* Name, Badge, Email Stack */}
                                        <div className="flex-1 min-w-0 space-y-0.5">
                                            <p className="text-sm font-semibold truncate leading-tight">
                                                {user?.name || 'Super Admin'}
                                            </p>
                                            <Badge 
                                                variant="outline" 
                                                className="w-fit h-4 px-1.5 text-[9px] bg-gradient-to-r from-cyan-500/10 to-teal-500/10 border-cyan-500/30 text-cyan-700 dark:text-cyan-400 font-semibold"
                                            >
                                                ADMIN
                                            </Badge>
                                            <p className="text-[10px] text-muted-foreground truncate leading-tight">
                                                {user?.email || 'admin@societify.com'}
                                            </p>
                                        </div>
                                    </div>

                                    {/* Buttons Row */}
                                    <div className="flex items-center gap-2">
                                        <button
                                            className="flex-1 flex items-center justify-center gap-1.5 px-2.5 py-1.5 rounded-lg bg-muted hover:bg-muted/80 text-xs font-medium transition-colors"
                                            title="Profile"
                                        >
                                            <Settings className="h-3 w-3" />
                                            Profile
                                        </button>
                                        <button
                                            onClick={handleLogout}
                                            className="flex-1 flex items-center justify-center gap-1.5 px-2.5 py-1.5 rounded-lg bg-red-500/10 hover:bg-red-500/20 text-red-600 dark:text-red-400 text-xs font-medium transition-colors"
                                            title="Logout"
                                        >
                                            <LogOut className="h-3 w-3" />
                                            Logout
                                        </button>
                                    </div>
                                </div>

                                {/* Collapsed Icon State */}
                                <div className="hidden group-data-[collapsible=icon]:flex flex-col items-center gap-2">
                                    <div className="relative">
                                        <div className="absolute -inset-1 bg-gradient-to-r from-sky-500 to-teal-500 rounded-full blur opacity-50" />
                                        <Avatar className="h-10 w-10 relative border-2 border-background">
                                            <AvatarImage src={user?.avatar} alt={user?.name} />
                                            <AvatarFallback className="bg-gradient-to-br from-sky-500 to-teal-500 text-white font-bold">
                                                {user?.name ? getInitials(user.name) : 'SA'}
                                            </AvatarFallback>
                                        </Avatar>
                                    </div>
                                    <button
                                        onClick={handleLogout}
                                        className="p-2 rounded-lg hover:bg-red-500/10 text-muted-foreground hover:text-red-600 dark:hover:text-red-400 transition-colors"
                                        title="Logout"
                                    >
                                        <LogOut className="h-4 w-4" />
                                    </button>
                                </div>
                            </div>
                        </div>
                    </SidebarFooter>
                </Sidebar>

                <AppContent variant="sidebar" className="overflow-x-hidden">
                    <AppSidebarHeader breadcrumbs={breadcrumbs} />
                    {children}
                </AppContent>
            </AppShell>
        </FlashMessagesProvider>
    );
}
