import { AppContent } from '@/components/app-content';
import { AppShell } from '@/components/app-shell';
import { AppSidebarHeader } from '@/components/app-sidebar-header';
import {
    Sidebar,
    SidebarContent,
    SidebarFooter,
    SidebarHeader,
    SidebarMenu,
    SidebarMenuButton,
    SidebarMenuItem,
    SidebarGroup,
    SidebarGroupLabel,
    SidebarGroupContent,
    SidebarSeparator,
} from '@/components/ui/sidebar';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { Badge } from '@/components/ui/badge';
import { NavMain } from '@/components/nav-main';
import { NavUser } from '@/components/nav-user';
import FlashMessagesProvider from '@/components/flash-messages-provider';
import { type BreadcrumbItem, type NavItem } from '@/types';
import { Link, router, usePage } from '@inertiajs/react';
import {
    LayoutGrid,
    Building2,
    LogOut,
    Settings,
    Crown,
} from 'lucide-react';
import { type PropsWithChildren } from 'react';
import AppLogo from '@/components/app-logo';
import { getMenuForRole } from '@/config/menus';
import { EmergencyAlertOverlay } from '@/components/emergency-alert-overlay';



interface SocietyAdminLayoutProps {
    breadcrumbs?: BreadcrumbItem[];
    society: {
        id: number;
        name: string;
        code?: string;
    };
}

export default function SocietyAdminLayout({
    children,
    breadcrumbs = [],
    society,
}: PropsWithChildren<SocietyAdminLayoutProps>) {
    const { auth } = usePage().props as any;
    const user = auth.user;
    const permissions = auth.permissions || [];

    const handleLogout = () => {
        router.post('/logout');
    };

    const getInitials = (name: string) => {
        return name
            .split(' ')
            .map(n => n[0])
            .join('')
            .toUpperCase()
            .slice(0, 2);
    };

    // Generate menu items based on role and permissions
    const navItems = getMenuForRole(
        user?.role?.name || 'society_admin', 
        society.id, 
        permissions
    );

    return (
        <FlashMessagesProvider>
            {/* Global Gradient Background */}
            <div className="fixed inset-0 -z-10 bg-societify-gradient-animated opacity-10 dark:opacity-5 pointer-events-none" />

            <AppShell variant="sidebar">
                <Sidebar collapsible="icon" variant="inset" className="bg-white/80 dark:bg-gray-950/80 backdrop-blur-xl border-r border-gray-200/50 dark:border-white/10 shadow-xl">
                    <SidebarHeader className="border-b border-sidebar-border/50">
                        <SidebarMenu>
                            <SidebarMenuItem>
                                <SidebarMenuButton size="lg" asChild className="group h-auto py-3">
                                    <Link
                                        href={`/societies/${society.id}/dashboard`}
                                        prefetch
                                    >
                                        <div className="flex items-center gap-3 w-full">
                                            {/* Gradient Logo Container */}
                                            <div className="flex h-10 w-10 shrink-0 items-center justify-center rounded-xl bg-gradient-to-br from-sky-600 via-cyan-600 to-teal-500 shadow-lg shadow-cyan-500/30 group-hover:shadow-cyan-500/50 transition-shadow">
                                                <Building2 className="h-5 w-5 text-white" />
                                            </div>
                                            {/* Brand Text */}
                                            <div className="flex flex-col items-start group-data-[collapsible=icon]:hidden">
                                                <span className="text-sm font-bold bg-gradient-to-r from-sky-600 via-cyan-600 to-teal-500 dark:from-sky-400 dark:via-cyan-400 dark:to-teal-400 bg-clip-text text-transparent whitespace-normal leading-tight text-left">
                                                    {society.name}
                                                </span>
                                                <span className="text-xs text-muted-foreground">
                                                    {society.code || 'Society Admin'}
                                                </span>
                                            </div>
                                        </div>
                                    </Link>
                                </SidebarMenuButton>
                            </SidebarMenuItem>
                        </SidebarMenu>
                    </SidebarHeader>

                    <SidebarContent className="px-2 py-4 overflow-y-auto scrollbar-thin scrollbar-thumb-gray-300 hover:scrollbar-thumb-gray-400 dark:scrollbar-thumb-gray-700 dark:hover:scrollbar-thumb-gray-600 scrollbar-track-transparent">
                        {navItems.map((group, index) => (
                            <div key={group.label}>
                                <SidebarGroup>
                                    <SidebarGroupLabel className="text-xs font-semibold text-muted-foreground/80 px-2 mb-2">
                                        {group.label}
                                    </SidebarGroupLabel>
                                    <SidebarGroupContent>
                                        <NavMain items={group.items} />
                                    </SidebarGroupContent>
                                </SidebarGroup>
                                {index < navItems.length - 1 && <SidebarSeparator className="my-2" />}
                            </div>
                        ))}
                    </SidebarContent>

                    <SidebarFooter className="p-3">
                        <div className="relative group">
                            {/* Gradient Glow Effect */}
                            <div className="absolute -inset-0.5 bg-gradient-to-r from-sky-600 via-cyan-600 to-teal-500 rounded-xl blur opacity-20 group-hover:opacity-40 transition duration-300" />
                            
                            {/* Profile Card */}
                            <div className="relative rounded-xl border border-border/50 bg-sidebar/50 backdrop-blur-md p-3 shadow-lg">
                                <div className="flex flex-col gap-3 group-data-[collapsible=icon]:hidden">
                                    {/* Avatar + User Info Row */}
                                    <div className="flex items-center gap-3">
                                        <div className="relative shrink-0">
                                            <div className="absolute -inset-1 bg-gradient-to-r from-sky-600 to-cyan-600 rounded-full blur opacity-50" />
                                            <Avatar className="h-10 w-10 relative border-2 border-background">
                                                <AvatarImage src={user?.avatar} alt={user?.name} />
                                                <AvatarFallback className="bg-gradient-to-br from-sky-600 to-cyan-600 text-white font-bold text-sm">
                                                    {user?.name ? getInitials(user.name) : 'SA'}
                                                </AvatarFallback>
                                            </Avatar>
                                        </div>
                                        
                                        {/* Name, Badge, Email Stack */}
                                        <div className="flex-1 min-w-0 space-y-0.5">
                                            <p className="text-sm font-semibold truncate leading-tight">
                                                {user?.name || 'Admin'}
                                            </p>
                                            <Badge 
                                                variant="outline" 
                                                className="w-fit h-4 px-1.5 text-[9px] bg-cyan-500/10 border-cyan-500/30 text-cyan-700 dark:text-cyan-300 font-semibold"
                                            >
                                                {user?.role?.name === 'society_admin' ? 'ADMIN' : 'USER'}
                                            </Badge>
                                            <p className="text-[10px] text-muted-foreground truncate leading-tight">
                                                {user?.email}
                                            </p>
                                        </div>
                                    </div>

                                    {/* Buttons Row */}
                                    <div className="flex items-center gap-2">
                                        <Link
                                            href="/settings/profile"
                                            className="flex-1 flex items-center justify-center gap-1.5 px-2.5 py-1.5 rounded-lg bg-muted hover:bg-muted/80 text-xs font-medium transition-colors"
                                            title="Profile"
                                        >
                                            <Settings className="h-3 w-3" />
                                            Profile
                                        </Link>
                                        <button
                                            onClick={handleLogout}
                                            className="flex-1 flex items-center justify-center gap-1.5 px-2.5 py-1.5 rounded-lg bg-red-500/10 hover:bg-red-500/20 text-red-600 dark:text-red-400 text-xs font-medium transition-colors"
                                            title="Logout"
                                        >
                                            <LogOut className="h-3 w-3" />
                                            Logout
                                        </button>
                                    </div>
                                </div>

                                {/* Collapsed Icon State */}
                                <div className="hidden group-data-[collapsible=icon]:flex flex-col items-center gap-2">
                                    <div className="relative">
                                        <div className="absolute -inset-1 bg-gradient-to-r from-sky-600 to-cyan-600 rounded-full blur opacity-50" />
                                        <Avatar className="h-10 w-10 relative border-2 border-background">
                                            <AvatarImage src={user?.avatar} alt={user?.name} />
                                            <AvatarFallback className="bg-gradient-to-br from-sky-600 to-cyan-600 text-white font-bold">
                                                {user?.name ? getInitials(user.name) : 'SA'}
                                            </AvatarFallback>
                                        </Avatar>
                                    </div>
                                    <button
                                        onClick={handleLogout}
                                        className="p-2 rounded-lg hover:bg-red-500/10 text-muted-foreground hover:text-red-600 dark:hover:text-red-400 transition-colors"
                                        title="Logout"
                                    >
                                        <LogOut className="h-4 w-4" />
                                    </button>
                                </div>
                            </div>
                        </div>
                    </SidebarFooter>
                </Sidebar>

                <AppContent variant="sidebar" className="overflow-x-hidden">
                    <AppSidebarHeader breadcrumbs={breadcrumbs} />
                    {children}
                </AppContent>
            </AppShell>
            
            {/* Full-screen emergency alert overlay */}
            <EmergencyAlertOverlay societyId={society.id} pollingInterval={5000} />
        </FlashMessagesProvider>
    );
}
