import { Link, usePage } from '@inertiajs/react';
import { Button } from '@/components/ui/button';
import { Moon, Sun, Menu, X, Facebook, Twitter, Instagram, Linkedin, Mail } from 'lucide-react';
import { useState, useEffect } from 'react';
import { Input } from '@/components/ui/input';

interface PublicLayoutProps {
    children: React.ReactNode;
}

export default function PublicLayout({ children }: PublicLayoutProps) {
    const { settings, auth } = usePage().props as any;
    const [isScrolled, setIsScrolled] = useState(false);
    const [mobileMenuOpen, setMobileMenuOpen] = useState(false);

    const [theme, setTheme] = useState<'light' | 'dark'>('light');

    useEffect(() => {
        const savedTheme = localStorage.getItem('theme') as 'light' | 'dark' | null;
        if (savedTheme) {
            setTheme(savedTheme);
            document.documentElement.classList.toggle('dark', savedTheme === 'dark');
        } else if (window.matchMedia('(prefers-color-scheme: dark)').matches) {
            setTheme('dark');
            document.documentElement.classList.toggle('dark', true);
        }
    }, []);

    const toggleTheme = () => {
        const newTheme = theme === 'light' ? 'dark' : 'light';
        setTheme(newTheme);
        localStorage.setItem('theme', newTheme);
        document.documentElement.classList.toggle('dark', newTheme === 'dark');
    };

    const appName = settings?.app_name || 'Societify';
    const appLogo = settings?.app_logo;
    const appLogoDark = settings?.app_logo_dark;
    const primaryColor = settings?.primary_color || '#10b981';

    // Get the appropriate logo based on theme
    const currentLogo = theme === 'dark' && appLogoDark ? appLogoDark : appLogo;

    useEffect(() => {
        const handleScroll = () => {
            setIsScrolled(window.scrollY > 10);
        };
        window.addEventListener('scroll', handleScroll);
        return () => window.removeEventListener('scroll', handleScroll);
    }, []);

    // Apply primary color to CSS variable
    useEffect(() => {
        if (primaryColor) {
            document.documentElement.style.setProperty('--primary', primaryColor);
        }
    }, [primaryColor]);

    return (
        <div className="min-h-screen bg-background flex flex-col font-sans antialiased transition-colors duration-300">
            {/* Premium Navbar */}
            <header className="fixed top-0 left-0 right-0 z-50 px-4 py-3">
                <div className={`max-w-6xl mx-auto transition-all duration-500 ${
                    isScrolled || mobileMenuOpen 
                        ? 'bg-white/80 dark:bg-slate-900/80 backdrop-blur-xl border border-slate-200/50 dark:border-slate-700/50 shadow-lg shadow-slate-900/5 dark:shadow-black/20 rounded-2xl' 
                        : 'bg-transparent'
                }`}>
                    <div className="px-4 lg:px-6 h-16 flex items-center justify-between">
                        {/* Logo */}
                        <Link href="/" className="flex items-center gap-2.5 font-bold text-xl tracking-tight group">
                            {currentLogo ? (
                                <img src={currentLogo} alt={appName} className="h-12 w-auto max-w-[220px] object-contain transition-transform group-hover:scale-105" />
                            ) : (
                                <>
                                    <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-cyan-500 to-teal-500 flex items-center justify-center text-white font-bold shadow-lg shadow-cyan-500/25 transition-transform group-hover:scale-105 group-hover:shadow-cyan-500/40">
                                        {appName.charAt(0)}
                                    </div>
                                    <span className="bg-gradient-to-r from-slate-900 to-slate-700 dark:from-white dark:to-slate-300 bg-clip-text text-transparent text-2xl">{appName}</span>
                                </>
                            )}
                        </Link>

                        {/* Desktop Navigation */}
                        <nav className="hidden lg:flex items-center gap-6">
                            {/* Nav Links - Clean & Bold */}
                            <div className="flex items-center gap-6 mr-6">
                                {settings?.enable_landing_page === '1' && (
                                    <Link 
                                        href="/" 
                                        className="text-sm font-bold text-slate-700 dark:text-slate-200 hover:text-cyan-600 dark:hover:text-cyan-400 transition-colors"
                                    >
                                        Home
                                    </Link>
                                )}
                                <Link 
                                    href="/#features" 
                                    className="text-sm font-bold text-slate-700 dark:text-slate-200 hover:text-cyan-600 dark:hover:text-cyan-400 transition-colors"
                                >
                                    Features
                                </Link>
                                {settings?.enable_pricing_page === '1' && (
                                    <Link 
                                        href="/pricing" 
                                        className="text-sm font-bold text-slate-700 dark:text-slate-200 hover:text-cyan-600 dark:hover:text-cyan-400 transition-colors"
                                    >
                                        Pricing
                                    </Link>
                                )}
                                <Link 
                                    href="/about" 
                                    className="text-sm font-bold text-slate-700 dark:text-slate-200 hover:text-cyan-600 dark:hover:text-cyan-400 transition-colors"
                                >
                                    About
                                </Link>
                            </div>
                            
                            {/* Actions */}
                            <div className="flex items-center gap-3">
                                {/* Theme Toggle */}
                                <button
                                    onClick={toggleTheme}
                                    className="w-9 h-9 rounded-full bg-slate-100 dark:bg-slate-800 flex items-center justify-center text-slate-600 dark:text-slate-300 hover:bg-slate-200 dark:hover:bg-slate-700 transition-colors"
                                >
                                    {theme === 'light' ? <Moon className="h-[18px] w-[18px]" /> : <Sun className="h-[18px] w-[18px]" />}
                                </button>

                                {auth.user ? (
                                    <Link href="/dashboard">
                                        <Button className="rounded-full px-5 h-9 bg-gradient-to-r from-cyan-600 to-teal-600 hover:from-cyan-700 hover:to-teal-700 shadow-md shadow-cyan-500/20 hover:shadow-cyan-500/30 transition-all text-sm font-medium">
                                            Dashboard
                                        </Button>
                                    </Link>
                                ) : (
                                    <>
                                        <Link href="/login">
                                            <Button variant="ghost" className="rounded-full px-4 h-9 text-sm font-medium text-slate-600 dark:text-slate-300 hover:text-slate-900 dark:hover:text-white hover:bg-slate-100 dark:hover:bg-slate-800">
                                                Log in
                                            </Button>
                                        </Link>
                                        {settings?.enable_registration === '1' && (
                                            <Link href="/register">
                                                <Button className="rounded-full px-5 h-9 bg-gradient-to-r from-cyan-600 to-teal-600 hover:from-cyan-700 hover:to-teal-700 shadow-md shadow-cyan-500/20 hover:shadow-cyan-500/30 transition-all text-sm font-medium">
                                                    Get Started
                                                </Button>
                                            </Link>
                                        )}
                                    </>
                                )}
                            </div>
                        </nav>

                        {/* Mobile Menu Toggle */}
                        <div className="lg:hidden flex items-center gap-2">
                            <button
                                onClick={toggleTheme}
                                className="w-9 h-9 rounded-full bg-slate-100 dark:bg-slate-800 flex items-center justify-center text-slate-600 dark:text-slate-300"
                            >
                                {theme === 'light' ? <Moon className="h-[18px] w-[18px]" /> : <Sun className="h-[18px] w-[18px]" />}
                            </button>
                            <button
                                className="w-9 h-9 rounded-full bg-slate-100 dark:bg-slate-800 flex items-center justify-center text-slate-600 dark:text-slate-300"
                                onClick={() => setMobileMenuOpen(!mobileMenuOpen)}
                            >
                                {mobileMenuOpen ? <X className="h-5 w-5" /> : <Menu className="h-5 w-5" />}
                            </button>
                        </div>
                    </div>

                    {/* Mobile Nav - Slide Down */}
                    {mobileMenuOpen && (
                        <div className="lg:hidden border-t border-slate-200/50 dark:border-slate-700/50 p-4 animate-in slide-in-from-top-2 duration-200">
                            <div className="flex flex-col gap-1">
                                <Link href="/" className="px-4 py-3 text-sm font-medium rounded-xl text-slate-700 dark:text-slate-200 hover:bg-slate-100 dark:hover:bg-slate-800 transition-colors" onClick={() => setMobileMenuOpen(false)}>
                                    Home
                                </Link>
                                <Link href="/#features" className="px-4 py-3 text-sm font-medium rounded-xl text-slate-700 dark:text-slate-200 hover:bg-slate-100 dark:hover:bg-slate-800 transition-colors" onClick={() => setMobileMenuOpen(false)}>
                                    Features
                                </Link>
                                {settings?.enable_pricing_page === '1' && (
                                    <Link href="/pricing" className="px-4 py-3 text-sm font-medium rounded-xl text-slate-700 dark:text-slate-200 hover:bg-slate-100 dark:hover:bg-slate-800 transition-colors" onClick={() => setMobileMenuOpen(false)}>
                                        Pricing
                                    </Link>
                                )}
                                <Link href="/about" className="px-4 py-3 text-sm font-medium rounded-xl text-slate-700 dark:text-slate-200 hover:bg-slate-100 dark:hover:bg-slate-800 transition-colors" onClick={() => setMobileMenuOpen(false)}>
                                    About
                                </Link>
                            </div>
                            <div className="flex flex-col gap-2 mt-4 pt-4 border-t border-slate-200/50 dark:border-slate-700/50">
                                {auth.user ? (
                                    <Link href="/dashboard" className="w-full" onClick={() => setMobileMenuOpen(false)}>
                                        <Button className="w-full justify-center rounded-xl h-11 bg-gradient-to-r from-cyan-600 to-teal-600">Dashboard</Button>
                                    </Link>
                                ) : (
                                    <>
                                        <Link href="/login" className="w-full" onClick={() => setMobileMenuOpen(false)}>
                                            <Button variant="outline" className="w-full justify-center rounded-xl h-11 border-slate-200 dark:border-slate-700">Log in</Button>
                                        </Link>
                                        {settings?.enable_registration === '1' && (
                                            <Link href="/register" className="w-full" onClick={() => setMobileMenuOpen(false)}>
                                                <Button className="w-full justify-center rounded-xl h-11 bg-gradient-to-r from-cyan-600 to-teal-600">Get Started</Button>
                                            </Link>
                                        )}
                                    </>
                                )}
                            </div>
                        </div>
                    )}
                </div>
            </header>

            {/* Main Content - No top padding to allow hero to extend behind header */}
            <main className="flex-1">
                {children}
            </main>

            {/* Footer */}
            {/* Premium Footer */}
            <footer className="relative pt-24 pb-12 overflow-hidden border-t border-slate-200 dark:border-slate-800 bg-slate-50/50 dark:bg-slate-900/50 backdrop-blur-sm">
                 {/* Background Glow */}
                 <div className="absolute top-0 left-1/4 w-96 h-96 bg-cyan-500/5 rounded-full blur-[100px] pointer-events-none" />
                 <div className="absolute bottom-0 right-1/4 w-96 h-96 bg-teal-500/5 rounded-full blur-[100px] pointer-events-none" />

                <div className="container mx-auto px-4 relative z-10">
                    <div className="grid gap-12 lg:gap-8 lg:grid-cols-12 mb-16">
                        {/* Brand Column */}
                        <div className="lg:col-span-4">
                            <div className="flex items-center gap-2 mb-6">
                                {currentLogo ? (
                                    <img src={currentLogo} alt={appName} className="h-10 w-auto max-w-[180px] object-contain" />
                                ) : (
                                    <div className="flex items-center gap-2">
                                        <div className="h-10 w-10 rounded-xl bg-gradient-to-br from-cyan-500 to-teal-500 flex items-center justify-center text-white font-bold shadow-lg shadow-cyan-500/25">
                                            {appName.charAt(0)}
                                        </div>
                                        <span className="text-xl font-bold bg-gradient-to-r from-slate-900 to-slate-700 dark:from-white dark:to-slate-300 bg-clip-text text-transparent">{appName}</span>
                                    </div>
                                )}
                            </div>
                            <p className="text-slate-600 dark:text-slate-400 mb-8 leading-relaxed max-w-sm">
                                The complete operating system for modern housing societies. Simplify management, enhance security, and build a vibrant community with {appName}.
                            </p>
                            <div className="flex gap-4">
                                <a href="#" className="w-10 h-10 rounded-full bg-white dark:bg-slate-800 border border-slate-200 dark:border-slate-700 flex items-center justify-center text-slate-500 dark:text-slate-400 hover:text-white hover:bg-cyan-600 hover:border-cyan-600 dark:hover:bg-cyan-600 dark:hover:border-cyan-600 transition-all shadow-sm">
                                    <Facebook className="h-5 w-5" />
                                </a>
                                <a href="#" className="w-10 h-10 rounded-full bg-white dark:bg-slate-800 border border-slate-200 dark:border-slate-700 flex items-center justify-center text-slate-500 dark:text-slate-400 hover:text-white hover:bg-cyan-600 hover:border-cyan-600 dark:hover:bg-cyan-600 dark:hover:border-cyan-600 transition-all shadow-sm">
                                    <Twitter className="h-5 w-5" />
                                </a>
                                <a href="#" className="w-10 h-10 rounded-full bg-white dark:bg-slate-800 border border-slate-200 dark:border-slate-700 flex items-center justify-center text-slate-500 dark:text-slate-400 hover:text-white hover:bg-pink-600 hover:border-pink-600 dark:hover:bg-pink-600 dark:hover:border-pink-600 transition-all shadow-sm">
                                    <Instagram className="h-5 w-5" />
                                </a>
                                <a href="#" className="w-10 h-10 rounded-full bg-white dark:bg-slate-800 border border-slate-200 dark:border-slate-700 flex items-center justify-center text-slate-500 dark:text-slate-400 hover:text-white hover:bg-blue-600 hover:border-blue-600 dark:hover:bg-blue-600 dark:hover:border-blue-600 transition-all shadow-sm">
                                    <Linkedin className="h-5 w-5" />
                                </a>
                            </div>
                        </div>
                        
                        {/* Links Columns */}
                        <div className="lg:col-span-2">
                            <h4 className="font-bold text-slate-900 dark:text-white mb-6">Product</h4>
                            <ul className="space-y-4">
                                <li><Link href="/#features" className="text-slate-600 dark:text-slate-400 hover:text-cyan-600 dark:hover:text-cyan-400 font-medium transition-colors">Features</Link></li>
                                <li><Link href="/pricing" className="text-slate-600 dark:text-slate-400 hover:text-cyan-600 dark:hover:text-cyan-400 font-medium transition-colors">Pricing</Link></li>
                                <li><Link href="/about" className="text-slate-600 dark:text-slate-400 hover:text-cyan-600 dark:hover:text-cyan-400 font-medium transition-colors">Testimonials</Link></li>
                                <li><Link href="/faq" className="text-slate-600 dark:text-slate-400 hover:text-cyan-600 dark:hover:text-cyan-400 font-medium transition-colors">FAQ</Link></li>
                            </ul>
                        </div>
                        
                        <div className="lg:col-span-2">
                            <h4 className="font-bold text-slate-900 dark:text-white mb-6">Company</h4>
                            <ul className="space-y-4">
                                <li><Link href="/about" className="text-slate-600 dark:text-slate-400 hover:text-cyan-600 dark:hover:text-cyan-400 font-medium transition-colors">About Us</Link></li>
                                <li><a href="#" className="text-slate-600 dark:text-slate-400 hover:text-cyan-600 dark:hover:text-cyan-400 font-medium transition-colors">Careers</a></li>
                                <li><a href="#" className="text-slate-600 dark:text-slate-400 hover:text-cyan-600 dark:hover:text-cyan-400 font-medium transition-colors">Blog</a></li>
                                <li><a href="#" className="text-slate-600 dark:text-slate-400 hover:text-cyan-600 dark:hover:text-cyan-400 font-medium transition-colors">Contact</a></li>
                            </ul>
                        </div>
                        
                        {/* Newsletter Column */}
                        <div className="lg:col-span-4">
                            <h4 className="font-bold text-slate-900 dark:text-white mb-6">Stay Updated</h4>
                            <p className="text-slate-600 dark:text-slate-400 mb-6 text-sm">
                                Subscribe to our newsletter for the latest updates, features, and community management tips.
                            </p>
                            <div className="flex gap-2">
                                <Input 
                                    placeholder="Enter your email" 
                                    className="bg-white dark:bg-slate-800 border-slate-200 dark:border-slate-700 focus:ring-cyan-500 focus:border-cyan-500 rounded-xl" 
                                />
                                <Button size="icon" className="rounded-xl bg-gradient-to-r from-cyan-600 to-teal-600 hover:from-cyan-700 hover:to-teal-700 shadow-lg shadow-cyan-500/20 shrink-0">
                                    <Mail className="h-4 w-4" />
                                </Button>
                            </div>
                        </div>
                    </div>
                    
                    {/* Bottom Bar */}
                    <div className="pt-8 border-t border-slate-200 dark:border-slate-700 flex flex-col md:flex-row justify-between items-center gap-4">
                        <p className="text-sm text-slate-500 dark:text-slate-400">© {new Date().getFullYear()} {appName}. All rights reserved.</p>
                        <div className="flex gap-6 text-sm">
                            <Link href="/privacy-policy" className="text-slate-500 dark:text-slate-400 hover:text-slate-900 dark:hover:text-white font-medium transition-colors">Privacy Policy</Link>
                            <Link href="/terms" className="text-slate-500 dark:text-slate-400 hover:text-slate-900 dark:hover:text-white font-medium transition-colors">Terms of Service</Link>
                            <Link href="/refund-policy" className="text-slate-500 dark:text-slate-400 hover:text-slate-900 dark:hover:text-white font-medium transition-colors">Refund Policy</Link>
                        </div>
                    </div>
                </div>
            </footer>
        </div>
    );
}
