import Swal from 'sweetalert2';
import { usePage } from '@inertiajs/react';
import { useEffect } from 'react';

// SweetAlert2 configuration
const Toast = Swal.mixin({
    toast: true,
    position: 'top-end',
    showConfirmButton: false,
    timer: 3000,
    timerProgressBar: true,
    didOpen: (toast) => {
        toast.onmouseenter = Swal.stopTimer;
        toast.onmouseleave = Swal.resumeTimer;
    }
});

export interface AlertMessage {
    type: 'success' | 'error' | 'warning' | 'info';
    message: string;
    title?: string;
}

// Hook to automatically show flash messages from Laravel
export function useFlashMessages() {
    const { props } = usePage<{
        flash?: {
            success?: string;
            error?: string;
            warning?: string;
            info?: string;
        };
    }>();

    useEffect(() => {
        if (!props.flash) return;

        if (props.flash.success) {
            showToast('success', props.flash.success);
        }
        if (props.flash.error) {
            showToast('error', props.flash.error);
        }
        if (props.flash.warning) {
            showToast('warning', props.flash.warning);
        }
        if (props.flash.info) {
            showToast('info', props.flash.info);
        }
    }, [props.flash]);
}

// Toast notification (auto-dismiss)
export function showToast(
    type: 'success' | 'error' | 'warning' | 'info',
    message: string,
    title?: string
) {
    return Toast.fire({
        icon: type,
        title: title || message,
        text: title ? message : undefined,
    });
}

// Alert dialog (requires confirmation)
export function showAlert(
    type: 'success' | 'error' | 'warning' | 'info' | 'question',
    title: string,
    message?: string
) {
    return Swal.fire({
        icon: type,
        title,
        text: message,
        confirmButtonText: 'OK',
        customClass: {
            confirmButton: 'bg-primary hover:bg-primary/90 text-white px-6 py-2 rounded-lg font-medium',
        },
        buttonsStyling: false,
    });
}

// Confirmation dialog
export function showConfirm(
    title: string,
    message?: string,
    confirmText: string = 'Yes, proceed',
    cancelText: string = 'Cancel'
) {
    return Swal.fire({
        icon: 'question',
        title,
        text: message,
        showCancelButton: true,
        confirmButtonText: confirmText,
        cancelButtonText: cancelText,
        customClass: {
            confirmButton: 'bg-primary hover:bg-primary/90 text-white px-6 py-2 rounded-lg font-medium mr-2',
            cancelButton: 'bg-gray-200 hover:bg-gray-300 text-gray-800 px-6 py-2 rounded-lg font-medium',
        },
        buttonsStyling: false,
        reverseButtons: true,
    });
}

// Delete confirmation
export function showDeleteConfirm(
    itemName: string = 'this item',
    message?: string
) {
    return Swal.fire({
        icon: 'warning',
        title: 'Are you sure?',
        text: message || `You are about to delete ${itemName}. This action cannot be undone.`,
        showCancelButton: true,
        confirmButtonText: 'Yes, delete it!',
        cancelButtonText: 'Cancel',
        customClass: {
            confirmButton: 'bg-red-600 hover:bg-red-700 text-white px-6 py-2 rounded-lg font-medium mr-2',
            cancelButton: 'bg-gray-200 hover:bg-gray-300 text-gray-800 px-6 py-2 rounded-lg font-medium',
        },
        buttonsStyling: false,
        reverseButtons: true,
    });
}

// Loading dialog
export function showLoading(title: string = 'Loading...', message?: string) {
    return Swal.fire({
        title,
        text: message,
        allowOutsideClick: false,
        didOpen: () => {
            Swal.showLoading();
        },
    });
}

// Close any open alert
export function closeAlert() {
    Swal.close();
}

// Default export with all methods
const useAlert = () => ({
    toast: showToast,
    alert: showAlert,
    confirm: showConfirm,
    deleteConfirm: showDeleteConfirm,
    loading: showLoading,
    close: closeAlert,
});

export default useAlert;
