import { type NavItem } from '@/types';
import {
    LayoutGrid,
    Settings,
    Users,
    Building2,
    Receipt,
    CreditCard,
    Landmark,
    TrendingDown,
    Wrench,
    ShieldCheck,
    Megaphone,
    FileText,
    Calendar,
    Layers,
    UserCog,
    MessageSquare,
    CalendarDays,
    Wallet,
} from 'lucide-react';

export interface MenuItem extends NavItem {
    permission?: string;
    roles?: string[];
}

/**
 * Get menu items for Super Admin (Full Access)
 */
export function getSuperAdminMenu(): MenuGroup[] {
    return [
        {
            label: 'DASHBOARD',
            items: [
                {
                    title: 'Dashboard',
                    href: '/super-admin/dashboard',
                    icon: LayoutGrid,
                },
            ],
        },
        {
            label: 'MANAGEMENT',
            items: [
                {
                    title: 'Societies',
                    href: '/super-admin/societies',
                    icon: Building2,
                },
                {
                    title: 'Users',
                    href: '/super-admin/users',
                    icon: Users,
                },
            ],
        },
        {
            label: 'SUBSCRIPTION',
            items: [
                {
                    title: 'Subscription Plans',
                    href: '/super-admin/subscription-plans',
                    icon: CreditCard,
                },
            ],
        },
        {
            label: 'SETTINGS',
            items: [
                {
                    title: 'Settings',
                    href: '/super-admin/settings',
                    icon: Settings,
                },
            ],
        },
    ];
}

/**
 * Get menu items for Society Admin (Full Access - 12 items)
 */
export interface MenuGroup {
    label: string;
    items: MenuItem[];
}

/**
 * Get menu items for Society Admin (Full Access - 12 items)
 */
export function getSocietyAdminMenu(societyId: number | string): MenuGroup[] {
    return [
        {
            label: 'DASHBOARD',
            items: [
                {
                    title: 'Dashboard',
                    href: `/societies/${societyId}/dashboard`,
                    icon: LayoutGrid,
                    permission: 'dashboard.view',
                },
            ],
        },
        {
            label: 'MANAGEMENT',
            items: [
                {
                    title: 'Residents',
                    href: `/societies/${societyId}/residents`,
                    icon: Users,
                    permission: 'residents.view',
                },
                {
                    title: 'Units & Buildings',
                    href: `/societies/${societyId}/units`,
                    icon: Building2,
                    permission: 'units.view',
                },
                {
                    title: 'Facilities',
                    href: `/societies/${societyId}/facilities`,
                    icon: CalendarDays,
                },
                {
                    title: 'Visitors',
                    href: `/societies/${societyId}/visitors`,
                    icon: ShieldCheck,
                    permission: 'visitors.view',
                },
                {
                    title: 'Maintenance',
                    href: `/societies/${societyId}/maintenance`,
                    icon: Wrench,
                    permission: 'maintenance.view',
                },
                {
                    title: 'Team',
                    href: `/societies/${societyId}/team`,
                    icon: UserCog,
                    permission: 'team.view',
                },
            ],
        },
        {
            label: 'FINANCE',
            items: [
                {
                    title: 'Bill Cycles',
                    href: `/societies/${societyId}/bill-cycles`,
                    icon: Calendar,
                    permission: 'billing.cycles.view',
                },
                {
                    title: 'Structures',
                    href: `/societies/${societyId}/maintenance-structures`,
                    icon: Layers,
                    permission: 'billing.bills.view',
                },
                {
                    title: 'Billing',
                    href: `/societies/${societyId}/bills`,
                    icon: Receipt,
                    permission: 'billing.bills.view',
                },
                {
                    title: 'Payments',
                    href: `/societies/${societyId}/payments`,
                    icon: CreditCard,
                    permission: 'payments.view',
                },
                {
                    title: 'Expenses',
                    href: `/societies/${societyId}/expenses`,
                    icon: TrendingDown,
                    permission: 'expenses.view',
                },
                {
                    title: 'Bank Accounts',
                    href: `/societies/${societyId}/bank-accounts`,
                    icon: Landmark,
                    permission: 'bank_accounts.view',
                },
                {
                    title: 'Collector Wallets',
                    href: `/societies/${societyId}/collectors/wallets`,
                    icon: Wallet,
                },
            ],
        },
        {
            label: 'COMMUNICATION & REPORTS',
            items: [
                {
                    title: 'Announcements',
                    href: `/societies/${societyId}/announcements`,
                    icon: Megaphone,
                    permission: 'announcements.view',
                },
                {
                    title: 'Community',
                    href: `/societies/${societyId}/community`,
                    icon: MessageSquare,
                },
                {
                    title: 'Reports',
                    href: `/societies/${societyId}/reports`,
                    icon: FileText,
                    permission: 'reports.view',
                },
            ],
        },
        {
            label: 'MONETIZATION & ADS',
            items: [
                {
                    title: 'Ad Requests',
                    href: `/societies/${societyId}/monetization/requests`,
                    icon: FileText,
                },
                {
                    title: 'Active Campaigns',
                    href: `/societies/${societyId}/monetization`,
                    icon: Megaphone,
                    exact: true,
                },
                {
                    title: 'Slot Configuration',
                    href: `/societies/${societyId}/monetization/settings`,
                    icon: Settings,
                },
            ],
        },
        {
            label: 'SETTINGS',
            items: [
                {
                    title: 'Society Settings',
                    href: `/societies/${societyId}/settings`,
                    icon: Settings,
                    permission: 'society.settings.view',
                },
            ],
        },
    ];
}

/**
 * Get menu items for Sub-Admin (Limited Access - 8 items)
 */
export function getSubAdminMenu(societyId: number | string): MenuGroup[] {
    return [
        {
            label: 'DASHBOARD',
            items: [
                {
                    title: 'Dashboard',
                    href: `/societies/${societyId}/sub-admin/dashboard`,
                    icon: LayoutGrid,
                    permission: 'dashboard.view',
                },
            ],
        },
        {
            label: 'MANAGEMENT',
            items: [
                {
                    title: 'Residents',
                    href: `/societies/${societyId}/residents`,
                    icon: Users,
                    permission: 'residents.view',
                },
                {
                    title: 'Units',
                    href: `/societies/${societyId}/units`,
                    icon: Building2,
                    permission: 'units.view',
                },
                {
                    title: 'Visitors',
                    href: `/societies/${societyId}/visitors`,
                    icon: ShieldCheck,
                    permission: 'visitors.view',
                },
                {
                    title: 'Maintenance',
                    href: `/societies/${societyId}/maintenance`,
                    icon: Wrench,
                    permission: 'maintenance.view',
                },
            ],
        },
        {
            label: 'FINANCE',
            items: [
                {
                    title: 'Billing',
                    href: `/societies/${societyId}/bills`,
                    icon: Receipt,
                    permission: 'billing.bills.view',
                },
                {
                    title: 'Expenses',
                    href: `/societies/${societyId}/expenses`,
                    icon: TrendingDown,
                    permission: 'expenses.view',
                },
            ],
        },
        {
            label: 'COMMUNICATION',
            items: [
                {
                    title: 'Announcements',
                    href: `/societies/${societyId}/announcements`,
                    icon: Megaphone,
                    permission: 'announcements.view',
                },
            ],
        },
    ];
}

/**
 * Get menu items for Accountant (Financial Only - 6 items)
 */
export function getAccountantMenu(societyId: number | string): MenuGroup[] {
    return [
        {
            label: 'DASHBOARD',
            items: [
                {
                    title: 'Financial Dashboard',
                    href: `/societies/${societyId}/accountant/dashboard`,
                    icon: LayoutGrid,
                    permission: 'dashboard.view',
                },
            ],
        },
        {
            label: 'FINANCE',
            items: [
                {
                    title: 'Billing',
                    href: `/societies/${societyId}/bills`,
                    icon: Receipt,
                    permission: 'billing.bills.view',
                },
                {
                    title: 'Payments & Reconciliation',
                    href: `/societies/${societyId}/payments`,
                    icon: CreditCard,
                    permission: 'payments.view',
                },
                {
                    title: 'Bank Accounts',
                    href: `/societies/${societyId}/bank-accounts`,
                    icon: Landmark,
                    permission: 'bank_accounts.view',
                },
                {
                    title: 'Expenses',
                    href: `/societies/${societyId}/expenses`,
                    icon: TrendingDown,
                    permission: 'expenses.view',
                },
            ],
        },
        {
            label: 'REPORTS',
            items: [
                {
                    title: 'Financial Reports',
                    href: `/societies/${societyId}/reports`,
                    icon: FileText,
                    permission: 'reports.financial',
                },
            ],
        },
    ];
}

/**
 * Get menu items based on user role
 */
export function getMenuForRole(
    roleName: string,
    societyId: number | string,
    userPermissions: string[] = []
): MenuGroup[] {
    let menuGroups: MenuGroup[] = [];

    switch (roleName) {
        case 'society_admin':
            menuGroups = getSocietyAdminMenu(societyId);
            break;
        case 'sub_admin':
            menuGroups = getSubAdminMenu(societyId);
            break;
        case 'accountant':
            menuGroups = getAccountantMenu(societyId);
            break;
        default:
            menuGroups = [];
    }

    // Filter menu items based on permissions if provided
    if (userPermissions.length > 0) {
        menuGroups = menuGroups.map(group => ({
            ...group,
            items: group.items.filter((item) => {
                if (!item.permission) {
                    return true; // Include items without permission requirement
                }
                return userPermissions.includes(item.permission);
            })
        })).filter(group => group.items.length > 0); // Remove empty groups
    }

    return menuGroups;
}

