import { useEffect, useState, useCallback, useRef } from 'react';
import { AlertTriangle, Phone, MapPin, X, Volume2 } from 'lucide-react';
import { Button } from '@/components/ui/button';

interface EmergencyAlert {
    id: number;
    status: string;
    user: {
        id: number;
        name: string;
        phone: string | null;
    };
}

interface EmergencyData {
    active: boolean;
    message?: string;
    location?: string;
    timestamp?: string;
    alert?: EmergencyAlert;
}

interface EmergencyAlertOverlayProps {
    societyId: number;
    pollingInterval?: number; // in milliseconds
}

export function EmergencyAlertOverlay({ 
    societyId, 
    pollingInterval = 5000 
}: EmergencyAlertOverlayProps) {
    const [emergency, setEmergency] = useState<EmergencyData | null>(null);
    const [dismissed, setDismissed] = useState<number | null>(null);
    const [isResolving, setIsResolving] = useState(false);
    const audioRef = useRef<HTMLAudioElement | null>(null);

    const checkEmergency = useCallback(async () => {
        try {
            const response = await fetch(`/societies/${societyId}/emergency/active`, {
                headers: {
                    'Accept': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest',
                },
                credentials: 'include',
            });
            
            if (response.ok) {
                const data: EmergencyData = await response.json();
                
                if (data.active && data.alert?.id !== dismissed) {
                    setEmergency(data);
                    // Play alert sound
                    if (audioRef.current) {
                        audioRef.current.play().catch(() => {});
                    }
                } else if (!data.active) {
                    setEmergency(null);
                    setDismissed(null);
                }
            }
        } catch (error) {
            console.error('Failed to check emergency status:', error);
        }
    }, [dismissed, societyId]);

    useEffect(() => {
        // Initial check
        checkEmergency();
        
        // Set up polling
        const interval = setInterval(checkEmergency, pollingInterval);
        
        return () => clearInterval(interval);
    }, [checkEmergency, pollingInterval]);

    const handleDismiss = () => {
        if (emergency?.alert?.id) {
            setDismissed(emergency.alert.id);
        }
        setEmergency(null);
    };

    const handleResolve = async () => {
        if (!emergency?.alert?.id) return;
        
        setIsResolving(true);
        try {
            // Get CSRF token from meta tag
            const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '';
            
            const response = await fetch(`/societies/${societyId}/emergency/${emergency.alert.id}/resolve`, {
                method: 'POST',
                headers: {
                    'Accept': 'application/json',
                    'Content-Type': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest',
                    'X-CSRF-TOKEN': csrfToken,
                },
                credentials: 'include',
            });
            
            if (response.ok) {
                setEmergency(null);
                setDismissed(null);
            }
        } catch (error) {
            console.error('Failed to resolve emergency:', error);
        } finally {
            setIsResolving(false);
        }
    };

    if (!emergency?.active) return null;

    return (
        <>
            {/* Audio element for alert sound */}
            <audio ref={audioRef} loop>
                <source src="/sounds/emergency-siren.mp3" type="audio/mpeg" />
            </audio>

            {/* Full-screen overlay */}
            <div className="fixed inset-0 z-[9999] flex items-center justify-center bg-red-900/95 backdrop-blur-sm animate-pulse">
                {/* Animated background rings */}
                <div className="absolute inset-0 flex items-center justify-center overflow-hidden">
                    <div className="absolute w-[200%] h-[200%] animate-[spin_10s_linear_infinite]">
                        <div className="absolute inset-0 bg-gradient-to-r from-transparent via-red-500/20 to-transparent" />
                    </div>
                </div>

                {/* Close button */}
                <button
                    onClick={handleDismiss}
                    className="absolute top-4 right-4 p-2 rounded-full bg-white/10 hover:bg-white/20 text-white transition-colors"
                    title="Dismiss (alert will reappear if not resolved)"
                >
                    <X className="h-6 w-6" />
                </button>

                {/* Alert card */}
                <div className="relative z-10 max-w-lg w-full mx-4 bg-white rounded-2xl shadow-2xl overflow-hidden animate-bounce-slow">
                    {/* Header */}
                    <div className="bg-gradient-to-r from-red-600 to-red-700 px-6 py-4">
                        <div className="flex items-center gap-3">
                            <div className="p-3 bg-white/20 rounded-full animate-pulse">
                                <AlertTriangle className="h-8 w-8 text-white" />
                            </div>
                            <div>
                                <h2 className="text-2xl font-bold text-white">
                                    🚨 EMERGENCY ALERT
                                </h2>
                                <p className="text-red-100 text-sm">
                                    Immediate attention required
                                </p>
                            </div>
                        </div>
                    </div>

                    {/* Content */}
                    <div className="p-6 space-y-4">
                        <p className="text-gray-800 text-lg font-medium">
                            {emergency.message}
                        </p>

                        <div className="space-y-3">
                            <div className="flex items-center gap-3 text-gray-600">
                                <MapPin className="h-5 w-5 text-red-500" />
                                <span className="font-medium">{emergency.location}</span>
                            </div>
                            
                            {emergency.alert?.user?.phone && (
                                <div className="flex items-center gap-3 text-gray-600">
                                    <Phone className="h-5 w-5 text-green-500" />
                                    <a 
                                        href={`tel:${emergency.alert.user.phone}`}
                                        className="font-medium text-blue-600 hover:underline"
                                    >
                                        {emergency.alert.user.phone}
                                    </a>
                                </div>
                            )}

                            <div className="text-sm text-gray-500">
                                Triggered at: {emergency.timestamp ? new Date(emergency.timestamp).toLocaleString() : 'N/A'}
                            </div>
                        </div>
                    </div>

                    {/* Actions */}
                    <div className="px-6 py-4 bg-gray-50 flex gap-3">
                        <Button
                            onClick={handleResolve}
                            disabled={isResolving}
                            className="flex-1 bg-green-600 hover:bg-green-700 text-white font-semibold py-3"
                        >
                            {isResolving ? 'Resolving...' : '✓ Mark as Resolved'}
                        </Button>
                        <Button
                            onClick={handleDismiss}
                            variant="outline"
                            className="flex-1 py-3"
                        >
                            Dismiss
                        </Button>
                    </div>
                </div>

                {/* Mute button */}
                <button
                    onClick={() => {
                        if (audioRef.current) {
                            audioRef.current.paused 
                                ? audioRef.current.play() 
                                : audioRef.current.pause();
                        }
                    }}
                    className="absolute bottom-4 right-4 p-3 rounded-full bg-white/10 hover:bg-white/20 text-white transition-colors"
                    title="Toggle sound"
                >
                    <Volume2 className="h-6 w-6" />
                </button>
            </div>
        </>
    );
}
