import { Breadcrumbs } from '@/components/breadcrumbs';
import { SidebarTrigger } from '@/components/ui/sidebar';
import { Button } from '@/components/ui/button';
import {
    DropdownMenu,
    DropdownMenuContent,
    DropdownMenuItem,
    DropdownMenuLabel,
    DropdownMenuSeparator,
    DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import { Badge } from '@/components/ui/badge';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { type BreadcrumbItem as BreadcrumbItemType } from '@/types';
import { usePage, router } from '@inertiajs/react';
import { useAppearance } from '@/hooks/use-appearance';
import { 
    Bell, 
    MessageSquare, 
    User, 
    Settings, 
    LogOut,
    Clock,
    CheckCircle2,
    AlertCircle,
    Shield,
    Sun,
    Moon
} from 'lucide-react';

export function AppSidebarHeader({
    breadcrumbs = [],
}: {
    breadcrumbs?: BreadcrumbItemType[];
}) {
    const { auth, notifications: sharedNotifications } = usePage().props as any;
    const user = auth?.user;
    const { appearance, updateAppearance } = useAppearance();
    const { url } = usePage();

    // Determine logout URL based on current route
    const isSuperAdmin = url?.startsWith('/super-admin');
    const logoutUrl = isSuperAdmin ? '/super-admin/logout' : '/logout';

    // Use real notifications from backend
    const notifications = sharedNotifications || [];

    // Mock messages - TODO: Replace with real data from backend
    const messages: any[] = [];

    const unreadNotifications = notifications.filter(n => n.unread).length;
    const unreadMessages = messages.filter(m => m.unread).length;

    const getInitials = (name: string) => {
        return name
            .split(' ')
            .map(n => n[0])
            .join('')
            .toUpperCase()
            .slice(0, 2);
    };

    const getNotificationIcon = (type: string) => {
        switch (type) {
            case 'success':
                return <CheckCircle2 className="h-4 w-4 text-emerald-600" />;
            case 'warning':
                return <AlertCircle className="h-4 w-4 text-amber-600" />;
            default:
                return <Bell className="h-4 w-4 text-blue-600" />;
        }
    };

    return (
        <header className="sticky top-0 z-40 flex h-16 shrink-0 items-center gap-2 bg-background border-b border-border/40 px-6 transition-[width,height] ease-linear group-has-data-[collapsible=icon]/sidebar-wrapper:h-12 md:px-4">
            {/* Left Side - Breadcrumbs */}
            <div className="flex flex-1 items-center gap-3">
                <SidebarTrigger className="-ml-1" />
                <div className="h-6 w-px bg-border/50" />
                <div className="flex items-center gap-2">
                    <div className="p-1.5 rounded-lg bg-gradient-to-br from-sky-500 to-teal-500 shadow-sm">
                        <Shield className="h-4 w-4 text-white" />
                    </div>
                    <Breadcrumbs breadcrumbs={breadcrumbs} />
                </div>
            </div>

            {/* Right Side - Actions */}
            <div className="flex items-center gap-2">
                {/* Notifications */}
                <DropdownMenu>
                    <DropdownMenuTrigger asChild>
                        <Button variant="ghost" size="icon" className="relative">
                            <Bell className="h-5 w-5" />
                            {unreadNotifications > 0 && (
                                <Badge 
                                    variant="destructive" 
                                    className="absolute -top-1 -right-1 h-5 w-5 rounded-full p-0 flex items-center justify-center text-xs"
                                >
                                    {unreadNotifications}
                                </Badge>
                            )}
                        </Button>
                    </DropdownMenuTrigger>
                    <DropdownMenuContent align="end" className="w-80">
                        <DropdownMenuLabel className="flex items-center justify-between">
                            <span className="font-semibold">Notifications</span>
                            {unreadNotifications > 0 && (
                                <Badge variant="secondary" className="text-xs">
                                    {unreadNotifications} new
                                </Badge>
                            )}
                        </DropdownMenuLabel>
                        <DropdownMenuSeparator />
                        <div className="max-h-[400px] overflow-y-auto">
                            {notifications.map((notification) => (
                                <DropdownMenuItem
                                    key={notification.id}
                                    className="flex flex-col items-start gap-2 p-3 cursor-pointer"
                                >
                                    <div className="flex items-start gap-3 w-full">
                                        <div className="mt-0.5">
                                            {getNotificationIcon(notification.type)}
                                        </div>
                                        <div className="flex-1 space-y-1">
                                            <div className="flex items-center justify-between gap-2">
                                                <p className="text-sm font-medium leading-none">
                                                    {notification.title}
                                                </p>
                                                {notification.unread && (
                                                    <div className="h-2 w-2 rounded-full bg-blue-600" />
                                                )}
                                            </div>
                                            <p className="text-xs text-muted-foreground">
                                                {notification.message}
                                            </p>
                                            <div className="flex items-center gap-1 text-xs text-muted-foreground">
                                                <Clock className="h-3 w-3" />
                                                {notification.time}
                                            </div>
                                        </div>
                                    </div>
                                </DropdownMenuItem>
                            ))}
                        </div>
                        <DropdownMenuSeparator />
                        <DropdownMenuItem className="text-center justify-center text-sm text-primary cursor-pointer">
                            View all notifications
                        </DropdownMenuItem>
                    </DropdownMenuContent>
                </DropdownMenu>

                {/* Messages */}
                <DropdownMenu>
                    <DropdownMenuTrigger asChild>
                        <Button variant="ghost" size="icon" className="relative">
                            <MessageSquare className="h-5 w-5" />
                            {unreadMessages > 0 && (
                                <Badge 
                                    variant="destructive" 
                                    className="absolute -top-1 -right-1 h-5 w-5 rounded-full p-0 flex items-center justify-center text-xs"
                                >
                                    {unreadMessages}
                                </Badge>
                            )}
                        </Button>
                    </DropdownMenuTrigger>
                    <DropdownMenuContent align="end" className="w-80">
                        <DropdownMenuLabel className="flex items-center justify-between">
                            <span className="font-semibold">Messages</span>
                            {unreadMessages > 0 && (
                                <Badge variant="secondary" className="text-xs">
                                    {unreadMessages} new
                                </Badge>
                            )}
                        </DropdownMenuLabel>
                        <DropdownMenuSeparator />
                        <div className="max-h-[400px] overflow-y-auto">
                            {messages.map((message) => (
                                <DropdownMenuItem
                                    key={message.id}
                                    className="flex items-start gap-3 p-3 cursor-pointer"
                                >
                                    <Avatar className="h-8 w-8">
                                        <AvatarFallback className="text-xs bg-gradient-to-br from-cyan-500 to-blue-500 text-white">
                                            {getInitials(message.from)}
                                        </AvatarFallback>
                                    </Avatar>
                                    <div className="flex-1 space-y-1">
                                        <div className="flex items-center justify-between gap-2">
                                            <p className="text-sm font-medium leading-none">
                                                {message.from}
                                            </p>
                                            {message.unread && (
                                                <div className="h-2 w-2 rounded-full bg-blue-600" />
                                            )}
                                        </div>
                                        <p className="text-xs text-muted-foreground line-clamp-1">
                                            {message.message}
                                        </p>
                                        <div className="flex items-center gap-1 text-xs text-muted-foreground">
                                            <Clock className="h-3 w-3" />
                                            {message.time}
                                        </div>
                                    </div>
                                </DropdownMenuItem>
                            ))}
                        </div>
                        <DropdownMenuSeparator />
                        <DropdownMenuItem className="text-center justify-center text-sm text-primary cursor-pointer">
                            View all messages
                        </DropdownMenuItem>
                    </DropdownMenuContent>
                </DropdownMenu>

                {/* Theme Toggle */}
                <div className="inline-flex items-center rounded-full bg-muted p-1 shadow-sm border border-border/50">
                    <button
                        onClick={() => updateAppearance('light')}
                        className={`inline-flex items-center gap-1.5 rounded-full px-3 py-1.5 text-xs font-medium transition-all ${
                            appearance === 'light'
                                ? 'bg-background text-foreground shadow-sm'
                                : 'text-muted-foreground hover:text-foreground'
                        }`}
                        title="Light mode"
                    >
                        <Sun className="h-3.5 w-3.5" />
                    </button>
                    <button
                        onClick={() => updateAppearance('dark')}
                        className={`inline-flex items-center gap-1.5 rounded-full px-3 py-1.5 text-xs font-medium transition-all ${
                            appearance === 'dark'
                                ? 'bg-background text-foreground shadow-sm'
                                : 'text-muted-foreground hover:text-foreground'
                        }`}
                        title="Dark mode"
                    >
                        <Moon className="h-3.5 w-3.5" />
                    </button>
                    <button
                        onClick={() => updateAppearance('system')}
                        className={`inline-flex items-center gap-1.5 rounded-full px-3 py-1.5 text-xs font-medium transition-all ${
                            appearance === 'system'
                                ? 'bg-background text-foreground shadow-sm'
                                : 'text-muted-foreground hover:text-foreground'
                        }`}
                        title="System theme"
                    >
                        <Shield className="h-3.5 w-3.5" />
                    </button>
                </div>

                <div className="h-6 w-px bg-border" />

                {/* User Profile */}
                <DropdownMenu>
                    <DropdownMenuTrigger asChild>
                        <Button variant="ghost" className="relative h-10 gap-2 px-2">
                            <Avatar className="h-8 w-8">
                                <AvatarImage src={user?.avatar} alt={user?.name} />
                                <AvatarFallback className="bg-gradient-to-br from-sky-500 to-teal-500 text-white font-semibold">
                                    {user?.name ? getInitials(user.name) : 'SA'}
                                </AvatarFallback>
                            </Avatar>
                            <div className="hidden md:flex flex-col items-start text-left">
                                <span className="text-sm font-semibold">{user?.name || 'Super Admin'}</span>
                                <span className="text-xs text-muted-foreground">Administrator</span>
                            </div>
                        </Button>
                    </DropdownMenuTrigger>
                    <DropdownMenuContent align="end" className="w-56">
                        <DropdownMenuLabel>
                            <div className="flex flex-col space-y-1">
                                <p className="text-sm font-medium leading-none">{user?.name || 'Super Admin'}</p>
                                <p className="text-xs leading-none text-muted-foreground">
                                    {user?.email || 'admin@societify.com'}
                                </p>
                            </div>
                        </DropdownMenuLabel>
                        <DropdownMenuSeparator />
                        <DropdownMenuItem className="cursor-pointer">
                            <User className="mr-2 h-4 w-4" />
                            Profile
                        </DropdownMenuItem>
                        <DropdownMenuItem className="cursor-pointer">
                            <Settings className="mr-2 h-4 w-4" />
                            Settings
                        </DropdownMenuItem>
                        <DropdownMenuSeparator />
                        <DropdownMenuItem 
                            className="cursor-pointer text-red-600 focus:text-red-600 focus:bg-red-50 dark:focus:bg-red-950/50"
                            onClick={() => router.post(logoutUrl)}
                        >
                            <LogOut className="mr-2 h-4 w-4" />
                            Log out
                        </DropdownMenuItem>
                    </DropdownMenuContent>
                </DropdownMenu>
            </div>
        </header>
    );
}
