import { useState, useEffect } from 'react';
import { AlertTriangle, CreditCard, ArrowRight, RefreshCw, IndianRupee, Loader2, CheckCircle2, Clock } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { router, usePage } from '@inertiajs/react';
import axios from 'axios';

interface PaymentRequiredProps {
    status: 'pending_payment' | 'trial' | 'expired' | 'expired_grace';
    message: string;
    has_payment_gateway?: boolean;
}

declare global {
    interface Window {
        Razorpay: any;
    }
}

export default function PaymentRequired({ status, message, has_payment_gateway = true }: PaymentRequiredProps) {
    const { auth } = usePage().props as any;
    const [isLoading, setIsLoading] = useState(false);
    const [error, setError] = useState<string | null>(null);
    const [paymentSuccess, setPaymentSuccess] = useState(false);
    const [razorpayLoaded, setRazorpayLoaded] = useState(false);

    // Load Razorpay script
    useEffect(() => {
        // Check if Razorpay is already loaded
        if (window.Razorpay) {
            setRazorpayLoaded(true);
            return;
        }

        const script = document.createElement('script');
        script.src = 'https://checkout.razorpay.com/v1/checkout.js';
        script.async = true;
        
        script.onload = () => {
            setRazorpayLoaded(true);
        };
        
        script.onerror = () => {
            console.error('Failed to load Razorpay checkout script');
            setError('Failed to load payment gateway. Please refresh the page or contact support.');
            setRazorpayLoaded(false);
        };
        
        document.body.appendChild(script);
        
        return () => {
            if (document.body.contains(script)) {
                document.body.removeChild(script);
            }
        };
    }, []);

    const handleLogout = () => {
        router.post('/logout');
    };

    const handleRefresh = () => {
        router.reload();
    };

    const handlePayNow = async () => {
        setIsLoading(true);
        setError(null);

        // Check if Razorpay is loaded
        if (!window.Razorpay) {
            setError('Payment gateway is not loaded. Please refresh the page and try again.');
            setIsLoading(false);
            return;
        }

        try {
            // Create payment order for subscription renewal (using web route, not API)
            const response = await axios.post('/subscription/create-payment-order');

            if (!response.data.success) {
                const errorMsg = response.data.message 
                    || response.data.error 
                    || 'Failed to create payment order';
                console.error('Payment order creation failed:', response.data);
                setError(errorMsg);
                setIsLoading(false);
                return;
            }

            const { order_id, amount, key, subscription, gateway } = response.data;

            // Validate required fields
            if (!order_id || !amount || !key) {
                setError('Invalid payment configuration. Please contact support.');
                setIsLoading(false);
                return;
            }

            const options = {
                key: key,
                amount: amount * 100, // Convert to paise
                currency: 'INR',
                name: 'Societify',
                description: `${subscription?.plan_name || 'Subscription'} - ${status === 'expired' || status === 'expired_grace' ? 'Renewal' : 'Activation'}`,
                order_id: order_id,
                handler: async function (paymentResponse: any) {
                    try {
                        // Verify payment and activate subscription (using web route)
                        const verifyResponse = await axios.post('/subscription/verify-payment', {
                            gateway: gateway || 'razorpay',
                            razorpay_payment_id: paymentResponse.razorpay_payment_id,
                            razorpay_order_id: paymentResponse.razorpay_order_id,
                            razorpay_signature: paymentResponse.razorpay_signature,
                        });

                        if (verifyResponse.data.success) {
                            setPaymentSuccess(true);
                            setTimeout(() => {
                                router.reload();
                            }, 2000);
                        } else {
                            setError(verifyResponse.data.message || 'Payment verification failed');
                        }
                    } catch (err: any) {
                        setError(err.response?.data?.message || 'Payment verification failed');
                    }
                    setIsLoading(false);
                },
                prefill: {
                    name: auth?.user?.name || '',
                    email: auth?.user?.email || '',
                    contact: auth?.user?.phone || '',
                },
                theme: {
                    color: '#10b981', // Green theme
                },
                modal: {
                    ondismiss: function () {
                        setIsLoading(false);
                        setError('Payment was cancelled. Please try again.');
                    },
                },
                // Disable tracking to avoid ad blocker issues (optional)
                config: {
                    display: {
                        blocks: {
                            banks: {
                                name: "All payment methods",
                                instruments: [
                                    {
                                        method: "upi"
                                    },
                                    {
                                        method: "card"
                                    },
                                    {
                                        method: "netbanking"
                                    }
                                ]
                            }
                        },
                        sequence: ["block.banks"],
                        preferences: {
                            show_default_blocks: true
                        }
                    }
                }
            };

            try {
                const razorpay = new window.Razorpay(options);
                razorpay.on('payment.failed', function (response: any) {
                    console.error('Payment failed:', response);
                    setError(response.error?.description || 'Payment failed. Please try again.');
                    setIsLoading(false);
                });
                razorpay.open();
            } catch (razorpayError: any) {
                console.error('Error opening Razorpay:', razorpayError);
                setError('Failed to open payment gateway. Please try again or contact support.');
                setIsLoading(false);
            }
        } catch (err: any) {
            console.error('Payment error:', err);
            const errorData = err.response?.data || {};
            const errorMessage = errorData.message 
                || errorData.error 
                || err.message 
                || 'An error occurred while initiating payment';
            
            // If server indicates redirect to onboarding, show message and redirect
            if (errorData.redirect_to_onboarding) {
                setError(errorMessage + ' Redirecting...');
                setTimeout(() => {
                    router.visit('/dashboard');
                }, 2000);
                setIsLoading(false);
                return;
            }
            
            setError(errorMessage);
            setIsLoading(false);
        }
    };

    if (paymentSuccess) {
        return (
            <div className="min-h-screen bg-gradient-to-br from-gray-50 to-gray-100 dark:from-gray-900 dark:to-gray-800 flex items-center justify-center p-4">
                <div className="max-w-xl w-full">
                    <div className="bg-white dark:bg-gray-800 rounded-2xl shadow-xl overflow-hidden">
                        <div className="bg-gradient-to-r from-green-500 to-emerald-500 p-6 text-white">
                            <div className="flex items-center gap-4">
                                <div className="w-14 h-14 bg-white/20 rounded-full flex items-center justify-center">
                                    <CheckCircle2 className="w-8 h-8" />
                                </div>
                                <div>
                                    <h1 className="text-2xl font-bold">Payment Successful!</h1>
                                    <p className="text-white/80 text-sm">Your subscription is now active</p>
                                </div>
                            </div>
                        </div>
                        <div className="p-8 text-center">
                            <Loader2 className="w-8 h-8 animate-spin mx-auto text-green-500 mb-4" />
                            <p className="text-gray-600 dark:text-gray-400">Redirecting to your dashboard...</p>
                        </div>
                    </div>
                </div>
            </div>
        );
    }

    return (
        <div className="min-h-screen bg-gradient-to-br from-gray-50 to-gray-100 dark:from-gray-900 dark:to-gray-800 flex items-center justify-center p-4">
            <div className="max-w-xl w-full">
                {/* Main Card */}
                <div className="bg-white dark:bg-gray-800 rounded-2xl shadow-xl overflow-hidden">
                    {/* Header - Different colors for different statuses */}
                    <div className={`p-6 text-white ${
                        status === 'expired' || status === 'expired_grace'
                            ? 'bg-gradient-to-r from-red-500 to-rose-600'
                            : 'bg-gradient-to-r from-amber-500 to-orange-500'
                    }`}>
                        <div className="flex items-center gap-4">
                            <div className="w-14 h-14 bg-white/20 rounded-full flex items-center justify-center">
                                {status === 'expired' || status === 'expired_grace' ? (
                                    <Clock className="w-8 h-8" />
                                ) : (
                                    <AlertTriangle className="w-8 h-8" />
                                )}
                            </div>
                            <div>
                                <h1 className="text-2xl font-bold">
                                    {status === 'trial' && 'Trial Ended'}
                                    {status === 'pending_payment' && 'Payment Required'}
                                    {status === 'expired' && 'Subscription Expired'}
                                    {status === 'expired_grace' && 'Subscription Expired'}
                                </h1>
                                <p className="text-white/80 text-sm">
                                    {status === 'trial' && 'Your trial period has expired'}
                                    {status === 'pending_payment' && 'Complete payment to access your dashboard'}
                                    {status === 'expired' && 'Please renew to continue using the platform'}
                                    {status === 'expired_grace' && 'Renew now before you lose access'}
                                </p>
                            </div>
                        </div>
                    </div>

                    {/* Content */}
                    <div className="p-8 space-y-6">
                        <div className="bg-amber-50 dark:bg-amber-900/20 border border-amber-200 dark:border-amber-800 rounded-xl p-4">
                            <p className="text-amber-800 dark:text-amber-200">
                                {message}
                            </p>
                            {(status === 'expired' || status === 'expired_grace' || status === 'trial' || status === 'pending_payment') && (
                                <p className="text-amber-700 dark:text-amber-300 text-sm mt-2 font-medium">
                                    💡 Note: If you don't have payment access, please contact your super admin for subscription renewal.
                                </p>
                            )}
                        </div>

                        {/* Error Message */}
                        {error && (
                            <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-xl p-4">
                                <p className="text-red-700 dark:text-red-300 text-sm">{error}</p>
                            </div>
                        )}

                        <div className="space-y-4">
                            <div className="flex items-start gap-3">
                                <div className="w-8 h-8 rounded-full bg-blue-100 dark:bg-blue-900/30 flex items-center justify-center flex-shrink-0">
                                    <CreditCard className="w-4 h-4 text-blue-600 dark:text-blue-400" />
                                </div>
                                <div>
                                    <h3 className="font-semibold text-gray-900 dark:text-white">
                                        Activate Your Subscription
                                    </h3>
                                    <p className="text-sm text-gray-500 dark:text-gray-400">
                                        Click the button below to complete your payment and activate your subscription instantly.
                                    </p>
                                </div>
                            </div>
                        </div>

                        {/* Pay Now Button - Primary Action - Only show if payment gateway is available */}
                        {has_payment_gateway ? (
                            <Button 
                                onClick={handlePayNow}
                                disabled={isLoading || !razorpayLoaded}
                                className="w-full bg-gradient-to-r from-green-500 to-emerald-600 hover:from-green-600 hover:to-emerald-700 text-white py-6 text-lg rounded-xl shadow-lg shadow-green-200 dark:shadow-green-900/20 disabled:opacity-50 disabled:cursor-not-allowed"
                            >
                                {isLoading ? (
                                    <>
                                        <Loader2 className="w-5 h-5 mr-2 animate-spin" />
                                        Processing...
                                    </>
                                ) : !razorpayLoaded ? (
                                    <>
                                        <Loader2 className="w-5 h-5 mr-2 animate-spin" />
                                        Loading payment gateway...
                                    </>
                                ) : (
                                    <>
                                        <IndianRupee className="w-5 h-5 mr-2" />
                                        {status === 'expired' || status === 'expired_grace' 
                                            ? 'Renew Now' 
                                            : 'Pay Now & Activate'}
                                    </>
                                )}
                            </Button>
                        ) : (
                            <div className="w-full bg-gray-100 dark:bg-gray-800 border border-gray-300 dark:border-gray-700 rounded-xl p-4 text-center">
                                <p className="text-gray-600 dark:text-gray-400 text-sm">
                                    <span className="font-semibold">No payment gateway configured.</span>
                                    <br />
                                    Please contact your super admin to configure payment gateways or renew your subscription.
                                </p>
                            </div>
                        )}

                        {/* Secondary Actions */}
                        <div className="pt-2 flex flex-col sm:flex-row gap-3">
                            <Button 
                                onClick={handleRefresh}
                                variant="outline"
                                className="flex-1"
                                disabled={isLoading}
                            >
                                <RefreshCw className="w-4 h-4 mr-2" />
                                Refresh Status
                            </Button>
                            <Button 
                                onClick={handleLogout}
                                variant="ghost"
                                className="flex-1"
                                disabled={isLoading}
                            >
                                Logout
                                <ArrowRight className="w-4 h-4 ml-2" />
                            </Button>
                        </div>
                    </div>
                </div>

                {/* Footer */}
                <p className="text-center text-sm text-gray-500 dark:text-gray-400 mt-6">
                    Need help? Contact support at{' '}
                    <a href="mailto:support@societify.in" className="text-blue-600 hover:underline">
                        support@societify.in
                    </a>
                </p>
            </div>
        </div>
    );
}
