import '../css/app.css';

import { createInertiaApp } from '@inertiajs/react';
import { resolvePageComponent } from 'laravel-vite-plugin/inertia-helpers';
import { StrictMode } from 'react';
import { createRoot } from 'react-dom/client';
import { initializeTheme } from './hooks/use-appearance';

const appName = import.meta.env.VITE_APP_NAME || 'Laravel';

// Get all pages using glob pattern
const pages = import.meta.glob('./pages/**/*.tsx', { eager: false });

// Create a normalized lookup map for case-insensitive and path variations
const createPageLookup = () => {
    const lookup: Record<string, string> = {};
    const keys = Object.keys(pages);
    
    keys.forEach(key => {
        // Store original key
        lookup[key] = key;
        
        // Store lowercase version for case-insensitive lookup
        lookup[key.toLowerCase()] = key;
        
        // Store without leading ./ for path variations
        const withoutDot = key.replace(/^\.\//, '');
        lookup[withoutDot] = key;
        lookup[withoutDot.toLowerCase()] = key;
        
        // Extract filename for direct lookup
        const filename = key.split('/').pop() || '';
        if (filename) {
            lookup[filename.toLowerCase()] = key;
        }
    });
    
    return lookup;
};

const pageLookup = createPageLookup();

createInertiaApp({
    title: (title) => (title ? `${title} - ${appName}` : appName),
    resolve: (name) => {
        // Try multiple path formats
        const possiblePaths = [
            `./pages/${name}.tsx`,
            `pages/${name}.tsx`,
            `${name}.tsx`,
            name.toLowerCase() + '.tsx',
            `./pages/${name.toLowerCase()}.tsx`,
        ];
        
        // Find the first matching key in our lookup
        let matchedKey: string | undefined;
        for (const path of possiblePaths) {
            matchedKey = pageLookup[path.toLowerCase()];
            if (matchedKey) {
                break;
            }
        }
        
        if (matchedKey) {
            return resolvePageComponent(matchedKey, pages);
        }
        
        // Fallback: try resolvePageComponent with the standard path
        // This will throw an error if not found, which will be caught and shown
        const standardPath = `./pages/${name}.tsx`;
        
        // Log for debugging
        console.error('Page resolution failed:', {
            component: name,
            triedPaths: possiblePaths,
            availablePages: Object.keys(pages).slice(0, 10),
            lookupKeys: Object.keys(pageLookup).slice(0, 20),
        });
        
        return resolvePageComponent(standardPath, pages);
    },
    setup({ el, App, props }) {
        const root = createRoot(el);

        root.render(
            <StrictMode>
                <App {...props} />
            </StrictMode>,
        );
    },
    progress: {
        color: '#8b5cf6',
    },
});

// This will set light / dark mode on load...
initializeTheme();
