<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // For SQLite, we need to recreate the table to modify enum constraints
        // For MySQL/PostgreSQL, we can use MODIFY COLUMN
        
        if (DB::connection()->getDriverName() === 'sqlite') {
            // SQLite approach: Create new table, copy data, rename
            Schema::create('bill_cycles_new', function (Blueprint $table) {
                $table->id();
                $table->foreignId('society_id')->constrained('societies')->cascadeOnDelete();
                $table->string('name');
                $table->year('year');
                $table->date('generate_date');
                $table->date('due_date');
                $table->enum('status', ['draft', 'generated', 'posted', 'closed'])->default('draft');
                $table->timestamps();

                $table->index(['society_id', 'year']);
                $table->index('status');
            });

            // Copy existing data
            DB::statement('INSERT INTO bill_cycles_new SELECT * FROM bill_cycles');

            // Drop old table and rename new one
            Schema::drop('bill_cycles');
            Schema::rename('bill_cycles_new', 'bill_cycles');
        } else {
            // MySQL/PostgreSQL approach
            DB::statement("ALTER TABLE bill_cycles MODIFY COLUMN status ENUM('draft', 'generated', 'posted', 'closed') DEFAULT 'draft'");
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Revert the status change
        if (DB::connection()->getDriverName() === 'sqlite') {
            Schema::create('bill_cycles_old', function (Blueprint $table) {
                $table->id();
                $table->foreignId('society_id')->constrained('societies')->cascadeOnDelete();
                $table->string('name');
                $table->year('year');
                $table->date('generate_date');
                $table->date('due_date');
                $table->enum('status', ['draft', 'generated', 'closed'])->default('draft');
                $table->timestamps();

                $table->index(['society_id', 'year']);
                $table->index('status');
            });

            // Update 'posted' to 'generated' before copying
            DB::statement("UPDATE bill_cycles SET status = 'generated' WHERE status = 'posted'");
            
            DB::statement('INSERT INTO bill_cycles_old SELECT * FROM bill_cycles');
            Schema::drop('bill_cycles');
            Schema::rename('bill_cycles_old', 'bill_cycles');
        } else {
            DB::statement("UPDATE bill_cycles SET status = 'generated' WHERE status = 'posted'");
            DB::statement("ALTER TABLE bill_cycles MODIFY COLUMN status ENUM('draft', 'generated', 'closed') DEFAULT 'draft'");
        }
    }
};
