<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('facilities', function (Blueprint $table) {
            $table->id();
            $table->foreignId('society_id')->constrained()->cascadeOnDelete();
            $table->string('name');
            $table->text('description')->nullable();
            $table->json('images')->nullable();
            $table->integer('capacity')->nullable();
            $table->text('amenities')->nullable(); // JSON or comma separated
            $table->text('rules')->nullable();
            $table->text('cancellation_policy')->nullable();
            $table->boolean('is_active')->default(true);
            $table->boolean('requires_approval')->default(false);
            $table->softDeletes();
            $table->timestamps();
        });

        Schema::create('facility_slots', function (Blueprint $table) {
            $table->id();
            $table->foreignId('facility_id')->constrained()->cascadeOnDelete();
            $table->string('name'); // e.g., "Morning Slot", "Full Day"
            $table->time('start_time');
            $table->time('end_time');
            $table->decimal('price', 10, 2)->default(0);
            $table->integer('max_bookings_per_slot')->default(1); // Usually 1 for a hall, but maybe more for a generic resource
            $table->boolean('is_active')->default(true);
            $table->timestamps();
        });

        Schema::create('facility_bookings', function (Blueprint $table) {
            $table->id();
            $table->foreignId('society_id')->constrained()->cascadeOnDelete();
            $table->foreignId('facility_id')->constrained()->cascadeOnDelete();
            $table->foreignId('slot_id')->constrained('facility_slots')->cascadeOnDelete();
            $table->foreignId('user_id')->constrained()->cascadeOnDelete(); // The resident/user the booking is for
            $table->foreignId('booked_by')->nullable()->constrained('users')->nullOnDelete(); // Who performed the action (Admin/Collector/Self)
            $table->date('booking_date');
            $table->decimal('amount', 10, 2);
            $table->string('payment_status')->default('pending'); // pending, paid, failed, refunded
            $table->string('payment_method')->nullable(); // cash, online, cheque
            $table->string('booking_status')->default('pending'); // pending, confirmed, cancelled, rejected, completed
            $table->text('purpose')->nullable();
            $table->integer('guest_count')->nullable();
            $table->text('admin_notes')->nullable();
            $table->text('cancellation_reason')->nullable();
            $table->timestamp('checked_in_at')->nullable();
            $table->timestamp('checked_out_at')->nullable();
            $table->text('payment_details')->nullable(); // JSON for transaction IDs etc
            $table->timestamps();
            $table->softDeletes();

            // Prevent double booking: Unique slot per facility per date (if max_bookings is 1)
            // We can't strictly enforce unique key here if max_bookings > 1, but for halls it's usually 1.
            // Leaving index for performance.
            $table->index(['facility_id', 'booking_date', 'slot_id']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('facility_bookings');
        Schema::dropIfExists('facility_slots');
        Schema::dropIfExists('facilities');
    }
};
