<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        if (DB::getDriverName() === 'sqlite') {
            // 1. Create new table with updated schema
            Schema::create('units_temp', function (Blueprint $table) {
                $table->id();
                $table->foreignId('society_id')->constrained('societies')->cascadeOnDelete();
                $table->foreignId('building_id')->nullable()->constrained('buildings')->nullOnDelete();
                $table->string('unit_no');
                // Updated enum values
                $table->enum('type', ['apartment', 'shop', 'office', 'parking', 'flat', 'bungalow'])->default('apartment');
                $table->decimal('area', 8, 2)->nullable();
                // Updated enum values
                $table->enum('status', ['vacant', 'occupied', 'maintenance', 'rented'])->default('vacant');
                $table->foreignId('user_id')->nullable()->constrained('users')->nullOnDelete();
                $table->timestamps();
                $table->softDeletes();
            });

            // 2. Copy data from old table to new table
            // We need to be careful if existing data violates new constraints, but here we are ADDING options, so it should be fine.
            // However, we must ensure we copy all columns.
            DB::statement('INSERT INTO units_temp (id, society_id, building_id, unit_no, type, area, status, user_id, created_at, updated_at, deleted_at) SELECT id, society_id, building_id, unit_no, type, area, status, user_id, created_at, updated_at, deleted_at FROM units');

            // 3. Drop old table
            Schema::drop('units');

            // 4. Rename new table to old table name
            Schema::rename('units_temp', 'units');

            // 5. Recreate indexes
            Schema::table('units', function (Blueprint $table) {
                $table->unique(['society_id', 'unit_no']);
                $table->index(['society_id', 'status']);
                $table->index('building_id');
            });
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Reverting this is complex and risky as we might lose data if we revert to fewer enum options.
        // For this fix, we generally don't provide a down method that reverts the schema change 
        // because it's a fix for a broken state in SQLite.
    }
};
